function log() {
    // Logs all arguments if the console is defined
    if( console && console.log ) console.log.apply(console, arguments)
}

var baseRequest = { "jsonrpc": "2.0",
                    "method": "runCmds",
                    "params": {
                        "version": 1
                    },
                    "id": null
                  };
                  
var baseAutoCompleteRequest = { "jsonrpc": "2.0",
                                "method": "getCommandCompletions",
                                "params": {},
                                "id": "autoComplete"
                              };

var global_id = 0;                              
function EapiClient(){
   this.runCmds = function( params, id, cb ) {
      var request = $.extend( true, {}, baseRequest );
      request[ "params" ] = params;
      request[ "id" ] = id || ( "EapiExplorer-" + global_id );
      global_id++;
      requestViewer.getSession().setValue( JSON.stringify( request, null, 2 ) );
      logRequest(request);

      // We'll handle the POST success with the returned promise
      var dfd = $.Deferred();
      var response = $.post("/command-api", JSON.stringify( request ), null, "json");
      var backtrace = (new Error).stack;
      response.done(function( data ){ 
         if ( data.result ) {
            logResponse( request.id, request.params.cmds, data );
            if( jQuery.isFunction( cb ) ){
               try {
                  cb( data.result );
               } catch (err) {
                  handleUserError(err);
                  throw err;
               }
            }
            dfd.resolve( data.result );
         } else if (data.error) {            
            logCollapsible( "consoleError",
                            "JSON-RPC error " + data.error.code + ": " + data.error.message,
                            JSON.stringify(data.error, null, 2) + "\nBacktrace " + backtrace );
            dfd.reject( data.error.code, data.error.message, data.error.data );
         }
      });
      return dfd;
   };
}

var scriptEditor = null;
var requestViewer = null;
var responseViewer = null;
var simpleRequestEditor = null;

var TAB_SIZE = 2;

function createNewSession( text ) {
   session =  new ace.EditSession( text, "ace/mode/javascript" ); 
   session.setTabSize( TAB_SIZE );
   return session;
}

var scripts = [];
$.each( SAMPLE_SCRIPTS, function( index, data ){
   functionData = data.script.toString().match( /function [A-Za-z0-9$_ ]+\(\)[^{]*{([.\s\S]*)}/m )[ 1 ].trim();
   scripts.push( { "scriptName": data.scriptName,
                   "session": createNewSession( functionData ) } );
} );

$(document).ready( function() {
   var langTools = ace.require("ace/ext/language_tools");
   var HashHandler = ace.require( "ace/keyboard/hash_handler" ).HashHandler
   simpleRequestEditor = ace.edit("simpleRequestEditor");
   simpleRequestEditor.setShowPrintMargin(false);
   simpleRequestEditor.$blockScrolling = Infinity;
   simpleRequestEditor.setTheme("ace/theme/chrome");
   simpleRequestEditor.setOptions({
      enableLiveAutocompletion: true
   });
   var hh = new HashHandler([{
      bindKey: "shift-enter",
      descr: "",
      exec: runSimpleRequest,
   }]);
   simpleRequestEditor.keyBinding.addKeyboardHandler( hh );
   hh = new HashHandler([{
      bindKey: "tab",
      descr: "",
      exec: function() { $( "#version" ).focus() },
   }]);
   simpleRequestEditor.keyBinding.addKeyboardHandler( hh );

   var url = window.location.origin;
   if (!url) {
      // Thanks, I.E.
      url = window.location.protocol + "//" + window.location.hostname + 
         (window.location.port ? ':' + window.location.port: '');
   }
   $( "#apiUrl" ).val( url + "/command-api" );
   
   var showCommandCompleter = {
      getCompletions: function( editor, session, pos, prefix, callback ) {
         if( editor != simpleRequestEditor || pos.column === 0 ) { 
            callback(null, []); 
            return; 
         }
         var lineText = session.getLine( pos.row ).substring( 0, pos.column );
         var request = $.extend( true, {}, baseAutoCompleteRequest );
         request.params[ "command" ] = lineText;
         $.post( "/command-api", JSON.stringify( request ), function( data ){
            var completions = []
            if ( data.result.complete )
               completions.push( { name: "<cr>", value: "<cr>", score: 20, meta: 'Complete command' } )
            for ( var c in data.result.completions )
               completions.push( { name: c, value: c, score: 10, meta: data.result.completions[ c ] } )
            callback( null, completions );
         } )
      } 
   };
   langTools.addCompleter( showCommandCompleter );
   
   scriptEditor = ace.edit("scriptEditor");
   scriptEditor.setShowPrintMargin(false);
   scriptEditor.$blockScrolling = Infinity;
   scriptEditor.setTheme("ace/theme/chrome");
   scriptEditor.setSession( scripts[ 0 ].session );
   scriptEditor.setOptions({
      enableBasicAutocompletion: true,
      enableLiveAutocompletion: true,
      fontSize: "9.5pt",
   });
   var hh = new HashHandler([{
      bindKey: "shift-enter",
      descr: "",
      exec: runCode,
   }])
   scriptEditor.keyBinding.addKeyboardHandler( hh );


   
   requestViewer = ace.edit("requestViewer");
   requestViewer.$blockScrolling = Infinity;
   requestViewer.setTheme("ace/theme/chrome");
   requestViewer.setReadOnly( true );
   requestViewer.getSession().setMode( "ace/mode/json" );
   requestViewer.getSession().setValue(
      '"Enter commands above and click \'Submit POST request\'"' );
   
   responseViewer = ace.edit("responseViewer");
   responseViewer.$blockScrolling = Infinity;
   responseViewer.setTheme("ace/theme/chrome");
   responseViewer.setReadOnly( true );
   responseViewer.setShowPrintMargin( false );
   responseViewer.getSession().setMode( "ace/mode/json" );
   responseViewer.getSession().setOptions({ wrap: true });
   
   // lets create all of the scripts that we have
   $.each( scripts, function( index, obj ) {
       var listItem = $( "<li><a href='#'>" + obj.scriptName + "</a></li>" ).addClass( "scriptChoice" );
       if( index == 0 ) {
         listItem.addClass( "active" );
       }
       listItem.click( function(e) {
          scriptEditor.setSession( scripts[ index ].session );
          $( ".scriptChoice" ).removeClass( "active" );
          listItem.addClass( "active" );
          $( "#consoleOutput" ).empty();
          e.preventDefault(); // Don't scroll to top.
       } );
      $( "#fileSelector" ).append( listItem );
   } );
   
   // we send a request to the backend so we are asked our credentials
   var request = $.extend( true, {}, baseAutoCompleteRequest );
   request.params[ "command" ] = "show version";
   $.post( "/command-api", JSON.stringify( request ), function( data ){} );

   $( "#simpleRequestTabSelector" ).click( function() {
      $( "#simpleRequestTabSelector" ).addClass( "active" );
      $( "#scriptEditorTabSelector" ).removeClass( "active" );
      $( "#scriptEditorTab" ).hide();
      $( "#simpleRequestTab" ).show();
      simpleRequestEditor.focus();
   } );
   $( "#scriptEditorTabSelector" ).click( function() {
      $( "#scriptEditorTabSelector" ).addClass( "active" );
      $( "#simpleRequestTabSelector" ).removeClass( "active" );
      $( "#scriptEditorTab" ).show();
      $( "#simpleRequestTab" ).hide();
      scriptEditor.focus();
   } );

   $( "#simpleRequestTabSelector" ).click();
   
   $( "#submitCode" ).tooltip().click( runCode );
   $( "#submitSimpleRequest" ).tooltip().click( runSimpleRequest );

   $( '#showRequestCheckbox' ).change(function() { $(".consoleRequest").toggle() });
   $( '#showResponseCheckbox' ).change(function() { $(".consoleResponse").toggle() });
   $( "#showDocumentationLink" ).click(function(e) {
      $( "#showDocumentationLink" ).hide();
      $( "div.apiDocumentation" ).show();
      e.preventDefault();
   });
   
   createSimpleRequestHelp();
   createSimpleRequestExamples();
} ); 

function createSimpleRequestHelp() {
   $("#apiUrlLabel").popover({
      trigger: "hover", html: true, placement: "bottom",
      title: "The API URL",
      content: 'This URL is the endpoint that responds to eAPI <code>POST</code> requests.'});
   $("#cmdsLabel").popover({
      trigger: "hover", html: true, placement: "bottom",
      title: "The <code>cmds</code> parameter",
      content: 'Contains the list of CLI commands you wish to run. On each line, either enter the command you wish to run or the corresponding complex command JSON as documented in the "Overview" page.'});
   $("#versionLabel").popover({
      trigger: "hover", html: true, placement: "left",
      title: "The <code>version</code> parameter",
      content: 'The global version of the API you are querying. At this time either use <code>1</code> or <code>"latest"</code>.'});
   $("#formatLabel").popover({
      trigger: "hover", html: true, placement: "left",
      title: "The <code>format</code> parameter",
      content: 'Indicates whether you want each command to return machine-readable JSON or human-readable ASCII output.'});
   $("#timestampsLabel").popover({
      trigger: "hover", html: true, placement: "left",
      title: "The <code>timestamps</code> parameter",
      content: 'An optional parameter. If <code>true</code>, return the per-command execution time.'});
   $("#autoCompleteLabel").popover({
      trigger: "hover", html: true, placement: "left",
      title: "The <code>autoComplete</code> parameter",
      content: 'An optional parameter. If <code>true</code>, eAPI will accept partial commands.'});
   $("#expandAliasesLabel").popover({
      trigger: "hover", html: true, placement: "left",
      title: "The <code>expandAliases</code> parameter",
      content: 'An optional parameter. If <code>true</code>, eAPI will accept single command aliases.'});
   $("#requestIdLabel").popover({
      trigger: "hover", html: true, placement: "left",
      title: "The JSON-RPC <code>id</code> parameter",
      content: 'An optional parameter that is echoed back in the response.'});
   $("#includeErrorDetailLabel").popover({
      trigger: "hover", html: true, placement: "left",
      title: "The <code>includeErrorDetail</code> parameter",
      content: 'An optional parameter. If <code>true</code>, eAPI will include extra information in an <code>errorDetail</code> field in the error response.'});
}


function loadSimpleRequest(request) {
   // Given a request, load up the commands.
   if (request.version) {
      $( "#version" ).val( request.version );
   }
   if (request.cmds) {
      simpleRequestEditor.getSession().setValue( request.cmds );
   }
   if (request.format) {
      $( "#format" ).val( request.format );
   }
}

function createSimpleRequestExamples() {
   $( "#requestExampleVersion" ).click(function(e) {
      loadSimpleRequest({cmds: "show version"});
      e.preventDefault();
   });
   $( "#requestExampleAcl" ).click(function(e) {
      loadSimpleRequest({cmds: '{"cmd": "enable", "input": "my_enable_passw0rd"}\nconfigure\nip access-list eapiExample\n10 permit ip any any'});
      e.preventDefault();
   });
   $( "#requestExampleVirtRouter" ).click(function(e) {
      loadSimpleRequest({cmds: '{"cmd": "show ip virtual-router", "revision": 2}'})
      e.preventDefault();
   });
   $( "#requestExampleRunningConf" ).click(function(e) {
      loadSimpleRequest({cmds: '{"cmd": "enable", "input": "my_enable_passw0rd"}\nshow running-config'});
      e.preventDefault();
   });
}

function runSimpleRequest(e){
   responseViewer.getSession().setValue( '"Waiting for response..."' );
   
   version = $( "#version" ).val();
   if( version != "latest" ){
      version = parseInt( version );
   }
   
   var cmds = simpleRequestEditor.getSession().getValue().split( "\n" );
   var sanitizedCmds = [];
   var prevCmd = "";
   $.each( cmds, function( index, cmd ) {
      if( cmd == "" || cmd.indexOf( "!!" ) === 0 ) {
         return;
      }
      cmd = prevCmd ? prevCmd + "\n" + cmd : cmd;
      var cmdParam = null;
      log("ok....", cmd, 'first', '"' + cmd.charAt(0) + '"');
      if ( cmd.charAt(0) == "{" ) {
         if ( cmd.charAt(cmd.length - 1) == "}" ) {
            try {
               cmdParam = JSON.parse(cmd);
            } catch (e) {
               alert("Invalid JSON for command\n   " + cmd + "\n\n" + e);
               return;
            }
         } else {
            // We haven't seen the "}" yet
            prevCmd = cmd;
            return
         }
      } else {
         // Looks like this is just a regular string
         if ( cmd.charAt(0) == "\"" && cmd.charAt( cmd.length - 1 ) == "\"" ) {
            // Strip any quotation marks that the user may have added.
            cmd = cmd.substring( 1, cmd.length - 1 );
         }
         cmdParam = cmd;
      }
      
      if (cmdParam) {
         sanitizedCmds.push( cmdParam );
         prevCmd = "";
      }
   } );
   if (prevCmd) {
      alert("Complex command JSON was not complete: " + prevCmd);
   }
   
   var request = $.extend( true, {}, baseRequest );
   params = { "format":  $( "#requestFormat" ).val().replace(/"/g, ""),
              "timestamps": $( "#timestamps" ).val()=='true',
              "autoComplete": $( "#autoComplete" ).val()=='true',
              "expandAliases": $( "#expandAliases" ).val()=='true',
              "includeErrorDetail": $( "#includeErrorDetail" ).val()=='true',
              "cmds": sanitizedCmds,
              "version": version };
   request.params = params;
   request.id = $( "#requestId" ).val();
   requestViewer.getSession().setValue( JSON.stringify( request, null, 2 ) );
   $.post( "/command-api", JSON.stringify( request ), 
           function( data ){ 
              responseViewer.getSession().setValue( JSON.stringify( data, null, 2 ) );
           } );
   requestViewer.getSession().setValue( JSON.stringify( request, null, 2 ) );
   if (e && e.preventDefault) e.preventDefault();
}

function runCode(e){
   $( "#consoleOutput" ).empty();
   var debugPre = "try {"; // No newline so we don't mess up the line numbers
   var debugPost = "\n} catch (err) { handleUserError(err) }";
   eval( debugPre + scriptEditor.getSession().getValue() + debugPost );
   if (e && e.preventDefault) e.preventDefault();
}

function handleUserError(err) {
   logError(err);
   throw err;
}

function getLineNumberStr(err) {
   var linenum = err.lineNumber || err.line;
   if (linenum) {
      return " at line " + linenum;
   }
   return "";
}

function logCollapsible(logClass, header, payload) {
  $( "<div class='" + logClass + " collapsible'>" +
     "<div class='collapsible-header'>" +
     "<a href='#'><i class='icon-chevron-right'></i></a> " + header + "</div>" +
     "<div class='collapse'>" + payload + 
      "</div></div>" )
      .appendTo('#consoleOutput')
      .children('.collapsible-header').click(function(e) {
         $(this)
            .find('i').toggleClass('icon-chevron-right icon-chevron-down')
            .closest('.collapsible-header').siblings('.collapse').collapse('toggle');
         e.preventDefault();
      });
}

function logRequest(request) {
   var cmdStr = request.params.cmds.join( '; ');
   if (cmdStr.length > 40) {
      cmdStr = cmdStr.substring(0, 40) + "...";
   }
   logCollapsible('consoleRequest', "Request [id: " + request.id + "]: " + cmdStr,
                  JSON.stringify( request, null, 2 ));
}

function logResponse(id, commands, response) {
   var cmdStr = commands.join( '; ' );
   if (cmdStr.length > 40) {
      cmdStr = cmdStr.substring(0, 40) + "...";
   }
   logCollapsible('consoleResponse', "Response [id: " + id + "]: " + cmdStr,
                  JSON.stringify(response, null, 2 ));
}


function logMessage() {
   $( "#consoleOutput" ).append( "<div class='consoleMessage'>" +
                              Array.prototype.slice.call(arguments).join( ' ' ) + "</div>" );
}

function logError(err) {
   logCollapsible('consoleError', 'Error' + getLineNumberStr(err) + ':' + err,
                  "(You can use your browser's built-in Developer Tools to debug further)\n" +
                  err.stack);
}
