/* Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
Arista Networks, Inc. Confidential and Proprietary. */

'use strict';

var ALL_COMMANDS = [];
var ID_COMMAND_MAP = {};
var COMMAND_MODEL_MAP = {};
var MODELS = {};
var MODEL_HIERARCHY = {};
var currDisplayedCommands;
var populateCommandsWaiting = null;

function generateCommandId( command ) {
   return command.replace( /([^A-Za-z])/g, "_" );
}

function generateSubmodelOutput( level, modelId, checkSubclasses, printedModels ) {
   var curModel = MODELS[ modelId ];
   var containerDiv = $( '<div></div>' );
   if( $.inArray( modelId, printedModels ) !== -1 ){
      if( level === 0 ){
         containerDiv.append( 'see <a href="#submodel-' + modelId + '" class="mono">' + curModel.name + '</a>' );
      }
      return containerDiv;
   }
   
   printedModels.push( modelId );
   var modelDiv = $( '<div></div>' ).addClass( 'model' );
   modelDiv.append( '<span id="submodel-' + modelId + '" class="linkTarget"></span>' );
   var modelHeader = $( '<h3>' + curModel.name + '</h3>' );
   if( checkSubclasses && ( modelId in MODEL_HIERARCHY ) ) {
      var altModels = [];
      $.each( MODEL_HIERARCHY[ modelId ], function( index, altModel ) {
         altModels.push( '<a href="#submodel-' + altModel + '" class="mono">' + MODELS[ altModel ].name + '</a>' );
      } );
      modelHeader.append( '<br><small>Also could be an instance of ' + altModels.join(', ')  + '</small><br>' );
   }
   modelDiv.append( modelHeader );
   
   var attrTable = $( '<table></table>' ).addClass( 'table table-hover' );
   $.each( curModel.attributes, function( index, attribute ) {
      var sinceReleaseFlagStr = "";
      var sinceReleaseStr = "";
      if( attribute.sinceRelease ){
        sinceReleaseFlagStr = '  <font color="darkred">**</font>'
        sinceReleaseStr = '<br><font color="darkred">** Since ' + attribute.sinceRelease + '</font>';
      }
      var attrRow = $( '<tr></tr>' ).addClass( 'attr' );
      attrRow.append( $( '<td>' + attribute.name + sinceReleaseFlagStr + '</td>' ).addClass( 'attrName' ) );
      var attrType = $( '<td></td>' ).addClass( 'attrType' );
      attrType.append(  attribute.formatTypeStr );
      if( attribute.optional ) {
         attrType.append( ' - <em>optional</em>' );
      }
      
      attrRow.append( attrType );
      var helpStr =  attribute.helpStr.replace(/\n/g, "<br>");
      attrRow.append( $( '<td>' + helpStr + sinceReleaseStr + '</td>' ).addClass( 'helpDesc' ) );
      attrTable.append( attrRow );
   } );
   modelDiv.append( attrTable );
   
   $.each( curModel.seenModels, function( index, subModelId ) {
      modelDiv.append( generateSubmodelOutput( level + 1, subModelId, true, printedModels ) );
   } );
   
   containerDiv.append( modelDiv );
   
   if( modelId in MODEL_HIERARCHY ){
      $.each( MODEL_HIERARCHY[ modelId ], function( index, childId ) {
         containerDiv.append( generateSubmodelOutput( level, childId, false, printedModels ) );
      } );  
   }

   return containerDiv;
}

function populateModels( command ){
   $( "#modelsContainer" ).empty();
   var printedModels = [];
   var modelId = COMMAND_MODEL_MAP[ command ];
   var model = MODELS[ modelId ];
   $( "#modelsContainer" ).append( $( '<h2></h2>' ).attr( 'id', generateCommandId( command ) ).addClass( 'mono' ).text( 'show ' + command ) );
   $( "#modelsContainer" ).append( model.versionRev );
   $( "#modelsContainer" ).append( generateSubmodelOutput( 0, modelId, true, printedModels ) );

   // display the map of version to revision (skipping uninteresting version 1)
   var versionTable = $( '<table></table>' ).addClass( 'table table-bordered table-minWidth' );
   versionTable.append( '<tr><td>Version</td><td>Revision</td></tr>' );
   var lastRev = 1;
   $.each( model.revMap, function( index, row ) {
      var attrRow = $( '<tr></tr>' ).addClass( 'attr' );
      if ( row[0] != 1 ) {
         attrRow.append( $( '<td></td>' ).addClass( 'helpDesc' ).text( row[0] ) );
         attrRow.append( $( '<td></td>' ).addClass( 'helpDesc' ).text( row[1] ) );
      }
      lastRev = row[1];
      versionTable.append( attrRow );
   } );
   // the last revision might not be tied to a version yet (we call its revision 'latest')
   if ( model.revision != lastRev ) {
         lastRev = model.revision;
         var attrRow = $( '<tr></tr>' ).addClass( 'attr' );
         attrRow.append( $( '<td></td>' ).addClass( 'bullet needRev' ).addClass( "helpDesc" ).text( 'latest' ) );
         attrRow.append( $( '<td></td>' ).addClass( 'helpDesc' ).text( lastRev ) );
         versionTable.append( attrRow );
   }
   if ( lastRev > 1 ) {
      $( "#modelsContainer" ).append( versionTable );
   }

}

function populateCommandsDebounced( commands ){
   // Lets have a simple debouncer where we wait up to 100ms to make the changes
   if( !populateCommandsWaiting ){
      setTimeout( function() {
         populateCommands( populateCommandsWaiting ); 
         populateCommandsWaiting = null;
      }, 500 );
   }
   populateCommandsWaiting = commands;
}

function populateCommands( commands ){
   // if we are currently displaying the same commands, don't redraw the screen
   if( commands == currDisplayedCommands ) {
      return;
   }
   currDisplayedCommands = commands;
   $( "#commandContainer" ).empty();
   $( "#commandContainer" ).append( '<li class="nav-header">show commands</li>' );
   var commandsContainer = $('<div></div>')
   $.each( commands, function( index, command ) {
      var modelId = COMMAND_MODEL_MAP[ command ];
      var model = MODELS[ modelId ];
      var modelLi = $( '<li></li>' ).addClass( model.displayClass ).append( $( '<a href="#' + generateCommandId( command ) + '"></a>' ).text( command ) );
      modelLi.click( function() { populateModels( command ); } );
      modelLi.tooltip( { "title": command, "placement": "bottom", "delay": { "show": 200, "hide": 0 } } );
      commandsContainer.append( modelLi );
   } )
   $( "#commandContainer" ).append( commandsContainer );
}

function findWordInModel( modelId, searchWord, searchedModels ) {
   // handling for the recursive call in the 2nd `each` loop, which is necessary to
   // avoid endless loops when there is a cycle in the model's seenModels
   if( searchedModels == undefined ) {
       searchedModels = new Object();
       searchedModels[ modelId ] = 1;
   }
   else if( searchedModels[ modelId ] ) {
       return false;
   }
   else {
       searchedModels[ modelId ] = 1;
   }

   var curModel = MODELS[ modelId ];
   var wordFound = false;
   if( curModel.name == 'Dummy' ) { // when documentation.json does not exist, a
                                    // dummy model is generated
       return false;
   }

   $.each( MODELS[ modelId ].attributes, function( index, attribute ) {
      if( attribute.name.toLowerCase().indexOf( searchWord ) !== -1 ||
          attribute.helpStr.toLowerCase().indexOf( searchWord ) !== -1) {
          wordFound = true;
          return false;
      } 
   } );
   
   $.each( curModel.seenModels, function( index, subModelId ) {
      if( findWordInModel( subModelId, searchWord, searchedModels ) ){
          wordFound = true;
          return false;
      }
   } );

   if( curModel.versionRev.toLowerCase().indexOf( searchWord ) !== -1) {
       wordFound = true;
   } 
   return wordFound;
}

function findSubword( command, searchWord ) {
   if( command.indexOf( searchWord ) !== -1 ) {
      return true;
   }
   return findWordInModel( COMMAND_MODEL_MAP[ command ], searchWord );
}

function findCommandsContaining( searchCommand ) {
   searchCommand = searchCommand.toLowerCase();
   
   // if we are called with nothing or if just 1 word with show return all commands
   if( searchCommand.length === 0 || 
       ( searchCommand.split( " " ).length === 1 && 
         "show".indexOf( searchCommand ) === 0 ) ){
      return ALL_COMMANDS;
   }
   
   var filteredCommands = [];
   $.each( ALL_COMMANDS, function( index, command ) {
      var allWordsFound = true;
      $.each( searchCommand.split( " " ), function( searchIndex, searchWord ) {
         if( searchIndex === 0 && "show".indexOf( searchWord ) === 0 ) {
            return;
         }
         
         if( !findSubword( command, searchWord ) ){
            allWordsFound = false;
            return false;
         }
      } );
      if( allWordsFound ) {
         filteredCommands.push( command );
      }
   } );
   
   return filteredCommands;
}

function registerCommandFieldHandlers(){
  $( "#searchBox" ).keyup( function( eventData ) { 
       var filteredCommands = findCommandsContaining( $( "#searchBox" ).val() );
       populateCommandsDebounced( filteredCommands );
  } );
}

$(document).ready( function() {
   $.getJSON( "documentation.json", function( data ) {
      COMMAND_MODEL_MAP = data.COMMAND_MODEL_MAP;
      MODELS = data.MODELS;
      MODEL_HIERARCHY = data.MODEL_HIERARCHY;
      
      
      $.each( COMMAND_MODEL_MAP, function( command, modelId ) {
         ALL_COMMANDS.push( command );
      } );
      ALL_COMMANDS.sort();
      
      $.each( ALL_COMMANDS, function( index, command ) {
         var id = generateCommandId( command );
         ID_COMMAND_MAP[ id ] = command;
      } );
          
      if( window.location.hash ) {
         var id = window.location.hash.substring( 1 );
         if( id in ID_COMMAND_MAP ){
            populateModels( ID_COMMAND_MAP[ id ] );
         }
      }
      
      populateCommands( ALL_COMMANDS );
      registerCommandFieldHandlers();
    });
} );
