# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import gc, os
from contextlib import contextmanager

def getlock():
   ''' return a lock, or a null context manager if there's no support for threads '''
   try:
      import threading
      return threading.Lock()
   except ImportError:
      @contextmanager
      def null_context_manager():
         yield None
      return null_context_manager()

class gc_disabled( object ):
   ''' Provides a context manager to ensure GC is disabled while it is in scope.
   GC is disabled when the first thread enters the manager, and is re-enabled when
   the last thread exits it. '''

   hold_count = 0 # Number of threads currently blocking GC.
   gc_lock = getlock() # lock to protect hold_count, and gc.enable/gc.disable

   def __enter__( self ):
      with gc_disabled.gc_lock:
         if gc_disabled.hold_count == 0:
            gc.disable()
         gc_disabled.hold_count += 1
      return self

   def __exit__( self, typ, value, traceback ):
      with gc_disabled.gc_lock:
         gc_disabled.hold_count -= 1
         if gc_disabled.hold_count == 0:
            gc.enable()

   @classmethod
   def post_fork( cls ):
      '''
      Fixup things for a newly forked child.

      In the forked child of a threaded process, if another thread
      in the parent was interacting with the gc_lock, it's now in an
      indeterminate state in the child. This goes for other locks and
      state within the child - it's not a problem caused by gc_disabled
      per se - it is fundamentally unsafe to do complicated things in a
      forked child if the parent process was running threaded code. The
      only real use for fork in a threaded process should really to be
      in order to have it call exec()

      Because some code still tries to do things in such a forked child,
      we reset the gc_lock after a fork, and leave our hold_count
      unchanged. This gives us two scenarios:

      For multi-threaded processes:
         As long as the thread does not return from the gc_disabled
         context it is running, we will keep gc disabled, and any nested
         use of gc_disabled will not fall over the now reset lock

         If we exit the enclosing context manager, all bets are off,
         and you're likely to run into difficulties here or elsewhere.

      For single-threaded processes:
         The reset of the gc_lock is harmless, because we are the only
         thread, and we do not currently hold the lock. The process is
         free to return from the gc_disabled context manager.
      '''
      assert cls.hold_count > 0
      assert not gc.isenabled(), "gc should be disabled"
      cls.gc_lock = getlock()

   def fork( self ):
      rc = os.fork()
      if rc == 0:
         self.post_fork()
      return rc
