# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

_TacModule = None

class TacLazyType( object ):
   '''
   TacLazyType is a proxy object that can be used at the module level.

   It is generally a good idea to avoid loading Tac Types into python at the
   module level during module init, especially in python plugins, etc.

   To avoid this slowdown, a TypeFuture.TacLazyType can be used instead.

   from TypeFuture import TacLazyType
   ArnetPrefix = TacLazyType( 'Arnet::Prefix' )
   print ArnetPrefix.attributes
   assert ArnetPrefix() == Tac.Value( 'Arnet::Prefix' )
   assert isinstance( Tac.Value( 'Arnet::Prefix' ), ArnetPrefix )
   '''

   def __init__( self, typeName ):
      self._typeName = typeName
      # Do not access __type, use the self._type property only.
      self.__type = None

   def __maybeResolve( self ):
      '''
      __maybeResolve() is used only in the implementation of the _type property
      and will set the state of the internal __type attribute used as a cache
      for the _type property.
      '''
      if self.__type is None:
         global _TacModule
         if _TacModule is None:
            import Tac as _imported
            _TacModule = _imported
         self.__type = _TacModule.Type( self._typeName )

   @property
   def _type( self ):
      self.__maybeResolve()
      return self.__type

   def __getattr__( self, name ):
      return getattr( self._type, name )

   def __call__( self, *args, **kwargs ):
      return self._type( *args, **kwargs )

   def __instancecheck__( self, instance ):
      return isinstance( instance, self._type )

   def __eq__( self, other ):
      return self._type == other

   def __hash__( self ):
      return hash( self._type )
