# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
from functools import total_ordering
import os

# Kernel Features
ROUTE_CACHE      =  1 # BUG103898
TAP_QDISK        =  2 # BUG102872
VLAN_1AD         =  3 # BUG101982
PERNS_IFINDEX    =  4 # BUG161232,BUG173228
IPV4_ADDR_LFT    =  5 # BUG145426
GRE_XNETNS       =  6 # BUG177285,BUG191296,BUG192293
CORE_NTFILE      =  7
TAP_SKB_ARRAY    =  8 # BUG255918
SET_IGMP_MAX_MEM =  9 # BUG235752, BUG284337
IPV6_ADDR_NOTIFY = 10 # BUG340648
IPV6_ENH_DAD     = 11 # BUG340651
I40E_DIS_SRC_PR  = 12 # BUG276522

_features_ref = {
   ROUTE_CACHE:      lambda: _removedInVersion( '3.6' ), # 89aef8921bfb
   TAP_QDISK:        lambda: _removedInVersion( '3.8' ), # 5d097109257c
   VLAN_1AD:         lambda: _addedInVersion( '3.14' ), # a0cdfcf3936
   PERNS_IFINDEX:    lambda: _addedInVersion( '3.7' ), # aa79e66eee5d
   IPV4_ADDR_LFT:    lambda: _addedInVersion( '3.9' ), # 5c766d642bcaf
   GRE_XNETNS:       lambda: _addedInVersion( '3.16' ), # b57708add314
   CORE_NTFILE:      lambda: _addedInVersion( '3.7' ), # 2aa362c49c31
   TAP_SKB_ARRAY:    lambda: _addedInVersion( '4.8' ), # 1576d9860599
   SET_IGMP_MAX_MEM: lambda: _addedInVersion( '4.6' ), # 815c52700746
   IPV6_ADDR_NOTIFY: lambda: _addedInVersion( '4.18' ), # a2d481b326c9
   IPV6_ENH_DAD:     lambda: _addedInVersion( '4.10' ), # adc176c54722
   I40E_DIS_SRC_PR:  lambda: _addedInVersion( '4.15' ), # 64615b5418ba
}

@total_ordering
class KernelVersion( object ):
   def __init__( self, version ):
      version = version.split( '-' )[ 0 ] # ignore distro revision numbers
      self.version = []
      for i in version.split( '.' ):
         if not i.isdigit():
            break
         self.version.append( int( i ) )
      assert len( self.version ) > 1

   def __eq__( self, other ):
      return self.version == other.version

   def __lt__( self, other ):
      return self.version < other.version

   __hash__ = None

   def __str__( self ):
      return '.'.join( [ str( i ) for i in self.version ] )

current_version = KernelVersion( os.uname()[ 2 ] )

def _removedInVersion( v ):
   return current_version < KernelVersion( v )

def _addedInVersion( v ):
   return current_version >= KernelVersion( v )

def supports( f ):
   return _features_ref[ f ]()
