#!/usr/bin/env python
# Copyright (c) 2009 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import signal

class ExecTimeout( object ):
   """ Context manager that uses SIGALRM to escape a blocking function
   call.
   For example:
   # A Timeout will be raised if recv doesn't return within 5 seconds.
   with ExecTimeout( 5 ):
      data = mySocket.recv( 200 )
   """

   class Timeout( Exception ):
      pass
   
   def __init__( self, timeout ):
      self.timeout_ = timeout
      self.oldSignalHandler_ = None
      
   def __enter__( self ):
      self.oldSignalHandler_ = signal.getsignal( signal.SIGALRM )
      def alrm( dig, stack ):
         raise self.Timeout( "Timed out after %d seconds" % self.timeout_ )
      signal.signal( signal.SIGALRM, alrm )
      signal.alarm( self.timeout_ )
      
   def __exit__( self, exceptionType, value, traceback ):
      signal.signal( signal.SIGALRM, self.oldSignalHandler_ )
      signal.alarm( 0 )
