# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import six
import sys
import Tac
import Tracing
from PyClient import PyClient

t0 = Tracing.trace0
t1 = Tracing.trace1

class EntityTreeReactorUnexpectedChange( Exception ):
   pass

# Placeholder for the singleton local monitor. Useful to keep the monitor persistent
# if created over PyClient.
localMonitor = None

def createLocalMonitor():
   '''Multiple calls for creation are idempotent. There is always only a single
   instance of the monitor.'''
   global localMonitor 
   if localMonitor is None:
      localMonitor = LocalMonitor()
   return localMonitor

# These paths churn a lot.
frequentChurnPaths = [
               '/ar/Sysdb/cell/1/hardware/pciDeviceStatusDir',
               '/ar/Sysdb/cell/2/hardware/pciDeviceStatusDir',
               '/ar/Sysdb/agent',
               '/ar/Sysdb/environment',
               '/ar/Sysdb/eventMon/status',
               '/ar/Sysdb/hardware/sand/fap/status',
               '/ar/Sysdb/hardware/fan/status',
               '/ar/Sysdb/interface/counter',
               '/ar/Sysdb/snmp/counters',
               '/ar/Sysdb/acl/status',
               '/ar/Sysdb/routing6',
   ]
# There are bugs in EntityTreeMonitor that causes assertion when these
# paths are monitored. Most probably, we are somehow ending up in the
# same location twice, and then assert.
badPaths = [ 
               '/ar/Sysdb/cell',
               '/ar/Sysdb/hardware', # hw/cell, hw/sol/config/cell, etc are bad
#               '/ar/Sysdb/hardware/cell',
#               '/ar/Sysdb/hardware/sol/config/cell',
           ]

def entityPathAllowedByUser( fullName ):
   return fullName not in localMonitor.blacklistPaths

def entityPathAllowedBySystem( fullName ):
   return not (
         fullName == '/' or
         fullName.startswith( '/introspection' ) or
         fullName.startswith( '/MountFacility' ) or
         fullName.startswith( '/singleton' ) or
         fullName.startswith( '/mount' ) or
         fullName.startswith( '/activities' ) )

class LocalMonitor( object ):
   def __init__( self ):
      # A dict containing mapping of full names (i.e. paths) to EntityTreeReactors 
      # specified with addPath or addEntity.
      self.reactors = {}
      # Like the above dict, except that it contains mappings for every 
      # EntityTreeReactor, even those not specified by addPath or addEntity, but 
      # created recursively as a consequence of addPath or addEntity.
      self.allReactors = {}

      # used by all EntityReactors to print a message when an
      # attribute changes.
      # On duts, the message will show up in the agent log file.
      self.loggingEnabled = False
      self.blacklistPaths = set()

      self.lastAttrChangeAt = Tac.endOfTime

   def _reactor( self, path ):
      '''Take a path and return the corresponding EntityTreeReactor if one exists.'''
      assert path in self.allReactors, \
            'Entity %s is not currently being monitored' % path
      return self.allReactors[ path ]

   # === Setting changeAllowed/changeAllowedAttrs ===
   def changeAllowedIs( self, path, changeAllowed ):
      '''Set the changeAllowed field for EntityTreeReactor corresponding to the 
      given path and then recursively set the changeAllowed field for all of the 
      descendant EntityTreeReactors under this path.'''
      self._reactor( path ).recursiveChangeAllowedIs( changeAllowed )

   def allPathsChangeAllowedIs( self, changeAllowed ):
      '''Set the changeAllowed field for all EntityTreeReactors under the control of
      this LocalMonitor.'''
      for reactor in six.itervalues( self.allReactors ):
         reactor.changeAllowedIs( changeAllowed )

   def addChangeAllowedAttr( self, path, attr ):
      '''Add attr to the list of changeAllowedAttrs for the EntityTreeReactor 
      corresponding to path. See EntityTreeReactor below for a description of 
      changeAllowedAttrs.'''
      entityTreeReactor = self._reactor( path )
      entityTreeReactor.addChangeAllowedAttr( attr )

   def removeChangeAllowedAttr( self, path, attr ):
      '''Remove attr from the list of changeAllowedAttrs for the EntityTreeReactor 
      corresponding to path. See EntityTreeReactor below for a description of 
      changeAllowedAttrs.'''
      entityTreeReactor = self._reactor( path )
      entityTreeReactor.removeChangeAllowedAttr( attr )

   # === Add ===
   def addPath( self, path, changeAllowed, changeAllowedAttrs=None ):
      '''changeAllowedAttrs applies to the entity at the given path i.e. the top 
      level entity only.  See EntityTreeReactor below for a description of 
      changeAllowedAttrs.'''
      if not entityPathAllowedBySystem( path ):
         print( "Error: Path", path, "is not monitorable." )
         return
      if not entityPathAllowedByUser( path ):
         print( "Error: Path", path, "is blacklisted." )
         return
      if path in self.reactors:
         print( "Error: Path", path, "is already monitored." )
         return
      for monitoredPath in self.reactors:
         if path.startswith( monitoredPath ):
            print( "Error: Path", path, "is already monitored under as",
                   monitoredPath )
            return
         if monitoredPath.startswith( path ):
            print( "Warning: subpath", monitoredPath, "is already monitored." )
            # still allow.
      try:
         ent = Tac.entity( path )
      except NameError:
         print( "Error: Path", path, "not found." )
         return
      self.reactors[ path ] = \
            EntityTreeReactor( ent, changeAllowed=changeAllowed, 
                               changeAllowedAttrs=changeAllowedAttrs )

   def blacklist( self, path ):
      self.blacklistPaths.add( path )

   def addPaths( self, paths, changeAllowed ):
      assert type( paths ) is list
      for path in paths:
         assert path
         self.addPath( path, changeAllowed )

   def addEntity( self, entity, changeAllowed, changeAllowedAttrs=None ):
      '''changeAllowedAttrs applies to the given entity i.e. the top level entity 
      only.  See EntityTreeReactor below for a description of changeAllowedAttrs.'''
      assert entity
      self.addPath( entity.fullName, changeAllowed, changeAllowedAttrs )

   def addEntities( self, entities, changeAllowed ):
      assert type( entities ) is list
      for entity in entities:
         self.addEntity( entity, changeAllowed )

   # === Delete ===
   def delPath( self, path ):
      if path not in self.reactors:
         return # ignore if non-existent
      self.reactors[ path ].close()
      del self.reactors[ path ]

   def delPaths( self, paths ):
      assert type( paths ) is list
      for path in paths:
         assert path
         self.delPath( path )

   def delAllPaths( self ):
      for reactor in six.itervalues( self.reactors ):
         reactor.close()
      self.reactors.clear()

   def delEntity( self, entity ):
      assert entity
      self.delPath( entity.fullName )

   def delEntities( self, entities ):
      assert type( entities ) is list
      for entity in entities:
         self.delEntity( entity )

   # === Information gathering ===
   def showSummary( self ):
      if self.lastAttrChangeAt == Tac.endOfTime:
         print( "No changes seen" )
      else:
         print( "Changes seen. last seen @", self.lastAttrChangeAt, "\n" )
      for reactor in six.itervalues( self.reactors ):
         reactor.showSummary( showDescendents=True )

   def loggingEnabledIs( self, enabled ): # pylint: disable-msg=R0201
      self.loggingEnabled = enabled

def createRemoteMonitor( sysname='ar', agentName='Sysdb' ):
   '''Multiple calls to createRemoteMonitor with same argument may end up creating
   multiple pyClient session to the same agent. However all of them will still refer
   to the same underlying monitor in the agent's address space.'''
   return RemoteMonitor( sysname, agentName )

class RemoteMonitor( object ):
   '''This class uses PyClient to instantiate a LocalMonitor in the given agent's
   address space. On agent reboot, PyClient will become defunct since reconnect
   is set to False. This should prevent unexpected results in the access.'''

   def __init__( self, sysname, agentName ):
      self.pyClient = PyClient( sysname, agentName, reconnect=False )
      self.pyClient.execute( "import EntityTreeMonitor" )
      self.pyClient.execute( "EntityTreeMonitor.createLocalMonitor()" )

   def addPaths( self, paths, changeAllowed=False ):
      assert type( paths ) is list
      assert changeAllowed # For remote agents, don't crash, just monitor.
      # Its easier to call addEntity rather than addEntities via PyClient.
      for path in paths:
         assert path
         print( self.pyClient.execute(
                "EntityTreeMonitor.localMonitor.addPath( '%s', %s )" %
                ( path, str( changeAllowed ) ) ) )
   def blacklist( self, paths ):
      assert type( paths ) is list
      for path in paths:
         print( self.pyClient.execute(
                "EntityTreeMonitor.localMonitor.blacklist( '%s' )" %
                path ) )

   def delPaths( self, paths ):
      assert type( paths ) is list
      for path in paths:
         assert path
         self.pyClient.execute(
               "EntityTreeMonitor.localMonitor.delPath( '%s' )" % path )

   def showSummary( self ):
      print( self.pyClient.execute(
         "EntityTreeMonitor.localMonitor.showSummary()" ) )

   def loggingEnabledIs( self, enabled ):
      self.pyClient.execute( "EntityTreeMonitor.localMonitor.loggingEnabledIs( "
            + str( enabled ) + " )"  )
      
# This is a workaround since Tac.py tries to call python's hash instead of the hash
# attribute defined in the class itself.
def dictKey( obj ):
   if hasattr( obj, "hash" ):
      return getattr( obj, "hash" )
   else:
      return hash( obj )

class EntityTreeReactor( Tac.Notifiee ):
   '''
   Reacts to any attribute of the given entity or any attribute of any
   sub entity changing.
   changeAllowed: if False, asserts if anything underneath the entity changes.
                        if True, it will keep creating sub reactors as more entities
                        get added. Currently the only useful case is to view logs of
                        entity changes. However we may add more introspection apis
                        to collect statistics/summary of what has changed.
                        Note that the flag is inherited down to all trees. Later on
                        the flag can be overridden for some particular entity's
                        reactor, but that doesn't cause sub reactors to override
                        their behaviour.
   changeAllowedAttrs: if changeAllowed is False, this set of attributes of the top 
                       level entity acts as a whitelist of attributes, for which if 
                       a change happens, we do not raise an exception.
   Currently, we don't go deep on non-instantiating attributes types. This probably
   is done to avoid loops, or duplicate work. However it also means that we are
   losing coverage !
   '''

   notifierTypeName = "*"

   def __init__( self, entity, changeAllowed=False, changeAllowedAttrs=None ):
      Tac.Notifiee.__init__( self, entity, filtered=False )
      t0( 'New reactor for entity', entity.fullName )
      import collections
      self.childReactors = collections.defaultdict( dict )
      self.attrChangeCounter_ = {}
      self.lastAttrChangeAt = Tac.endOfTime
      self.changeAllowed_ = changeAllowed
      if changeAllowedAttrs:
         assert type( changeAllowedAttrs ) == set
         self.changeAllowedAttrs_ = changeAllowedAttrs
      else:
         self.changeAllowedAttrs_ = set()
      for attrName in entity.attributes:
         attr = entity.tacType.attr( attrName )
         # attr is of type Tac::TacAttr, and not really the actual attr
         # which can be gotten by using getattr()
         t1( 'attr:', attrName, ':', attr, ',in entity', entity.fullName )
         if not self.attrAllowed( attr, entity, attrName ):
            t1( 'ignore' )
            continue
         if attr.isCollection:
            t1( '    attr is a collection' )
            for key, value in getattr( entity, attrName ).iteritems():
               t1( '    key', key )
               if value:
                  self.addEntityReactor( attrName, key, value )
         else:
            t1( '   attr is a singleton' )
            child = getattr( entity, attrName )
            if child:
               self.addEntityReactor( attrName, None, child )
      # Add myself to the global list of EntityTreeReactors.
      self.name_ = entity.fullName
      assert self.name_ not in localMonitor.allReactors, \
            'Entity %s is already being monitored' % self.name_
      localMonitor.allReactors[ self.name_ ] = self

   def addChangeAllowedAttr( self, attrName ):
      self.changeAllowedAttrs_.add( attrName )

   def removeChangeAllowedAttr( self, attrName ):
      assert attrName in self.changeAllowedAttrs_
      self.changeAllowedAttrs_.remove( attrName )

   def addEntityReactor( self, attrName, key, entity ):
      t1( 'entity', entity, 'in parents  attr:', attrName, 'key:', key )
      assert entityPathAllowedBySystem( entity.fullName )
      if not entityPathAllowedByUser( entity.fullName ):
         return
      if key is not None:
         t0( 'Create EntityTreeReactor for collection attr %s[' %
               attrName, key, ']' )
      else:
         t0( 'Create EntityTreeReactor for singleton', entity.fullName )
      childReactor = EntityTreeReactor( entity, self.changeAllowed_ )
      childReactor.inCollection = True
      self.childReactors[ attrName ][ dictKey( key ) ] = childReactor

   def removeEntityReactor( self, attrName, key ):
      t1( 'entity in parents  attr:', attrName, 'key:', key )
      if key is not None:
         t0( 'Delete EntityTreeReactor for collection attr %s[' %
               attrName, key, ']' )
      else:
         t0( 'Delete EntityTreeReactor for singleton attr', attrName )
      if attrName in self.childReactors and dictKey( key ) in \
            self.childReactors[ attrName ]:
         # This check is needed since some entities may have been ignored in
         # addEntityReactor.
         self.childReactors[ attrName ][ dictKey( key ) ].close()
         del self.childReactors[ attrName ][ dictKey( key ) ]

   def changeAllowedIs( self, allowed ):
      '''Set changeAllowed field for this EntityTreeReactor.'''
      t0( 'changeAllowedIs() for entity %s: ' % self.name_, allowed )
      self.changeAllowed_ = allowed

   def recursiveChangeAllowedIs( self, allowed ):
      '''Set changeAllowed field for this EntityTreeReactor and all of its descendant
      EntityTreeReactors.'''
      t0( 'recursiveChangeAllowedIs() for entity %s: ' % self.name_, allowed )
      self.changeAllowed_ = allowed
      for attr in six.iterkeys( self.childReactors ):
         for reactor in six.itervalues( self.childReactors[ attr ] ):
            reactor.recursiveChangeAllowedIs( allowed )

   def close( self ):
      # Remove myself from the global list of EntityTreeReactors.
      if hasattr( localMonitor, 'allReactors' ):
         del localMonitor.allReactors[ self.name_ ]
      for attr in self.childReactors.keys():
         for key in list( self.childReactors[ attr ] ):
            self.childReactors[ attr ][ dictKey( key ) ].close()
            del self.childReactors[ attr ][ dictKey( key ) ]
      Tac.Notifiee.close( self )

   def attrAllowed( self, attr, entity, attrName ): # pylint: disable-msg=R0201
      t1( '   attr', attr, 'entity', entity )
      # For Tac::Dir objects, only entityPtr collection is interesting.
      if entity.tacType.fullTypeName == "Tac::Dir":
         return attr.name == 'entityPtr'
      if not ( attr.instantiating and
               attr.memberType.isEntity and
               getattr( entity, attrName ) is not None ):
         return False
      return True

   def showSummary( self, showDescendents=False ):
      if any( self.attrChangeCounter_.values() ) :
         print( "For entity", self.notifier_.fullName, ", last change @",
                self.lastAttrChangeAt )
         for attr, count in six.iteritems( self.attrChangeCounter_ ):
            print( '  attr "%s"' % attr, ' changed', count, 'times' )
         print( "\n" )
      if showDescendents:
         for attrName in self.childReactors.keys():
            for key in self.childReactors[ attrName ].keys():
               self.childReactors[ attrName ][ dictKey( key ) ].showSummary(
                     showDescendents=showDescendents )

   def onAttribute( self, attr, key ):
      attrName = attr.name
      tacNow = Tac.now()
      entity = self.notifier_
      t1( 'attr', attr, ',name', attrName, ',in entity', entity.fullName )
      self.attrChangeCounter_[ attrName ] = 1 + self.attrChangeCounter_.setdefault(
                                                          attrName, 0 )
      self.lastAttrChangeAt = tacNow
      localMonitor.lastAttrChangeAt = tacNow

      if attr.isCollection:
         value = getattr( entity, attrName ).get( key )
         t0( 'entity: %s, attr: "%s", key: %s changed (%s)' % (
               entity.fullName, attrName, key,
               'removed' if value is None else 'added/modified' ) )
         if localMonitor.loggingEnabled:
            sys.stderr.write( str( Tac.now() ) + 
                  ': entity: %s, attr: "%s", key: %s %s\n' % (
                  entity.fullName, attrName, key,
                  'removed' if value is None else 'added/modified' ) )
      else:
         assert key is None
         value = getattr( entity, attrName )
         t0( 'entity: %s, attr: "%s" changed to %s' % (
               entity.fullName, attrName, str( value ) ) )
         if localMonitor.loggingEnabled:
            sys.stderr.write( str( Tac.now() ) +
                  ': entity: %s, attr: "%s" changed to %s\n' % (
                  entity.fullName, attrName, str( value ) ) )

      if not ( self.changeAllowed_ or attrName in self.changeAllowedAttrs_ ):
         keyStr = '' if key is None else ', key: "%s"' % key
         sys.stderr.write('entity: %s, attr: "%s"%s unexpectedly changed to %s\n' % (
               entity.fullName, attrName, keyStr, str( value ) ) )
         raise EntityTreeReactorUnexpectedChange

      # Add an entity reactor for this new entity if all conditions met
      if not self.attrAllowed( attr, entity, attrName ):
         t1( 'ignore' )
         return
      if value is not None:
         self.addEntityReactor( attrName, key, value )
      else:
         self.removeEntityReactor( attrName, key )
