#!/usr/bin/env python
# Copyright (c) 2006, 2007, 2008, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import Tac
import os

def sysname():
   try:
      return Tac.sysname()
   except: # pylint: disable-msg=W0702
      return 'ar'

def _cellStatus():
   """Get or create an object at /<sysname>/cell which exposes the
   cellType and cellId from C++."""
   sysroot = Tac.root.get( sysname() )
   if not sysroot:
      sysroot = Tac.root.newEntity( 'Tac::Dir', sysname() )
   cellStatus = sysroot.get( 'cell' )
   if not cellStatus:
      cellStatus = sysroot.newEntity( 'Cell::Status', 'cell' )
   return cellStatus

def cellId():
   """Return the cellId of the current cell, instantiating the local
   object /ar/cell if necessary"""
   return _cellStatus().cellId

def peerCellId( thisCellId=None ):
   """Return the peer cell id; note, this generally does not work
   in breadth tests, as it assumes the cellId is either 1 or 2.
   Use it only when you know you are on a physical switch!"""
   if thisCellId is None:
      thisCellId = cellId()
   return 3 - thisCellId

def cellType():
   """Return the cellType of the current cell, instantiating the local
   object /ar/cell if necessary"""
   return _cellStatus().cellType

def cellTypeIs( ipCellType ):
   _cellStatus().cellType = ipCellType 

def product():
   """Return the product corresponding to the system"""
   return _cellStatus().product

def electionMgrSupported():
   """Return if Election Manager is supported on a supervisor system, only valid with
   cellType supervisor"""
   return _cellStatus().electionMgrSupported

def cpuBmcMode():
   """Return if Cpu is running in BMC mode on a supervisor system, only valid with
   cellType supervisor"""
   return _cellStatus().cpuBmcMode

def _redundancyStatus():
   try:
      cellStr = str( cellId() )
      return Tac.root[ sysname() ][ 'Sysdb' ][ 'cell' ][ cellStr ]\
          [ 'redundancy' ][ 'status' ]
   except KeyError:
      return None

def activeCell():
   """Return the cellId of the active cell from Sysdb.  If 
   cell/<cellId>/redundancy/status.activeCellId is invalid, check 
   os.environ[ "ACTIVE_CELLID" ].  Cannot cache value because it 
   changes over time."""

   redundancyStatus = _redundancyStatus()
   if redundancyStatus and \
          redundancyStatus.activeCellId != redundancyStatus.invalidCellId:
      return redundancyStatus.activeCellId
   else:
      # Look to see if test put active cell into env.
      ac = os.environ.get( "ACTIVE_CELLID" )
      if ac is not None:
         return int( ac )
      # If we are running on a RedSup namespace dut, REDSUPDUT_LOCAL_REDMODE and
      # REDSUPDUT_PEER_REDMODE will be set to the paths for the respective files.
      # Otherwise check scd-em sysfs file
      localRedModePath = os.environ.get( "REDSUPDUT_LOCAL_REDMODE",
                               '/sys/arista/election_manager/local_redundancy_mode' )
      peerRedModePath = os.environ.get( "REDSUPDUT_PEER_REDMODE",
                               '/sys/arista/election_manager/peer_redundancy_mode' )
      myCellId = cellId()
      otherCellId = 3 - cellId() # XXX this is buggy since for tests
      # cell and slots are not same. However NorCalInit has the same problem.
      # Usually this is not a problem, since tests only use the fact that whether
      # the active cell is same as local cellId or not.
      if os.path.exists( localRedModePath ) and \
            os.path.exists( peerRedModePath ):
         if open( localRedModePath ).read() != 'active':
            return otherCellId
         # We are active. I don't know for sure in what order ElectionMgr
         # writes to the files, and asserts the wires. I will play somewhat
         # afe for now.
         if open( peerRedModePath ).read() == 'active':
            return otherCellId
         return myCellId

      # Assume I am active.
      return myCellId
   
def path( mountpath, cell=None ):
   # Convert a mount path to per-cell, so we can do

   # mg.mount( Cell.path( 'sys/net/status' ) )
   if cell is None:
      cell = cellId()
   return "cell/%d/%s" % ( cell, mountpath )

def mountpath( mountpath, cell=None ):
   # Convert a mount path to per-cell, but with the cell directory
   # specified as a mount path.
   if cell is None:
      cell = cellId()
   return "cell/%d*/%s" % ( cell, mountpath )

def root( sysdbRoot, cell=None ):
   # Convert sysdb root to the root of per-cell directory
   if cell is None:
      cell = cellId()
   return sysdbRoot[ 'cell' ][ str( cell ) ]


