#!/usr/bin/env python
# Copyright (c) 2008, 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import os
import six
import sys

# TODO: We should update the packages to directly use ArPyUtils and then
# remove these aliases. BUG174510
from ArPyUtils.Collections import shuffled # pylint: disable-msg=W0611
from ArPyUtils.Decorators import Extendable # pylint: disable-msg=W0611
from ArPyUtils.Decorators import memoize # pylint: disable-msg=W0611
from ArPyUtils.Decorators import overridable # pylint: disable-msg=W0611
from ArPyUtils.Decorators import revertableMemoize # pylint: disable-msg=W0611
from ArPyUtils.Decorators import unMemoize # pylint: disable-msg=W0611
from ArPyUtils.LockingFile import LockingFile # pylint: disable-msg=W0611

# Do not add an import of Tac or of something that indirectly imports
# Tac.  Ark is supposed to be lightweight to import and Tac is not.


class Template( object ):
   '''This is used to do variable substition into a string template.  Variables
   go into '<@ @>' sequences in the template string.  The code inside the sequence
   can be pretty much any python expression that eval can handle.

   You do the actual expansion by calling the expand method.  Pass it
   a dictionary of variable names with values. '''

   def __init__( self, templateStr ):
      import re
      self.templateStr_ = templateStr
      self.pat_ = re.compile( "<@(.*?)@>" )

   def expand( self, varargs ):
      result = self.pat_.sub( lambda m: str( eval( m.group( 1 ), varargs ) ),
                              self.templateStr_ )
      return result

class FileTemplate( Template ):
   '''Slurp a template string in from a file.  You can then use the expand method
   to replace variables in the file contents with values.'''

   def __init__( self, templateFileName ):
      with open( templateFileName ) as f:
         lines = f.readlines()
      fileString = "".join( lines )
      Template.__init__( self, fileString )

sysmod = sys.modules
def _getmod( name, hook ):
   if not name in sysmod:
      d = {}
      six.exec_( "import " + name, d )
      if hook:
         hook(sysmod[name])
   return sysmod[ name ]

class LazyModule( object ):
   """LazyModule behaves just like an imported module, but it does not
   actually import the requested module until some attribute of it is
   referenced.  You can use it like this:

   replace
     import rpm
   with
     rpm = LazyModule( 'rpm' )


   LazyModule forwards getattr, setattr, and hasattr to the real
   module, but with introspection (type, dir, isinstance), of course,
   the two are different.

   The importHook is a one-argument function to invoke immediately
   after the module is first imported.  The argument is the
   just-imported module.  This is useful for replacing top-level code
   that imports a module and then configures it, like this:

     import rpm
     rpm.setVerbosity( 1 )

   with

     def hook( rpm ): rpm.setVerbosity(1)
     rpm = LazyModule( 'rpm', hook )

   """

   __slots__ = ['__lazy__', '__hook__']
   def __init__( self, name, importHook=None ):
      object.__setattr__( self, '__lazy__', name )
      object.__setattr__( self, '__hook__', importHook )
   def __getattr__( self, x ):
      return getattr( _getmod( self.__lazy__, self.__hook__ ), x )
   def __setattr__( self, x, value ):
      return setattr( _getmod( self.__lazy__, self.__hook__ ), x, value )
   def __hasattr__( self, x ):
      return hasattr( _getmod( self.__lazy__, self.__hook__ ), x )
   def __repr__( self ):
      return "<LazyModule %s (%sloaded)>" % \
             (self.__lazy__, '' if (self.__lazy__ in sysmod) else "not " )

import configparser

class HelpfulConfigParser( configparser.ConfigParser ):
   def optionxform( self, option ):
      """Basically unity transform: the default is to lowercase"""
      return str( option )

   #pylint: disable-msg=W0221
   def get( self, section, option, default=None, **kwargs ):
      """Behave like dict.get"""
      try:
         # convert from unicode string to str
         return str( configparser.ConfigParser.get(
            self, section, option, **kwargs ) )
      except configparser.NoOptionError:
         return default
      except configparser.NoSectionError:
         return default

   def getitems( self, section, default=() ):
      """items with a default set of values"""
      try:
         # convert from unicode strings to str
         return [ ( str( k ), str( v ) ) for k, v in self.items( section ) ]
      except configparser.NoSectionError:
         return default

   if six.PY2:
      def set( self, section, option, value=None ):
         # Convert non-unicode strings in Python 2.
         # This prevents a DeprecationWarning for arguments with str type.
         if isinstance( section, bytes ):
            section = section.decode()
         if isinstance( option, bytes ):
            option = option.decode()
         if isinstance( value, bytes ):
            value = value.decode()
         super( HelpfulConfigParser, self ).set( section, option, value )

      def add_section( self, section ):
         # Convert non-unicode strings in Python 2.
         # This prevents a DeprecationWarning for arguments with str type.
         if isinstance( section, bytes ):
            section = section.decode()
         super( HelpfulConfigParser, self ).add_section( section )

#pylint: disable-msg=W0231
class LockingConfigParser( HelpfulConfigParser, LockingFile ):
   """A "with" helper
   When you create one, it reads and parses the given config file.
   When you delete it, it rewrites the modified config file.
   It can be otherwise treated as a ConfigParser object.
   e.g.,

   with LockingConfigParser( 'foo.txt' ) as c:
      c.set( 'general', 'foo', 'bar' )

   will make sure foo.txt contains "foo=bar" and all other options
   remain unchanged.
   """
   def __init__( self, filename ):
      LockingFile.__init__( self, filename, 0o666 )

   def __enter__( self ):
      LockingFile.__enter__( self )

      configparser.ConfigParser.__init__( self )
      self.read_file( self.file_, self.filename_ )
      if not self.has_section( "general" ):
         self.add_section( "general" )

      return self

   def __exit__( self, typeArg, value, tb ):
      self.file_.seek( 0 )
      self.write( self.file_ )
      self.file_.truncate( )
      LockingFile.__exit__( self, typeArg, value, tb )
      return False

class LockingConfigParserMixin( object ):
   """ Inherit from this class to reap the benefits of
   HelpfulConfigParser and LockingConfigParser.
   Requires a configFileName() method."""

   def readConfig( self ):
      #pylint: disable-msg=E1101
      filename = self.configFileName()
      config = HelpfulConfigParser()
      with LockingFile( filename, 0o666 ) as _lf:
         try:
            config.read_file( open( filename ), filename )
         except IOError:
            pass
      return config

   def writeConfig( self ):
      #pylint: disable-msg=E1101
      return LockingConfigParser( self.configFileName() )

def findInPath( filename, pathEnvVarName, accessFlags=os.F_OK ):
   """Searches for a file named 'filename' in the path contained in the environment
   variable named 'pathEnvVarName'.  If the filename contains any slashes, it is
   interpreted relative to the current directory and the path is ignored.

   To look for a file with a specific set of access permissions, 'accessFlags' may be
   set to the OR of some or all of os.R_OK, os.W_OK and os.X_OK.

   A ValueError is raised if the file cannot be found, or if it is found but doesn't
   have the required access permissions."""

   if '/' in filename:
      abspath = os.path.abspath( filename )
      if os.path.exists( abspath ):
         if os.access( abspath, accessFlags ):
            return abspath
         else:
            raise ValueError( "%s: permission denied" % filename )
      else:
         raise ValueError( "%s: no such file or directory" % filename )
   else:
      path = os.environ[ pathEnvVarName ]
      for p in path.split( ":" ):
         abspath = os.path.abspath( os.path.join( p, filename ) )
         if os.access( abspath, accessFlags ):
            return abspath
      raise ValueError( "%s: not found" % filename )

def getPlatform():
   platform = os.environ.get( 'EOS_PLATFORM' )
   if platform:
      return platform
   import re
   cmdlineFilename = '/proc/cmdline'
   cmdlineFile = open( cmdlineFilename )
   m = re.search(
         r"platform=(\w+)",
         cmdlineFile.read() )
   if m:
      return m.group( 1 )
   return None

def dumpEnvInfo( pid, out ):
   env = open( os.path.join( '/proc', pid, 'environ' ) ).read().split( '\x00' )
   print( "Process environment is", file=out )
   for i in env:
      print( "    ", i, file=out )

def dumpPortUserInfo( port, out ):
   """ Dump information about the process that is bound to TCP port."""
   import Tac
   try:
      output = Tac.run( ["lsof", "-i", "tcp:"+str(port), '-FpcLgRu'],
                        stdout=Tac.CAPTURE )
   except Tac.SystemCommandError:
      print( "lsof failed to find the process bound to port ",
             port, " -- perhaps it exited?", file=out )
      return
   data = {}
   for l in output.split('\n'):
      if l:
         data[l[0]] = l[1:]
   fields = [ 'ppid', 'ggroup', 'Rppid',
              'ccommand', 'uuid',
              'Lusername' ]
   for what in fields:
      d = data.get( what[0] )
      if d:
         print( "  %-10s" % what[ 1 : ], ":", d, file=out )

   dumpEnvInfo( data[ 'p' ], out )

def dumpSocknameUserInfo( sockname, out ):
   """Dump information about the process that is bound to Unix socket.
   'sockname' is a unix socket name, 'out' is a file-object where this
   information has to be written."""
   import Tac, re
   # lsof does not display linux abstract socket names, it just says 'socket',
   # so we use netstat instead
   try:
      output = Tac.run( ["netstat", "-xpa"],
                        stdout=Tac.CAPTURE )
   except Tac.SystemCommandError:
      print( "netstat failed", file=out )
      return
   # strip away first character if it is null
   if sockname and ( sockname[ 0 ] == '\x00' ):
      sockname = sockname[ 1: ]
   for l in output.split('\n'):
      w = l.split()
      if w and ( re.search( "%s$" % sockname, w[ -1 ] ) ):
         try:
            ( pid, proc ) = w[ -2 ].split( "/" )
         except ValueError:
            continue
         print( "pid:", pid, "process:", proc, file=out )

         dumpEnvInfo( pid, out )

def timestampToStr( timestamp, relative=True, now=None ):
   """If relative is True, converts a timestamp (in seconds, as
   returned by Tac.now() or now, if specified), to a string of the form
   '32 seconds ago'. Else, converts the timestamp to a string of the absolute
   time (based on the current timezone configuration)."""
   import Tac, datetime
   if now is None:
      now = Tac.now()
   if not timestamp:
      return 'never'
   elif relative:
      td = datetime.timedelta( seconds=int( now - timestamp ) )
      return str( td ) + ' ago'
   else:
      td = datetime.datetime.fromtimestamp( Tac.utcNow() - ( now - timestamp ) )
      return td.strftime( '%Y-%m-%d %H:%M:%S' )

def utcTimeRelativeToNowStr( utcTime ):
   '''Returns a string of the form "[time] ago" describing utcTime relative to now.
   '''
   import Tac, datetime
   if not utcTime:
      return 'never'
   td = datetime.timedelta( seconds=int( Tac.utcNow() - utcTime ) )
   return str( td ) + ' ago'

def switchTimeToUtc( switchTime ):
   import Tac
   if switchTime == 0.0:
      return switchTime
   return switchTime + Tac.utcNow() - Tac.now()

def dateTimeToSwitchTime( dateTimeInstance ):
   """ Return switchTime corresponding to datetime instance provided as argument.
   switchTime can be negative if dateTimeInstance was before switch was up."""
   from datetime import datetime
   import Tac
   dateTimeNow = datetime.now()
   switchTimeNow = Tac.now()
   dateTimeDelta = dateTimeInstance - dateTimeNow
   switchTime = switchTimeNow + dateTimeDelta.total_seconds()
   return switchTime

def configureLogManager( name ):
   """ By default, the logging feature provided by the tacc
   package sends all log messages to the user facility. However,
   for EOS software we send all messages to the Local4 facility.
   This allows us to filter all messages to /var/log/eos as well
   as /var/log/messages. All processes that want to log messages
   that will be visible to the user through 'show logging' must
   call this function to properly configure the log manager.
   For agents, this is done automatically in the AgentContainer
   class below """
   import Tac
   logManager = Tac.singleton( 'Tac::LogManager' )
   logManager.syslogProcTitle = name
   logManager.syslogFacility = 'logLocal4'
   return logManager

from contextlib import contextmanager
@contextmanager
def timing(s):
   import time
   t0 = time.time()
   yield
   print( "%s%s" % ( ( ( s + ": " ) if s else "" ), time.time() - t0 ) )

def atExit( func ):
   """Ark.atExit() registers a python callback that will get called
   even if the parent process exits hard.  It uses fork to achieve
   this, so the exit handler runs in a copy of the address space
   as it was when atExit was called

   It is relatively heavyweight -- it forks a child and uses
   setpdeathsig to achieve this.  However, it is much more reliable
   than the atexit module -- it ensures that the registered handler
   runs even with an abnormal exit of the interpreter, including by a
   SEGV, SIGKILL or os._exit()."""
   # A pipe is used to synchronize with the child process.  This
   # ensures that the caller's process does not return from atExit
   # until the child process that runs the exit handler has registered
   # itself with setpdeathsig
   ( r, w ) = os.pipe()
   cpid = os.fork()
   if not cpid:
      import signal, Tac
      signal.signal( signal.SIGINT, signal.SIG_IGN )
      def handleQuit( signum, frame ):
         func()
         #pylint: disable-msg=W0212
         os._exit( 0 )
      signal.signal( signal.SIGQUIT, handleQuit )
      Tac.setpdeathsig( signal.SIGQUIT )
      os.close(r)
      os.close(w)
      while True:
         signal.pause()
   os.close( w )
   os.read( r, 1 )
   os.close( r )


def installBacktraceHandlers():
   """(Re) install the signal handlers provided by libTac that
   generate a backtrace.  This is useful if, for instance, the signal
   handler was temporarily overwritten by python and we want to
   restore the backtrace-generating behavior.  The following handlers
   are overwritten to one that dumps a backtrace, as of 2010-11-15:

     SIGSEGV, SIGFPE, SIGQUIT, SIGILL, SIGABRT, SIGBUS, SIGSYS,
     SIGTRAP, SIGXCPU, SIGXFSZ, and SIGUSR1

   Each of these, except SIGUSR1, generates a backtrace and then
   returns from the signal handler with the default (SIG_DFL) signal
   handler installed.  This will normally cause the program to retry
   the signal-generating instruction and exit, except for SIGUSR1
   which simply generates a backtrace and keeps going."""
   import ctypes
   ark = ctypes.cdll.LoadLibrary( 'libArk.so' )
   ark.Ark_installBacktraceHandlers()

def unpage():
   """Remove existing mappings from the page tables.
   This command walks through the existing mappings and if
   it is safe it removes the mapping from the pages tables."""
   import ctypes
   ark = ctypes.cdll.LoadLibrary( 'libArk.so' )
   ark.Ark_unpage()

class ReversibleDict( dict ):
   """ReversibleDict is just like a read-only dictionary,
   but can cache the reversal for the key and value of the
   dictionary. The restriction is that the dictionary must
   have unique values."""
   r = None
   def __setitem__( self, key, value ):
      assert 0, 'read-only attribute'
   def __delitem__( self, key, value ):
      assert 0, 'read-only attribute'
   def reverse( self ):
      if self.r is None:
         self.r = {}
         for k, v in six.iteritems( self ):
            if v in self.r:
               raise ValueError( 'Duplicate key %s found' % v )
            self.r[ v ] = k
      return self.r

class synchronized( object ):
   """decorator to synchronize access to a function.
   An explicit lock can be passed, or it will automatically create one."""

   def __init__( self, lock=None ):
      if not lock:
         import threading
         lock = threading.Lock()
      self.lock_ = lock

   def __call__( self, func ):
      def func_wrapper( *args, **kwargs ):
         with self.lock_:
            return func( *args, **kwargs )

      return func_wrapper

