#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
A pure python tracing facility along the lines of tacc Fwk Tracing.

To use this module::

    from ArPyUtils.ArTrace import Tracing
    handle = Tracing.Handle( "MyModule" )
    
    handle.trace0( "A message from the past" )
    
    t8 = handle.trace8
    t8( "A level 8 message from a more recent past" )

The environment variables TRACE and TRACEFILE control what to trace where to
send the trace output.

"TRACE" takes the format <facility>/<level0>..<leveln>,... where facility is
the string used to create the trace handle and level is a number indicating
the levels for which tracing has to be enabled.

"export TRACE=MyModul*/08,TheirModule/345" will enable level 0 and 8 tracing
for all handles which start with the string "MyModul" and levels 3, 4 and 5
for the handle "TheirModule".

"export TRACE=*/*" will enable all handles and all levels.

By default the trace output is printed out to stderr. If "TRACEFILE" is set,
it says the file to which the trace output should be redirected.

"export TRACEFILE=/var/tmp/trace.out" will redirect all trace output to the
file "/var/tmp/trace.out".
"""

from __future__ import absolute_import, division, print_function
import os
import sys
import inspect
from datetime import datetime
import time
from os import path
from fnmatch import fnmatch

class Tracing( object ):
   """
   A pure python substitute for the tacc Tracing module. It doesn't support
   level ranges. Everything else should be a direct substitute - more or less.
   """
   traces = {}
   traceFile = None

   @classmethod
   def expandLevelSpec( cls, levelSpec ):
      level = 0
      if levelSpec == "*":
         level = 0xffff
      elif levelSpec.isdigit():
         for l in levelSpec:
            level |= 1 << int( l )
      return level

   @classmethod
   def traceEnable( cls ):
      traceEnableRe = os.getenv( "TRACE" )
      if traceEnableRe:
         for traceSpec in traceEnableRe.split( "," ):
            # TRACE=MyModule should enable all trace levels
            if '/' not in traceSpec:
               traceSpec = traceSpec + '/*'
            try:
               expr, level = traceSpec.split( "/" )
               cls.traces[ expr ] = cls.expandLevelSpec( level )
            except Exception: # pylint: disable-msg=W0703
               # That handle won't be enabled
               pass

   @classmethod
   def traceFileEnable( cls ):
      traceFile = os.getenv( "TRACEFILE" )
      if not traceFile:
         cls.traceFile = sys.stdout
      else:
         cls.traceFile = open( traceFile, "a" )

   @classmethod
   def Handle( cls, handleName ):
      for trace, levelSpec in cls.traces.items():
         if fnmatch( handleName, trace ):
            return Tracing( handleName, levelSpec )
      return Tracing( handleName, 0 )

   def makeTrace( self, level, *args ): # pylint: disable=unused-argument
      return lambda *args: self.trace( level, *args ) 

   def __init__( self, name, levelMask ):
      """
      Only those traces which match the level specification will point to a
      valid print method.
      """
      self.name = name
      level = 0
      while levelMask:
         if levelMask & 0x1:
            setattr( self, "trace{}".format( level ),
                     self.makeTrace( level, name ) )
         levelMask >>= 1
         level += 1

   # These are the default trace handlers. They don't do anything. When a
   # trace handle is created, these methods will be over-written by methods
   # created by 'makeTrace', as required by the level and handle.

   # pylint: disable-msg=C0321
   def trace0( self, *args ): pass
   def trace1( self, *args ): pass
   def trace2( self, *args ): pass
   def trace3( self, *args ): pass
   def trace4( self, *args ): pass
   def trace5( self, *args ): pass
   def trace6( self, *args ): pass
   def trace7( self, *args ): pass
   def trace8( self, *args ): pass
   # pylint: enable-msg=C0321

   def trace( self, level, *args ):
      self.traceImpl( level, self.name, *args )

   @classmethod
   def traceImpl( cls, level, name, *args ):
      date = datetime.fromtimestamp( time.time() ).isoformat( " " )
      text = "{} {} {:20} {} ".format( date, os.getpid(), name, level )
      text += " ".join( [ str( x ) for x in args ] )
      print( text, file=cls.traceFile )
      cls.traceFile.flush()

   #
   # Default trace handling. This is treated separately to keep the code clean
   #
   @classmethod
   def defaultTraceHandle( cls, level, *args ):
      """
      This method is inspired by src/tacc/Fwk/Python/lib/Tracing.py

      It implements a kind of memoization. When called for the first time, we
      determine whether tracing must be enabled for that module and level.
      This decision is saved as the method to call for that module. Next time
      around we retreive the saved method and directly invoke it.
      """
      glob = sys._getframe( 1 ).f_globals # pylint: disable-msg=W0212
      traceArray = None
      try:
         traceArray = glob[ "__arTraceHandles__" ]
         traceFn = traceArray[ level ]
      except KeyError:
         # traceArray could be empty dict as well
         if traceArray == None:
            glob[ "__arTraceHandles__" ] = traceArray = {}

         moduleName = glob[ "__name__" ]
         if moduleName == "__main__":
            moduleFile = getattr( sys.modules[ moduleName ],
                                  "__file__", "__main__" )
            moduleName = path.splitext( path.basename( moduleFile ) )[ 0 ]
         levelBit = 2 ** level
         def ignore( *args ): # pylint: disable=unused-argument
            pass
         traceFn = ignore
         for trace, levelSpec in cls.traces.items():
            if fnmatch( moduleName, trace ) and ( levelSpec & levelBit ):
               traceFn = lambda *args: cls.traceImpl( level, moduleName, *args )
               break
         traceArray[ level ] = traceFn

      # Saved or newly created, call the method
      traceFn( *args )

Tracing.traceEnable()
Tracing.traceFileEnable()

# Create the module level trace methods t0 - t9 and trace0 - trace9
t0 = trace0 = lambda *args: Tracing.defaultTraceHandle( 0, *args )
t1 = trace1 = lambda *args: Tracing.defaultTraceHandle( 1, *args )
t2 = trace2 = lambda *args: Tracing.defaultTraceHandle( 2, *args )
t3 = trace3 = lambda *args: Tracing.defaultTraceHandle( 3, *args )
t4 = trace4 = lambda *args: Tracing.defaultTraceHandle( 4, *args )
t5 = trace5 = lambda *args: Tracing.defaultTraceHandle( 5, *args )
t6 = trace6 = lambda *args: Tracing.defaultTraceHandle( 6, *args )
t7 = trace7 = lambda *args: Tracing.defaultTraceHandle( 7, *args )
t8 = trace8 = lambda *args: Tracing.defaultTraceHandle( 8, *args )
t9 = trace9 = lambda *args: Tracing.defaultTraceHandle( 9, *args )

def _stackInfoTrace( traceFunc, addClass, *args, **kwargs ):
   stack = inspect.stack()
   # The call stack *should* be at least 3 deep at this point:
   #   stack[0]: _stackInfoTrace itself.,
   #   stack[1]: the lambda method from traceWithFunctionName,
   #   stack[2]: the caller where the interesting function name is present.
   # If the trace method is called directly from the module level, the function name
   # is just '<module>', which we suppress instead of inserting in the trace method.
   # Element [3] in a frame is the function name (see pydoc inspect.getframeinfo).
   if len( stack ) < 3 or stack[ 2 ][ 3 ] == '<module>':
      traceFunc( *args, **kwargs )
      return
   frame = stack[ 2 ]
   callerInfo = []
   if addClass:
      theSelf = frame[ 0 ].f_locals.get( 'self' )
      if theSelf:
         callerInfo.append( theSelf.__class__.__name__ )
   callerInfo.append( frame[ 3 ] )
   func = '.'.join( callerInfo ) + ':'
   traceFunc( func, *args, **kwargs )

def traceWithFunctionName( traceFunc, addClass=True ):
   '''
   traceWithFunctionName converts traceFunc to prefix all trace messages with the
   calling function func (and class name, where applicable, if addClass is True).

   t0 = traceWithFunctionName( ArTrace.trace0 )

   def abc( x ):
      t0( 'x is', x )

   Will produce "abc: x is ..."
   '''
   return lambda *args: _stackInfoTrace( traceFunc, addClass, *args )
