#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from jsonrpclib import Server
import socket, time
import subprocess
import logging
import yaml, sys, os
import logging.handlers

class YamlConfig:
   def __init__( self, yamlFile ):
      self.fileName = yamlFile
      self.FORMAT = '%(asctime)s %(process)d '\
            '%(filename)s:%(lineno)-5s %(levelname)-8s %(message)s'

      with open( self.fileName ) as f:
         self.dataMap = yaml.load( f )

      if self.dataMap:
         self.routerIPOnWAN = self.dataMap[ 'routerIPOnWAN' ]
         self.routerIPOnWANWithMask = self.dataMap[ 'routerIPOnWANWithMask' ]
         self.routerMacAddress = self.dataMap[ 'routerMacAddress' ]
         self.defaultGateway = self.dataMap[ 'defaultGateway' ]
         self.routerIPOnLAN = self.dataMap[ 'routerIPOnLAN' ]
         self.chosenPort = self.dataMap[ 'chosenPort' ]
         self.intfId = self.dataMap[ 'intfId' ]
         self.bridgeName = self.dataMap[ 'bridgeName' ]
         self.vNicId = self.dataMap[ 'vNicId' ]
         self.vIntfId = self.dataMap[ 'vIntfId' ]
         self.username = self.dataMap[ 'username' ]
         self.polltime = self.dataMap[ 'polltime' ]
         self.vEOSDomainName = self.dataMap[ 'vEOSDomainName' ]
         self.logFile = self.dataMap[ 'logFile' ]
         self.logLevel = self.dataMap[ 'logLevel' ]

# State of vEOS
# could just be a bool
class State:
   Down = 'Down'
   Up = 'Up'

class VEosMonitor():
   def __init__( self, config ):
      self.config = config
      self.serverState = None
      self.pollCount = 0
      socket.setdefaulttimeout(10)

   def capi( self, cmds ):
      dut = self.config.routerIPOnLAN
      output = {}
      try:
         serverUrl = "http://%s:@%s/command-api" % \
               ( self.config.username, dut )
         logging.info( cmds )
         logging.info( 'Server is at %s' % serverUrl )
         veos = Server( serverUrl )
         output = veos.runCmds( 1, [ ] + cmds )
      except socket.error:
         pass
      logging.info( output )
      return output

   def shell( self, cmds ):
      cmds = [ "sudo" ] + cmds
      output, outputerror = subprocess.Popen( cmds,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE ).communicate()
      logging.info( 'Executing as root %s ==> %s, %s' %
            ( cmds, output, outputerror ) )
      return output


   def getVersion( self ):
      return self.capi( [ 'show version' ] )

   def flapVEOSIntf( self ):
      self.capi( [ 'enable',
            'configure', 'interface %s' % self.config.vIntfId,
            'shutdown']  )
      time.sleep( 2 )
      self.capi( [ 'enable',
            'configure', 'interface %s' % self.config.vIntfId,
            'no shutdown'] )
      time.sleep( 2 )
      self.capi( [ 'enable', 'ping %s' % self.config.defaultGateway ] )

   def addIntfToBridge( self, remove=False ):
      action = "addif"
      if remove:
         action = "delif"
      self.shell(
         [ "brctl",
            "%s" % action,
            "%s" % self.config.bridgeName,
            "%s" % self.config.intfId ] )

   def addIpAddrToServerIntf( self, remove=False ):
      action = "add"
      if remove:
         action = "del"
      self.shell(
         [ "ip", "addr",
            "%s" % action,
            "%s" % self.config.routerIPOnWANWithMask,
            "dev",
            "%s" % self.config.intfId ] )

   def sendGratituousArp( self ):
      self.shell(
         [ "arping", "-c", "5", "-AI",
            "%s" % self.config.intfId,
            "%s" % self.config.routerIPOnWAN ] )

   def defaultRouteWhenDown( self ):
      self.shell(
         [ "ip", "route", "del", "0.0.0.0/0", "via",
            "%s" % self.config.routerIPOnLAN ] )
      self.shell(
         [ "ip", "route", "add", "0.0.0.0/0", "via",
            "%s" % self.config.defaultGateway ] )

   def defaultRouteWhenUp( self ):
      self.shell(
         [ "ip", "route", "del", "0.0.0.0/0", "via",
            "%s" % self.config.defaultGateway ] )
      self.shell(
         [ "ip", "route", "add", "0.0.0.0/0", "via",
            "%s" % self.config.routerIPOnLAN ] )

   def addFilters( self ):
      self.shell(
         [ "iptables", "-A", "INPUT", "-p", "tcp", "-i", "%s" % self.config.intfId,
            "--dport", "%s" % self.config.chosenPort, "-j", "ACCEPT" ] )
      self.shell(
         [ "iptables", "-A", "INPUT", "-p", "tcp", "-i", "%s" % self.config.intfId,
            "--dport", "ssh", "-j", "DROP" ] )

   def removeFilters( self ):
      self.shell(
         [ "iptables", "-D", "INPUT", "-p", "tcp", "-i", "%s" % self.config.intfId,
            "--dport", "%s" % self.config.chosenPort, "-j", "ACCEPT" ] )
      self.shell(
         [ "iptables", "-D", "INPUT", "-p", "tcp", "-i", "%s" % self.config.intfId,
            "--dport", "ssh", "-j", "DROP" ] )

   def applyRules( self ):
      logging.info( 'Applying Rules as VEos is down.' )
      self.addIntfToBridge( remove=True )
      self.addIpAddrToServerIntf()
      self.defaultRouteWhenDown()
      self.sendGratituousArp()
      self.addFilters()

   def removeRules( self ):
      logging.info( 'Removing Rules as VEos is up.' )
      self.addIpAddrToServerIntf( remove=True )
      self.removeFilters()
      self.addIntfToBridge()
      self.defaultRouteWhenUp()
      for i in range( 0, 3 ):
         time.sleep( 1 )
         self.flapVEOSIntf()

   def vEOSUp( self ):
      output = self.getVersion()
      if not output:
         return State.Down
      else:
         logging.info( 'System Mac Obtained: %s, expected %s' %  (
               output[0]['systemMacAddress'], self.config.routerMacAddress ) )
         assert output[0]['systemMacAddress'] == self.config.routerMacAddress
         return State.Up

   def intfState( self ):
      statusFile = '/sys/class/net/%s/operstate' % self.config.intfId
      logging.info( 'Checking status in file %s' % statusFile )
      if not os.path.exists( statusFile ):
         logging.error( 'Does interface %s exists?' % self.config.intfId )
      else:
         with open( statusFile, 'r' ) as f:
            state = f.read()
            if 'up' in state:
               return State.Up
      return State.Down

   def vEOSIntfUp( self ):
      cmd = ['virsh', 'domif-setlink', '%s' % self.config.vEOSDomainName,
            '%s' % self.config.vNicId, 'up' ]
      self.shell( cmd )

   def vEOSIntfDown( self ):
      cmd = ['virsh', 'domif-setlink', '%s' % self.config.vEOSDomainName,
            '%s' % self.config.vNicId, 'down' ]
      self.shell( cmd )

   def run( self ):
      while True:
        self.poll()
        time.sleep( self.config.polltime )

   def poll( self ):
      self.pollCount += 1
      logging.info( 'Checking for VEOS State' )
      serverNewState = self.vEOSUp()
      logging.info( 'System is %s' % serverNewState )
      if serverNewState != self.serverState:
         self.serverState = serverNewState
         if self.serverState == State.Up:
           self.removeRules()
         else:
           self.applyRules()
      logging.info( 'Checking for Interface State'  )
      state = self.intfState()
      logging.info( 'Interface state is %s, propagating this state' % state )
      if state == State.Down:
         self.vEOSIntfDown()
      else:
         self.vEOSIntfUp()

def main( args ):
   if os.path.exists( "/etc/veos/veosmonitor.yaml" ):
      config = YamlConfig( "/etc/veos/veosmonitor.yaml" )
   else:
      config = YamlConfig( "veosmonitor.yaml" )
   logging.basicConfig( format=config.FORMAT, level=config.logLevel,
                        filename=config.logFile )
   v = VEosMonitor( config )
   v.run()

if __name__ == '__main__':
    main( sys.argv )
