# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import string
from functools import wraps
import tabulate

def CliTable( hdr_fmt, rows=tuple() ):
   """returns a CliTable object with the headers/format given in the hdr_fmt.

   The `hdr_fmt` is a string like "{Mode:16} {Description:64}" which will
   create a table with columns `Mode` and `Description` with the first
   being 16 characters wide and the second 64 characters wide.
   """
   tbl = empty_text_table( hdr_fmt.split() )
   for r in rows:
      tbl.append( r )
   return tbl

class TextTable( object ):
   columns = []

   text_format = 'simple'

   def __init__( self, rows=None ):
      self._rows = rows or []

   def __str__( self ):
      return tabulate.tabulate(
          [ _pack_row( row, self.column_formats ) for row in self._rows ],
          self.headers,
          tablefmt=self.text_format
      )

   @property
   def headers( self ):
      return [ _infer_header( x )[ 0 ] for x in self.columns ]

   @property
   def column_formats( self ):
      return [ _subs_columns( _infer_header( x )[ 1 ], n )
              for n, x in enumerate( self.columns ) ]

   def append( self, item ):
      self._rows.append( item )

   def __len__( self ):
      return len( self._rows )

def empty_text_table( columns ):
   outer_columns = columns

   class AnonymousTextTable( TextTable ):
      columns = outer_columns

   return AnonymousTextTable()

def text_formatter( formatter ):
   def decorator( f ):
      @wraps( f )
      def wrapper( cli, *args, **kwargs ):
         output = f( cli, *args, **kwargs )
         if cli and not cli.json_api:
            output = formatter( output )
         return output

      return wrapper

   return decorator

def _pack_row( row_data, column_formats ):
   return [
       _unpack_args( fmt.format, row_data ) for fmt in column_formats
   ]

def _infer_header( column_info ):
   if isinstance( column_info, ( list, tuple ) ):
      return column_info
   else:
      assert isinstance( column_info, basestring )
      heading = ','.join( string.upper( s[ 0 ] ) + s[ 1 : ]
                         for s in _get_format_field_names( column_info ) )
      return heading, column_info

def _subs_columns( format_string, n ):
   return format_string.replace( '{?', '{' + str( n ) )

def _get_format_field_names( format_string ):
   format_scan = string.Formatter().parse( format_string )
   return [
       field_name
       for _, field_name, _, _ in format_scan
       if field_name is not None
   ]

def _is_args( x ):
   try:
      ( lambda *args, **kwargs: None )( *x )
      return True
   except TypeError:
      return False

def _is_kwargs( x ):
   try:
      ( lambda *args, **kwargs: None )( **x )
      return True
   except TypeError:
      return False

def _unpack_args( f, args ):
   if _is_kwargs( args ):
      return f( **args )
   if _is_args( args ):
      return f( *args )

   return f( args )
