#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from struct import pack_into
from struct import unpack_from
import array, re

# the Arrow Python row structure is identical to that of c++.
# the length, type, and variable length column fields are the same.

# max column and max row length are also defined in arrow_types.h. Please be
# sure to change them thre if they are changed here.
MaxColumnLength = 32767
MaxRowLength = 65535

def hexchar( val, fill=4 ):
   return hex( int( val ) )[2:].zfill( fill )

def canonicalizeV6Addr( address ):
   address = re.sub( '((?<=:)|(?<=^))0{1,3}(?=[0-9a-fA-F])', '', address ).lower()
   vals = address.split( ':' )
   startOfStreak = None
   endOfStreak = None
   longestZeroStreak = 1
   currStart = None
   for i, v in enumerate( vals ):
      if int( v, 16 ) == 0:
         if currStart is not None:
            pass
         else:
            currStart = i
      else:
         if currStart is not None:
            if i - currStart > longestZeroStreak:
               startOfStreak = currStart
               endOfStreak = i
               longestZeroStreak = i - currStart
            currStart = None
   if currStart is not None:
      if len( vals ) - currStart > longestZeroStreak:
         startOfStreak = currStart
         endOfStreak = None
   if startOfStreak is not None:
      if startOfStreak == 0:
         ret = [ '' ]
      else:
         ret = vals [ 0 : startOfStreak ]
      if endOfStreak is not None:
         ret += [ '' ] + vals[ endOfStreak : ]
      else:
         ret +=  [ ':' ]
      return ':'.join( ret )
   else:
      return address

def v4AddrStringToValue( address ):
   vals = address.split( "." )
   assert len( vals ) == 4
   for v in vals:
      assert int( v ) <= 255 and int( v ) >= 0
   return ( int( vals[ 0 ] ) << 24 ) | ( int( vals[ 1 ] ) << 16 ) | \
          ( int( vals[ 2 ] ) << 8 ) | int( vals[ 3 ] )

def v4AddrValueToString( address ):
   ret = ""
   for i in range( 4 ):
      ret += str( ( address >> ( 8 * ( 3 - i ) ) ) & 0xFF )
      if i < 3:
         ret += "."
   return ret

def v6AddrStringToValue( address ):
   vals = address.split( ':' )
   # handle embedded ipv4 address
   dotCount = vals[ -1 ].count( '.' )
   if dotCount > 0:
      assert dotCount == 3
      v4Vals = [ hexchar( v, 2 ) for v in vals[ -1 ].split( '.' ) ]
      v4Vals = [ i + j for i, j in zip( v4Vals[ :: 2 ], v4Vals[ 1 :: 2 ] ) ]
      vals = vals[ : -1 ] + v4Vals
   # expand elided zeros
   elidedZeroCount = address.count( '::' )
   assert elidedZeroCount <= 1
   if elidedZeroCount == 1:
      if vals[ 0 ] == '':
         vals[ 0 ] = '0'
      if vals[ -1 ] == '':
         vals[ -1 ] = '0'
      zeros = [ '0' for _ in range( 9 - len( vals ) ) ]
      i = vals.index( '' )
      vals = vals[ : i ] + zeros + vals[ i + 1 : ]
   # verify address is legal
   assert len( vals ) == 8
   for v in vals:
      assert len( v ) <= 4
      assert int( v, 16 ) <= 0xFFFF and int( v, 16 ) >= 0
   # construct address value
   return ( ( int( vals[ 0 ], 16 ) << 112 ) | ( int( vals[ 1 ], 16 ) << 96 ) |
            ( int( vals[ 2 ], 16 ) << 80 ) | ( int( vals[ 3 ], 16 ) << 64 ) |
            ( int( vals[ 4 ], 16 ) << 48 ) | ( int( vals[ 5 ], 16 ) << 32 ) |
            ( int( vals[ 6 ], 16 ) << 16 ) | int( vals[ 7 ], 16 ) )

def v6AddrValueToString( address ):
   ret = ""
   for i in range( 8 ):
      ret += hexchar( ( address >> ( 16 * ( 7 - i ) ) ) & 0xFFFF )
      if i < 7:
         ret += ":"
   return canonicalizeV6Addr( ret )

class AddressFamily( object ):
   ipunknown_ = 0
   ipv4_ = 1
   ipv6_ = 2

class AddressLength( object ):
   ethAddr_ = 6
   ipv4_ = 4
   ipv6_ = 16

class ArrowTypes:
   ar_uint8   = 0
   ar_uint16  = 1
   ar_uint32  = 2
   ar_uint64  = 3
   ar_int8    = 4
   ar_int16   = 5
   ar_int32   = 6
   ar_int64   = 7
   ar_float   = 8
   ar_double  = 9
   ar_bool    = 10
   ar_string  = 11
   ar_struct  = 12
   ar_oid     = 13
   ar_len_t   = 14
   ar_EthAddr = 15
   ar_IpAddr = 16
   ar_Ip6Addr = 17
   ar_IpGenAddr = 18
   ar_IpRouteKey = 19
   ar_IpGenPrefix = 20
   ar_bits = 20
   ar_MaxType = ar_bits

IPv6HostMask = ( 1L << 128 ) - 1L
IPv4HostMask = ( 1L << 32 ) - 1L

def swap32( x ):
   return (((x << 24) & 0xFF000000) |
           ((x <<  8) & 0x00FF0000) |
           ((x >>  8) & 0x0000FF00) |
           ((x >> 24) & 0x000000FF))

class EthAddr():
   _addressLengthInWords = 3

   # address is one of:
   # - a string representing a mac address, in colon-separated notation
   # - a value in network byte order
   def __init__( self, address ):
      if isinstance( address, str ):
         self.address_ = self.addrStringToValue( address )
      elif isinstance( address, ( int, long ) ):
         self._validateAddress( address )
         self.address_ = address
      else:
         raise ValueError( "EthAddr must be either a string representing a mac "
            "address in colon-separated notation, or a 48-bit value in network byte "
            "order " )

   def __len__( self ):
      return AddressLength.ethAddr_

   def __bool__( self ):
      return self.address() == 0

   def __eq__( self, o ):
      return self.address() == o.address()

   def __ne__( self, o ):
      # __ne__ delegates to __eq__ if the former is not defined, but here it is for
      # completeness
      return self.address() != o.address()

   def __lt__( self, o ):
      return self.address() < o.address()

   def __le__( self, o ):
      return self.address() <= o.address()

   def __gt__( self, o ):
      return self.address() > o.address()

   def __ge__( self, o ):
      return self.address() >= o.address()

   def __hash__( self ):
      return hash( self.address() )

   def __repr__( self ):
      return self.addrValueToString( self.address() )

   # Returns the address in network byte order
   def address( self ):
      return self.address_

   # internal helper method for validating address argument as a 48-bit integer
   def _validateAddress( self, address ):
      if address >> 48:
         raise ValueError( "address exceeds a 48-bit value" )

   # internal helper method for validating index arguments
   def _validateWordIndex( self, index ):
      if index > ( EthAddr._addressLengthInWords - 1 ) or index < 0:
         raise ValueError( "Valid indices are between 0 and " +
                           ( EthAddr._addressLengthInWords - 1 ) + ", inclusive" )

   # internal helper method for validating index arguments
   def _validateByteIndex( self, index ):
      if index > ( AddressLength.ethAddr_ - 1 ) or index < 0:
         raise ValueError( "Valid indices are between 0 and " +
                           ( AddressLength.ethAddr_ - 1 ) + ", inclusive" )

   # accessor for a half-word of the of the mac address in host byte order.
   # Index 0 represents the most-significant half-word.
   # Raises ValueError if the index is invalid.
   def word( self, index ):
      self._validateWordIndex( index )
      return ( 0xFFFF & ( self.address() >>
                           ( 16 * ( EthAddr._addressLengthInWords - 1 - index ) ) ) )

   # mutator for a half-word of the of the mac address in host byte order.
   # Index 0 represents the most-significant half-word.
   # Raises ValueError if the index or the wordVal is invalid.
   def wordIs( self, index, wordVal ):
      self._validateWordIndex( index )
      if wordVal >> 16:
         raise ValueError( "address exceeds a 16-bit value" )
      leftShiftBits = 16 * ( EthAddr._addressLengthInWords - 1 - index )
      wordMask = ~( 0xFFFF << leftShiftBits )
      self.address_ = ( self.address() & wordMask ) | ( wordVal << leftShiftBits )

   # accessor for a byte of the address.
   # Index 0 represents the most-significant byte.
   # Raises ValueError if the index is invalid.
   def byte( self, index ):
      self._validateByteIndex( index )
      return ( 0xFF & ( self.address() >>
                           ( 8 * ( AddressLength.ethAddr_ - 1 - index ) ) ) )

   # mutator for a byte of the address.
   # Index 0 represents the most-significant byte.
   # Raises ValueError if the index or the byteVal is invalid.
   def byteIs( self, index, byteVal ):
      self._validateByteIndex( index )
      if byteVal >> 8:
         raise ValueError( "address exceeds an 8-bit value" )
      leftShiftBits = 8 * ( AddressLength.ethAddr_ - 1 - index )
      byteMask = ~( 0xFF << leftShiftBits )
      self.address_ = ( self.address() & byteMask ) | ( byteVal << leftShiftBits )

   def addrStringToValue( self, address ):
      byteList = address.split( ':' )
      if len( byteList ) != AddressLength.ethAddr_:
         raise ValueError( "address must be a colon-separated string of the form "
                           "xx:xx:xx:xx:xx:xx" )
      ret = 0
      for i in xrange( AddressLength.ethAddr_ ):
         byteVal = int( byteList[ i ], 16 )
         if not ( byteVal <= 0xFF and byteVal >= 0 ):
            raise ValueError( "Octets must be between 0 and 0xFF inclusive" )
         leftShiftBits = ( AddressLength.ethAddr_ - 1 - i ) * 8
         ret |= ( byteVal << leftShiftBits )
      return ret

   def addrValueToString( self, address ):
      self._validateAddress( address )
      # print lower-case hex numbers, filled to a width of 2
      ret = '{:02x}:{:02x}:{:02x}:{:02x}:{:02x}:{:02x}'.format(
         *[ self.byte( i ) for i in xrange( AddressLength.ethAddr_ ) ] )
      return ret

class IpAddr( object ):
   # address is one of:
   # - a string representing an ip address
   # - a value in network byte order
   def __init__( self, address ):
      if isinstance( address, str ):
         self.address_ = self.addrStringToValue( address )
      else:
         self.address_ = address

   def __len__( self ):
      return AddressLength.ipv4_

   def __eq__( self, o ):
      return self.address() == o.address()

   def __ne__ ( self, o ):
      return self.address() != o.address()

   def __lt__ ( self, o ):
      return self.address() < o.address()

   def __hash__( self ):
      return hash( self.address() )

   def __repr__( self ):
      return self.addrValueToString( self.address() )

   def address( self ):
      return self.address_

   def addrStringToValue( self, address ):
      return v4AddrStringToValue( address )

   def addrValueToString( self, address ):
      return v4AddrValueToString( address )

class Ip6Addr( IpAddr ):
   def __init__( self, address ):
      IpAddr.__init__( self, address )

   def __len__( self ):
      return AddressLength.ipv6_

   def addrStringToValue( self, address ):
      return v6AddrStringToValue( address )

   def addrValueToString( self, address ):
      return v6AddrValueToString( address )

class IpGenAddr( object ):
   def __init__( self, address, af=None ):
      if isinstance( address, Ip6Addr ):
         assert af is None or af == AddressFamily.ipv6_
         self.af_ = AddressFamily.ipv6_
         self.address_ = address.address()
      elif isinstance( address, IpAddr ):
         assert af is None or af == AddressFamily.ipv4_
         self.af_ = AddressFamily.ipv4_
         self.address_ = address.address()
      elif isinstance( address, str ):
         if address.count( ':' ) > 0:
            assert af is None or af == AddressFamily.ipv6_
            self.address_ = v6AddrStringToValue( address )
            self.af_ = AddressFamily.ipv6_
         elif address.count( '.' ) > 0:
            assert af is None or af == AddressFamily.ipv4_
            self.address_ = v4AddrStringToValue( address )
            self.af_ = AddressFamily.ipv4_
         else:
            assert False
      else:
         assert af is not None
         self.af_ = af
         self.address_ = address
      self.isIPv6_ = self.af() == AddressFamily.ipv6_

   def __len__( self ):
      if self.isIPv6():
         return AddressLength.ipv6_
      else:
         return AddressLength.ipv4_

   def __eq__( self, o ):
      return self.address() == o.address() and  self.af() == o.af() 

   def __ne__( self, o ):
      return self.address() != o.address() or  self.af() != o.af() 

   # IPv4 is less than IPv6
   def __lt__( self, o ):
      if self.af() != o.af():
         return not self.isIPv6() 
      return self.address() < o.address()
   
   def __hash__( self ):
      return hash( ( self.address(), self.af() ) )

   def __repr__( self ):
      if self.isIPv6():
         return v6AddrValueToString( self.address() )
      else:
         return v4AddrValueToString( self.address() )

   def address( self ):
      return self.address_

   def af( self ):
      return self.af_

   def isIPv6( self ):
      return self.isIPv6_

class IpGenPrefixBase( object ):
   def __init__( self, address, mask=None, af=None ):
      if isinstance( address, IpGenAddr ):
         self.af_ = address.af()
         self.address_ = address.address()
      elif isinstance( address, Ip6Addr):
         assert af is None or af == AddressFamily.ipv6_
         self.af_ = AddressFamily.ipv6_
         self.address_ = address.address()
      # must test for IpAddr after Ip6Addr because of the class hierarchy.
      elif isinstance( address, IpAddr ):
         assert af is None or af == AddressFamily.ipv4_
         self.af_ = AddressFamily.ipv4_
         self.address_ = address.address()
      elif isinstance( address, str ):
         if mask is None:
            assert address.count( '/' )
            address, mask = address.split( '/' )
         if address.count( ':' ) > 0:
            assert af is None or af == AddressFamily.ipv6_
            self.address_ = v6AddrStringToValue( address )
            self.af_ = AddressFamily.ipv6_
         elif address.count( '.' ) > 0:
            assert af is None or af == AddressFamily.ipv4_
            self.address_ = v4AddrStringToValue( address )
            self.af_ = AddressFamily.ipv4_
      else:
         assert af == AddressFamily.ipv4_ or af == AddressFamily.ipv6_
         self.af_ = af
         self.address_ = address
      self.isIPv6_ = ( self.af() == AddressFamily.ipv6_ )
      assert mask is not None
      mask = int( mask )
      assert 0 <= mask and ( mask <= 128 if self.isIPv6() else mask <= 32 )
      self.mask_ = mask

   def __len__( self ):
      if self.isIPv6():
         return AddressLength.ipv6_ + 1
      else:
         return AddressLength.ipv4_ + 1

   def __eq__( self, o ):
      return ( self.address() == o.address() and
               self.mask() == o.mask() and
               self.af() == o.af() )

   def __ne__( self, o ):
      return ( self.address() != o.address() or
               self.mask() != o.mask() or
               self.af() != o.af() )

   def __hash__( self ):
      return hash( ( self.address(), self.mask(), self.af() ) )


   def __repr__( self ):
      if self.isIPv6():
         return "%s/%r" % ( v6AddrValueToString( self.address() ), self.mask() )
      else:
         return "%s/%r" % ( v4AddrValueToString( self.address() ), self.mask() )

   def address( self ):
      return self.address_

   def mask( self ):
      return self.mask_

   def af( self ):
      return self.af_

   def subnet( self ):
      subnet = None
      if self.isIPv6():
         subnet = self.address() & ( IPv6HostMask << ( 128 - self.mask() ) )
         return IpGenAddr( subnet, AddressFamily.ipv6_ )
      else:
         subnet = self.address() & ( IPv4HostMask << ( 32 - self.mask() ) )
         return IpGenAddr( subnet, AddressFamily.ipv4_ )

   def isIPv6( self ):
      return self.isIPv6_
   
class IpGenPrefix( IpGenPrefixBase ):
   # less proceeds as follows:
   # IPv4 is less than IPv6 (what about ipunknown?)
   # self under mask is less than other under mask
   # if equal under mask, longer mask is less
   def __lt__( self, o ):
      if self.af() != o.af():
         return not self.isIPv6()
      mm = min( self.mask(), o.mask() )
      mm = 128 - mm if self.isIPv6() else 32 - mm
      # mask out the high bits to ensure unsigned comparisons
      mMask = ~( ( 1 << mm ) - 1 ) & IPv6HostMask
      saddr = self.address()
      oaddr = o.address()
      smm = saddr & mMask
      omm = oaddr & mMask
      if ( smm < omm ):
         return True
      if ( smm == omm ):
         return self.mask() > o.mask()
      
class IpRouteKey( IpGenPrefixBase ):
   def __lt__( self, o ):
      if self.af() != o.af():
         return not self.isIPv6()
      if self.address() != o.address():
         return  self.address() < o.address()
      else:
         return self.mask() < o.mask()

class BaseRow(object):
   def __init__( self, row = None, length = None ):
      if row is not None:
         if isinstance( row, array.array ):
            self.buf_ =  row
         elif isinstance( row, str ):
            self.buf_ = array.array( 'B', row )
         elif isinstance( row, BaseRow ):
            self.buf_ = array.array( 'B', '' )
            self.expandRow( row.buf_ )
         else:
            assert False, "cannot instantiate a row from type %s" % type( row )
      elif length is not None:
         self.buf_ = array.array( 'B', '\x00' * length )
         self.insertLen()
      else:
         self.buf_ = None

   def __nonzero__(self):
      return not ( self.buf_ is None or len( self.buf_ ) == 0 )

   def buf( self ):
      return self.buf_

   # to satisfy pylint, we must have these definitions before first use in a function
   typeLength = {
         ArrowTypes.ar_uint8:1,
         ArrowTypes.ar_uint16:2,
         ArrowTypes.ar_uint32:4,
         ArrowTypes.ar_uint64:8,
         ArrowTypes.ar_int8:1,
         ArrowTypes.ar_int16:2,
         ArrowTypes.ar_int32:4,
         ArrowTypes.ar_int64:8,
         ArrowTypes.ar_float:4,
         ArrowTypes.ar_double:8,
         ArrowTypes.ar_bool:1,
         ArrowTypes.ar_string:4,       # this is the size of the fixed column
         ArrowTypes.ar_struct:4,       # this is the size of the fixed column
         ArrowTypes.ar_EthAddr : 6,
         ArrowTypes.ar_IpAddr:4,
         ArrowTypes.ar_Ip6Addr:16,
         ArrowTypes.ar_IpGenAddr:4,    # this is the size of the fixed column
         ArrowTypes.ar_IpRouteKey:4,    # this is the size of the fixed column
         ArrowTypes.ar_IpGenPrefix:4,    # this is the size of the fixed column
         # FIXME: we keep this undefined for ar_bits since python is not
         # supported for bitset columns yet. The definition should be adjust to
         # take arColumn's elts and then we should compute the size of the
         # bitset
         }

   ar_rowLenOffset = 0
   ar_rowTypeOffset = ar_rowLenOffset + typeLength[ArrowTypes.ar_uint16]

   def __len__( self ):
      return len( self.buf_ )

   # fixed accessors applicable to all rows
   def rowLength( self ):
      return self.extU16( self.ar_rowLenOffset )

   def rowType( self ):
      return self.extU16( self.ar_rowTypeOffset )
   def rowTypeIs( self, v ):
      self.insertU16( self.ar_rowTypeOffset, v )

   def toString( self ):
      if not self:
         return "len = 0, Null"
      else:
         return "len = %d, type = %d" % \
               ( self.rowLength(), self.rowType() )

   # variable column type (VCT) is an object which describes
   # a variable column.  Variable columns are arrays or types
   # which are variable length such as strings and IpGenPrefix.
   #
   # the format of a VCT is as follows:
   #
   # |<---- fld1 ----->|<---- fld2 ----->|
   # +--------+--------+--------+--------+
   # |oooooooo|oooooooo|allllllll|lllllll|
   # +--------+--------+--------+--------+
   #  where:
   #    lllllllllllllll  -> 15 bit total length of data including any headers or
   #                        descriptors
   #    a                -> 1 bit 1 -> is an array; 0 -> non-array variable length
   #    oooooooooooooooo -> 16 bit offset from beginning of data for
   #                        variable length elements or to the array header
   #                        for arrays of fixed or variable length elements
   #

   def decodeVCT( self, offset):
      off = self.extU16( offset )
      fld2 = self.extU16( offset + 2 )
      isArray = ( fld2 & 0x8000 ) != 0
      length =  fld2 & 0x7FFF 
      return ( isArray, off, length )

   def encodeVCT( self, isArray, offset, length ):
      fld1 = offset
      fld2 = ( length & 0x7fff ) | ( 0x8000 if isArray else 0 )
      return ( fld1,  fld2 )
  
   # array column header is an object which describes the contents
   # of a column which is an array of fixed or variable length types
   #
   # The format for the ACH is as follows:
   #
   # +--------+--------+
   # |xxxvnnnn|nnnnnnnn|
   # +--------+--------+
   #  where:
   #    xxx          -> 3 bits - reserved for future use
   #    v            -> 1 bit - 0 is array of fixed elements
   #                            1 is array of variable sized elements
   #    nnnnnnnnnnnn -> 12 bits - number of elements in array
   #
   
   def decodeACH( self, offset ):   
      fld = self.extU16( offset )
      isVariable = ( fld & 0x1000 ) != 0
      numElements = fld & 0xFFF
      return ( isVariable, numElements )
   
   def encodeACH( self, isVariable, numElements ):
      return ( numElements & 0xfff ) | ( 0x1000 if isVariable else 0 )
   
   # column extractors
   def extU8( self, offset ):
      return unpack_from( "<B", self.buf_, offset )[0]
   
   def extU16( self, offset ):
      return unpack_from( "<H", self.buf_, offset )[0]
   
   def extU32( self, offset ):
      return unpack_from( "<I", self.buf_, offset )[0]
   
   def extU64( self, offset ):
      return unpack_from( "<Q", self.buf_, offset )[0]

   def extI8( self, offset ):
      return unpack_from( "<b", self.buf_, offset )[0]
   
   def extI16( self, offset ):
      return unpack_from( "<h", self.buf_, offset )[0]
   
   def extI32( self, offset ):
      return unpack_from( "<i", self.buf_, offset )[0]
   
   def extI64( self, offset ):
      return unpack_from( "<q", self.buf_, offset )[0]

   def extBool( self, offset ):
      return unpack_from( "<?", self.buf_, offset )[0]

   def extFloat( self, offset ):
      return unpack_from( "<f", self.buf_, offset )[0]

   def extDouble( self, offset ):
      return unpack_from( "<d", self.buf_, offset )[0]

   def extString( self, offset, length ):
      if length == 0:
         return None
      else:
         return unpack_from( "<%ds" % ( length - 1 ), self.buf_, offset )[0]

   def strFrom( self, offset ):
      ( off, length ) = self.decodeVCT( offset )[1:3]
      return self.extString( off, length )

   # a string extracted from an array has an offset biased by
   # the origin of the descriptor vector.
   def varStrFrom( self, offset, hdrBase ):
      ( off, length ) = self.decodeVCT( offset )[1:3]
      return self.extString( off + hdrBase, length )

   def extSubrow( self, offset, cl ):
      ( off, length ) = self.decodeVCT( offset )[1:3]
      return cl( row=self.buf_[ off : off + length ])
   
   def varExtSubrow( self, offset, hdrBase, cl ):
      ( off, length ) = self.decodeVCT( offset )[1:3]
      return cl( row=self.buf_[ off + hdrBase : off + hdrBase + length ] )

   def extV4Addr( self, offset ):
      return swap32( self.extU32( offset ) )

   def extV6Addr( self, offset ):
      return ( ( swap32( self.extU32( offset     ) ) << 96 ) +
               ( swap32( self.extU32( offset + 4 ) ) << 64 ) +
               ( swap32( self.extU32( offset + 8 ) ) << 32 ) +
                 swap32( self.extU32( offset + 12) ) )

   def extV4orV6Addr( self, offset, length ):
      if ( length == AddressLength.ipv6_ ):
         return ( self.extV6Addr( offset ), AddressFamily.ipv6_ )
      elif ( length == AddressLength.ipv4_):
         return ( self.extV4Addr( offset ), AddressFamily.ipv4_ )
      else:
         assert False, 'invalid length: ' + str( length )

   def extEthAddr( self, offset ):
      addr = 0
      for i in xrange( 3 ):
         wordVal = self.extU16( offset + ( 2 * i ) )
         addr |= ( wordVal << ( 16 * ( 2 - i ) ) )
      return EthAddr( addr )

   def extIpAddr( self, offset ):
      return IpAddr( self.extV4Addr( offset ) )

   def extIp6Addr( self, offset ):
      return Ip6Addr( self.extV6Addr( offset ) )
  
   def extIpGenAddr( self, offset ):
      ( off, length ) = self.decodeVCT( offset )[1:3]
      return self.extIpGenAddrDetail( off, length )
   
   # a IpGenAddr extracted from an array has an offset biased by
   # the origin of the descriptor vector.
   def varExtIpGenAddr( self, offset, hdrBase ):
      ( off, length ) = self.decodeVCT( offset )[1:3]
      return self.extIpGenAddrDetail( off + hdrBase, length )

   def extIpGenAddrDetail( self, off, length ):
      if length == 0:
         return None
      else:
         return IpGenAddr( *self.extV4orV6Addr( off, length ) )

   def extIpRouteKey( self, offset ):
      ( off, length ) = self.decodeVCT( offset )[1:3]
      return self.extIpRouteKeyDetail( off, length )

   def extIpGenPrefix( self, offset ):
      ( off, length ) = self.decodeVCT( offset )[1:3]
      return self.extIpGenPrefixDetail( off, length )

   # a IpGenPrefix extracted from an array has an offset biased by
   # the origin of the descriptor vector.
   def varExtIpRouteKey( self, offset, hdrBase ):
      ( off, length ) = self.decodeVCT( offset )[1:3]
      return self.extIpRouteKeyDetail( off + hdrBase, length )

   def varExtIpGenPrefix( self, offset, hdrBase ):
      ( off, length ) = self.decodeVCT( offset )[1:3]
      return self.extIpGenPrefixDetail( off + hdrBase, length )

   def extIpRouteKeyDetail( self, off, length ):
      if length == 0:
         return None
      else:
         mask = self.extU8( off )
         ( address, af ) = self.extV4orV6Addr( off + 1, length - 1 )
         return IpRouteKey( address, mask, af )

   def extIpGenPrefixDetail( self, off, length ):
      if length == 0:
         return None
      else:
         mask = self.extU8( off )
         ( address, af ) = self.extV4orV6Addr( off + 1, length - 1 )
         return IpGenPrefix( address, mask, af )

   def extBits( self, off ):
      assert False, "extBits: Bitset columns are not supported in python"

   def dynTypeFrom( self, offset, arType ):
      if arType < 0 or arType > ArrowTypes.ar_MaxType:
         print 'in dynTypeForm have bad arType %s\n' % arType
         return ""

      try:
         return self.extTypeFunc[arType]( self, offset )
      except KeyError:
         print 'in dynTypeFrom have bad format %s\n' % arType
         return ""

   def aryFrom( self, offset, arType ):
      if arType < 0 or arType > ArrowTypes.ar_MaxType:
         print 'in aryFrom have bad arType %s\n' % arType
         return ""

      ( hdrOffset, totalLength ) = self.decodeVCT( offset )[ 1:3 ]
      if hdrOffset == 0:
         return []
      numElements = self.decodeACH( hdrOffset )[ 1 ]
      elementSize = ( totalLength - 2 ) / numElements
      dOff = hdrOffset + 2
      val = [ self.dynTypeFrom ( dOff + off, arType )
              for off in range ( 0, numElements * elementSize, elementSize ) ] 
      return val

   # an array with variable length elements ( such as
   # strings or IpGenAddr ) has the following structure:
   #
   # -----+-----+--   --+-----+-----+--   --+---
   #      |l1 |*|  ...  |l2 |*|l3 |*|  ...  |
   # -----+----|+--   --+----|+----|+--   --+---
   #           |         ^   |     |         ^
   #           |         |   |     +---------+----->
   #           +---------+   +---------------+
   #
   # where:
   #   the entry with l1 is the descriptor for the array.
   #      l1 is the length (number of entries) of the array
   #      the adjacent "*" is the offset to a vector l1 long
   #      of descriptors
   #   the entry with l2 is the first descriptor in the
   #      array's descriptor array.  l2 is the length of the
   #      first value and "*" is the offset from the start of the
   #      descriptor array to the value.
   #   similarly for l3.
   def varAryFrom( self, offset, arType ):
      if arType < 0 or arType > ArrowTypes.ar_MaxType:
         print 'in varAryFrom have bad arType %s\n' % arType
         return ""

      hdrOffset = self.decodeVCT( offset )[1]
      if hdrOffset == 0:
         return []
      arraySize = self.decodeACH( hdrOffset )[1]
      descOff = hdrOffset + 2
      val = [ self.extTypeVarFunc[arType]( self, off, hdrOffset )
              for off in range( descOff, ( arraySize * 4 ) + descOff, 4 ) ]
      return val

   extTypeFunc = {
         ArrowTypes.ar_uint8 : extU8,
         ArrowTypes.ar_uint16 : extU16,
         ArrowTypes.ar_uint32 : extU32,
         ArrowTypes.ar_uint64 : extU64,
         ArrowTypes.ar_int8 : extI8,
         ArrowTypes.ar_int16 : extI16,
         ArrowTypes.ar_int32 : extI32,
         ArrowTypes.ar_int64 : extI64,
         ArrowTypes.ar_float : extFloat,
         ArrowTypes.ar_double : extDouble,
         ArrowTypes.ar_bool : extBool,
         ArrowTypes.ar_string : extString,
         ArrowTypes.ar_EthAddr : extEthAddr,
         ArrowTypes.ar_IpAddr : extIpAddr,
         ArrowTypes.ar_Ip6Addr : extIp6Addr,
         ArrowTypes.ar_IpGenAddr : extIpGenAddr,
         ArrowTypes.ar_IpRouteKey : extIpRouteKey,
         ArrowTypes.ar_IpGenPrefix : extIpGenPrefix,
         ArrowTypes.ar_bits : extBits
         }

   extTypeVarFunc = {
         ArrowTypes.ar_string:varStrFrom,
         ArrowTypes.ar_IpGenAddr:varExtIpGenAddr,
         ArrowTypes.ar_IpRouteKey:varExtIpRouteKey,
         ArrowTypes.ar_IpGenPrefix:varExtIpGenPrefix
         }
   
   # subrows are variable length objects, stored in the format shown for 
   # varAryFrom, but cannot use varAryFrom beacuse in order to instantiate
   # the object, we need the class for the object.
   def subrowAryFrom( self, offset, cl ):
      hdrOffset = self.decodeVCT( offset )[1]
      if hdrOffset == 0:
         return []
      arraySize = self.decodeACH( hdrOffset )[1]
      descOff = hdrOffset + 2
      val = [ self.varExtSubrow( off, hdrOffset, cl)
              for off in range( descOff, ( arraySize * 4 ) + descOff, 4 ) ]
      return val

   # column insertors

   # extend the row with a variable part at the end
   def expandRow( self, obj ):
      self.buf_.extend(obj)

   def insertLen( self ):
      length = len( self )
      assert length <= MaxRowLength, "attempt to set row to length %d > %d" % \
                                       ( length, MaxRowLength )
      self.insertU16( 0, length )

   def insertType( self, t ):
      self.insertU16( 2, t )

   def insertU8( self, offset, v ):
      pack_into( "<B", self.buf_, offset, v )
   
   def insertU16( self, offset, v ):
      pack_into( "<H", self.buf_, offset, v )
   
   def insertU32( self, offset, v ):
      pack_into( "<I", self.buf_, offset, v )
   
   def insertU64( self, offset, v ):
      pack_into( "<Q", self.buf_, offset, v )

   def insertI8( self, offset, v ):
      pack_into( "<b", self.buf_, offset, v )
   
   def insertI16( self, offset, v ):
      pack_into( "<h", self.buf_, offset, v )
   
   def insertI32( self, offset, v ):
      pack_into( "<i", self.buf_, offset, v )
   
   def insertI64( self, offset, v ):
      pack_into( "<q", self.buf_, offset, v )

   def insertBool( self, offset, v ):
      pack_into( "<?", self.buf_, offset, v )

   def insertFloat( self, offset, v ):
      pack_into( "<f", self.buf_, offset, v )

   def insertDouble( self, offset, v ):
      pack_into( "<d", self.buf_, offset, v )

   def insertEthAddr( self, offset, addr ):
      for i in xrange( 3 ):
         self.insertU16( offset + ( 2 * i ), addr.word( i ) )

   def insertIpAddr( self, offset, addr ):
      v = addr.address()
      self.insertU32( offset, swap32( v & 0xffffffff ) )

   def insertIp6Addr( self, offset, addr ):
      v = addr.address()
      self.insertU32( offset, swap32( ( v >> 96 ) ) & 0xffffffff )
      self.insertU32( offset + 4, swap32( ( v >> 64 ) ) & 0xffffffff )
      self.insertU32( offset + 8, swap32( ( v >> 32 ) ) & 0xffffffff )
      self.insertU32( offset + 12, swap32( v ) & 0xffffffff )

   def updateVCT( self, offset, isArray, off, length ):
      assert length <= MaxColumnLength, "%sfield is too long for column" % \
                                ("array " if isArray else "") 
      ( fld1, fld2 ) = self.encodeVCT( isArray, off, length )
      self.insertU16( offset, fld1 )
      self.insertU16( offset + 2, fld2)

   # the following are all variable length insertors.  When inserting variable
   # length columns, the new data is always appended to the end of the row.  If
   # there was an existing value already in the buffer, that space is not reclaimed.
   # this is not a problem when creating and populating new rows.  However, if
   # an existing row is populated  with a new value the row will grow by the size
   # of the new value each time the row is updated.
   def insertIpGenAddr( self, offset, val ):
      rowEnd = len( self )
      vLen = len( val )
      self.updateVCT( offset, False, rowEnd, vLen)
      self.varInsertIpGenAddr( val )
      self.insertLen()

   #  does not generate VCT and appends to the end of the row
   def varInsertIpGenAddr( self, val, length=None ):
      tr = BaseRow( length = length + 2 if length else len( val ) +2 )
      if val.isIPv6():
         tr.insertIp6Addr( 2, val )
      else:
         tr.insertIpAddr( 2, val )
      self.expandRow( tr.buf_[2:] )

   def insertIpRouteKey( self, offset, val ):
      self.insertIpGenPrefix( offset, val )
            
   def insertIpGenPrefix( self, offset, val ):
      rowEnd = len( self )
      vLen = len( val )
      self.updateVCT( offset, False, rowEnd, vLen)
      self.varInsertIpGenPrefix( val )
      self.insertLen()

   #  does not generate VCT and appends to the end of the row
   def varInsertIpGenPrefix( self, val ):
      sr = BaseRow( length = 3 )  # length of an int 
      sr.insertU8( 2, val.mask() )
      sr.varInsertIpGenAddr( val, len( val ) - 1 )
      self.expandRow( sr.buf_[2:] )

   def insertString( self, offset, v ):
      rowEnd = len( self )
      # Include the null byte when storing our length
      self.updateVCT( offset, False, rowEnd, len( v ) + 1 )
      self.varInsertString( v )
      self.insertLen()

   # does not generate a VCT and appends to the end of the row
   def varInsertString( self, v ):
      # Store a null byte to terminate the string
      self.expandRow( array.array( 'B', v + '\x00' ) )
      
   def insertSubrow( self, offset, v ):
      rowEnd = len( self )
      self.updateVCT( offset, False, rowEnd, len( v ) )
      self.varInsertSubrow( v )
      self.insertLen()
   
   def varInsertSubrow( self, v ):
      self.expandRow( v.buf_ )
  

   # param v is a list of arType
   def insertFixedAry( self, offset, v, arType ):
      if arType < 1 or arType > ArrowTypes.ar_MaxType:
         print 'in insertFixedAry have bad arType %s\n' % arType
         return ""

      rowEnd = len( self )
      elementSize = self.typeLength[arType]
      arraySize = len( v )
      valLen = 2 + elementSize * arraySize
      self.updateVCT( offset, True, rowEnd, valLen )
      
      tr = BaseRow( length = valLen )
      tr.insertU16( 0, self.encodeACH( False, arraySize ) )
      for i in xrange( arraySize ):
         tr.insertTypeFunc[arType]( tr, 2 + elementSize * i, v[i] )
      self.expandRow( tr.buf_ )
      self.insertLen()

   def insertVarAry( self, offset, v, arType ):
      if arType < 0 or arType > ArrowTypes.ar_MaxType:
         print 'in insertVarAry have bad arType %s\n' % arType
         return ""

      rowEnd = len( self )
      arraySize = len( v )
      descLen = arraySize * 4
      totalLen = 2 + descLen

      tr = BaseRow( length = 2 + descLen )
      tr.insertU16( 0, self.encodeACH( True, arraySize ) )
      # generate the descriptor array and collect the element
      # lengths to reduce the required recomputation.
      ( db, vb ) = ( 2, 2+descLen )
      for i in xrange( arraySize ):
         eLen = ( len( v[i] ) )
         if arType == ArrowTypes.ar_string:
            # For strings, include a terminating null byte
            eLen += 1
         tr.updateVCT( db, False, vb, eLen )
         db += 4
         vb += eLen
         totalLen += eLen
      self.updateVCT( offset, True, rowEnd, totalLen )

      # insert the values
      for i in xrange( arraySize ):
         ( tr.insertTypeVarFunc[arType] )( tr, v[i] )

      self.expandRow( tr.buf_ )
      self.insertLen()

   def insertBits( self, offset, v ):
      assert False, "insertBits: Bitset columns are not supported in python"

   insertTypeFunc = {
         ArrowTypes.ar_uint8 : insertU8,
         ArrowTypes.ar_uint16 : insertU16,
         ArrowTypes.ar_uint32 : insertU32,
         ArrowTypes.ar_uint64 : insertU64,
         ArrowTypes.ar_int8 : insertI8,
         ArrowTypes.ar_int16 : insertI16,
         ArrowTypes.ar_int32 : insertI32,
         ArrowTypes.ar_int64 : insertI64,
         ArrowTypes.ar_float : insertFloat,
         ArrowTypes.ar_double : insertDouble,
         ArrowTypes.ar_bool : insertBool,
         ArrowTypes.ar_string : insertString,
         ArrowTypes.ar_EthAddr : insertEthAddr,
         ArrowTypes.ar_IpAddr : insertIpAddr,
         ArrowTypes.ar_Ip6Addr : insertIp6Addr,
         ArrowTypes.ar_IpGenAddr : insertIpGenAddr,
         ArrowTypes.ar_IpRouteKey : insertIpGenPrefix,
         ArrowTypes.ar_IpGenPrefix : insertIpGenPrefix,
         ArrowTypes.ar_bits : insertBits
         }

   insertTypeVarFunc = {
         ArrowTypes.ar_string:varInsertString,
         ArrowTypes.ar_struct:varInsertSubrow,
         ArrowTypes.ar_IpGenAddr:varInsertIpGenAddr,
         ArrowTypes.ar_IpRouteKey:varInsertIpGenPrefix,
         ArrowTypes.ar_IpGenPrefix:varInsertIpGenPrefix
         }

   def formatBool( self, v ):
      return 'true' if v else 'false'

   def formatBits( self, v ):
      assert False, "formatBits: Bitset columns are not supported in python"
   
   # fmt must either be a formatting string or a function that returns a string
   def arrayToString( self, vals, fmt ):
      strs = [ fmt % ( v ) if isinstance( fmt, basestring ) else fmt( v )  
               for v in vals ]
      return ", ".join( strs )

   def structArrayToString( self, vals ):
      strs = [ "{ " + v.toString() + " }" for v in vals ]
      return ", ".join( strs )
   
