#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import tempfile
import traceback

import AaaClientLib
from CliShellLib import NonTtyCliConnector
import Tac
import Tracing
import Xmpp

t0 = Tracing.trace0

# pylint: disable-msg=W1401
class CliChat( Xmpp.Conversation ):
   """A conversation between the Cli and an XMPP user.

   This class causes commands to be executed on a Cli session in
   response to messages received over XMPP.
   """
   def __init__( self, xmppClient ):
      Xmpp.Conversation.__init__( self )
      self.disableAaa_ = xmppClient.disableAaa()
      self.sysname_ = xmppClient.entityManager().sysname()
      self.aaaManager_ = AaaClientLib.AaaManager( self.sysname_,
                                                xmppClient.config_.initialPrivLevel )

   def sendMessage( self, env, privLevel, uid, gid, message ):
      cliConnector = NonTtyCliConnector()
      stdin = tempfile.TemporaryFile()
      stdouterr = tempfile.TemporaryFile()
      try:
         stdin.write( message )
         stdin.flush()
         stdin.seek( 0 )
         argv = [ '-p=%s' % privLevel, '-s=%s' % self.sysname_ ]
         if self.disableAaa_:
            argv.append( '-A' )
         signalSock = cliConnector.connectToBackend( self.sysname_, argv, env,
                                                     uid, gid,
                                                     stdin.fileno(),
                                                     stdouterr.fileno(),
                                                     stdouterr.fileno() )
         signalSock.recv( 1 )
         signalSock.close()
         stdouterr.seek( 0 )
         return stdouterr.read()
      finally:
         stdin.close()
         stdouterr.close()

   def handlePresence( self, jid, presence ):
      # Presence handling is all done in //src/Xmpp/Xmpp.py
      pass

   def handleMessage( self, jid, message ):
      """Handles an XMPP message stanza "message" from XMPP JID "jid"."""
      result = ""
      aaaResults = self.aaaManager_.authorizeUser( jid.local.encode( "utf8" ), 
                                                   jid.domain.encode( "utf8" ),
                                                   'xmpp', 'xmpp', 'XMPP',
                                                   self.disableAaa_ )
      if not aaaResults[ "authorized" ]:
         return "%% %s" % AaaClientLib.AUTHZ_FAILED

      env = { "REALTTY": "XMPP", 
              "LOGNAME": str( jid.local.encode( "utf8" ) ),
              "SSH_CONNECTION": str( jid.domain.encode( "utf8" ) ) }
      if aaaResults[ "aaaSessionId" ] != "":
         env[ "AAA_AUTHN_ID" ] = aaaResults[ "aaaSessionId" ]
      try:
         result = self.sendMessage( env, aaaResults[ "privLevel" ], 
                                    aaaResults[ "uid" ], aaaResults[ "gid" ],
                                    message )
      except Exception as e: # pylint: disable-msg=W0703
         t0( "processRequest Exception", e )
         traceback.print_exc()  # Log stack trace to agent log.
         result = str( e )

      self.aaaManager_.closeSession( aaaResults[ "aaaSessionId" ] )
      return result

   def help( self ):
      return [ "I can run any enable mode CLI command" ]

def Plugin( ctx ):
   ctx.registerConversation( "cli", CliChat( ctx ) )
