#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing


traceHandle = Tracing.Handle( "XmppModelNameReactor" )
t0 = traceHandle.trace0
t9 = traceHandle.trace9


# This prefix in XMPP status messages is used to identify neighbor switches
# and is thus used as its status message prefix by switches. Unidentified
# switches us the suffix "Switch". Hardware models that report
# their version info before Xmpp connects, will show their model
# number as the suffix instead.
PRESENCE_PREFIX = "Arista Networks"


class ModelNameReactor( Tac.Notifiee ):
   """A reactor which responds to changes to the device model name."""

   notifierTypeName = "EntityMib::Fru"

   def __init__( self, root, configReactor ):
      t9( "Initializing", self.__class__.__name__ )
      Tac.Notifiee.__init__( self, root )
      self.configReactor_ = configReactor
      self.handleModelName()

   @Tac.handler( "modelName" )
   def handleModelName( self ):
      t9( "modelName:", repr( self.notifier_.modelName ) )
      if self.notifier_.modelName:
         t0( "Switch model name from EntityMib:", self.notifier_.modelName )
         self.configReactor_.status_.presenceStatus = "%s %s" % (
            PRESENCE_PREFIX, self.notifier_.modelName )
         # Reacted to by Xmpp.XmppStatusReactor to send a new presence
         self.configReactor_.status_.presenceLastUpdateTime = Tac.now()


class EntityMibRootReactor( Tac.Notifiee  ):
   """Entity Mib root reactor, used to manage the ModelNameReactor."""

   notifierTypeName = "EntityMib::Status"

   def __init__( self, status, configReactor ):
      t9( "Initializing", self.__class__.__name__ )
      Tac.Notifiee.__init__( self, status )
      self.modelNameReactor_ = None
      self.configReactor_ = configReactor
      self.handleRoot()

   @Tac.handler( "root" )
   def handleRoot( self ):
      t0( "Handling EntityMib root:", self.notifier_.root )
      assert not self.modelNameReactor_, "Should only be called once"
      if self.notifier_.root:
         self.modelNameReactor_ = ModelNameReactor( self.notifier_.root,
                                                    self.configReactor_ )
