#!/usr/bin/env python
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Logging

# Arista Logging messages emitted by the Xmpp agent
# Before each log line definition, define the result attribute to keep pylint happy

# The XMPP client fails to authenticate with an XMPP server (bug 36226)
XMPP_CLIENT_AUTHEN_FAIL = None
Logging.logD(
    id="XMPP_CLIENT_AUTHEN_FAIL",
    severity=Logging.logError,
    format="Authentication failure connecting to %s:%s",
    explanation=(
        "The authentication credentials provided are rejected "
        "by the configured XMPP server." ),
    recommendedAction=(
        "Confirm the authentication credentials configured "
        "for the XMPP server are correct by checking with your "
        "server administrator." ),
    minInterval=30.0,
    rateLimitArgs=2 )


# XMPP server host resolution failure
XMPP_DNS_LOOKUP_FAILURE = None
Logging.logD(
    id="XMPP_DNS_LOOKUP_FAILURE",
    severity=Logging.logError,
    format="DNS lookup for XMPP server %s failed: %s",
    explanation=(
        "There was a problem resolving the XMPP server host name using the DNS. "),
    recommendedAction=(
        "Confirm the XMPP server hostname exists in the DNS, that your "
        "switch has DNS name servers configured, and that they are reachable." ),
    minInterval=30.0,
    rateLimitArgs=1 )


# XMPP client successful connect
XMPP_CLIENT_CONNECTED = None
Logging.logD(
    id="XMPP_CLIENT_CONNECTED",
    severity=Logging.logInfo,
    format="Connected to %s:%s with JID %s",
    explanation=(
        "The XMPP client is connected to the named server using the specified "
        "Jabber ID (JID)" ),
    recommendedAction=Logging.NO_ACTION_REQUIRED,
    minInterval=1.0,
    rateLimitArgs=1 )


# XMPP client disconnect notification; this is for any disconnection, whether
# locally initiated, a socket error or remotely initiated.
XMPP_CLIENT_DISCONNECTED = None
Logging.logD(
    id="XMPP_CLIENT_DISCONNECTED",
    severity=Logging.logInfo,
    format="Connection to %s:%s closed",
    explanation="The connection between XMPP client and server closed",
    recommendedAction=Logging.NO_ACTION_REQUIRED,
    minInterval=1.0,
    rateLimitArgs=1 )


# XMPP client could not join a switch-group due to an authorization
# error from the XMPP server. Usually this means the password is
# incoreect.
XMPP_GROUP_NOT_AUTHORIZED = None
Logging.logD(
   id="XMPP_GROUP_NOT_AUTHORIZED",
   severity=Logging.logError,
   format="Incorrect password for switch group %s",
   explanation=(
      "The XMPP server did not authorize the XMPP client to join the switch group. "
      "Configure the correct password on the switch group for the group named "
      "in the error" ),
   recommendedAction=(
      "Check with the XMPP multi-user chat room (switch group) owner what the "
      "password should be, and configure that under 'management xmpp' on the "
      "appropriate 'switch-group' line" ),
    minInterval=30.0,
    rateLimitArgs=1 )


# XMPP client could not join a switch-group due because they are banned.
XMPP_GROUP_REGISTRATION_REQUIRED = None
Logging.logD(
   id="XMPP_GROUP_REGISTRATION_REQUIRED",
   severity=Logging.logError,
   format="Members-only switch-group %s requires pre-registration",
   explanation=(
      "The switch group is configured members-only, and this switch is not "
      "on the member list. The owner must add the switch to the member list." ),
   recommendedAction=(
      "Contact the XMPP multi-user chat room (switch group) owner and ask "
      "to have this switch added to the room member list" ),
    minInterval=30.0,
    rateLimitArgs=1 )


# XMPP client could not join a switch-group due because they are banned.
XMPP_GROUP_FORBIDDEN = None
Logging.logD(
   id="XMPP_GROUP_FORBIDDEN",
   severity=Logging.logError,
   format="Forbidden from joining switch group %s by server %s:%s",
   explanation=(
      "The XMPP server has explicitly banned this switch from joining "
      "the switch group" ),
   recommendedAction=(
      "Check with the XMPP multi-user chat room (switch group) owner as to "
      "why they have banned this switch." ),
    minInterval=30.0,
    rateLimitArgs=1 )
