#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import collections
import functools
import re
import socket
import ssl
import sys
import traceback
import uuid
import weakref

import ReversibleSecretCli
import Agent
import Aresolve
import Arnet.NsLib
import Cell
import IpUtils
import IpLibConsts
import Logging
import Plugins
import Tac
import Tracing
import TraceLogging

import XmppLogMsgs
import XmppModelNameReactor


# Send Python's logging module output to Trace logs
# level=1 is all possible levels of logs.
TraceLogging.setup( "Xmpp", level=1 )
# Import sleekxmpp after TraceLogging is setup
import sleekxmpp
import sleekxmpp.xmlstream
import sleekxmpp.plugins.xep_0004

traceHandle = Tracing.Handle( "Xmpp" )
t0 = traceHandle.trace0
t8 = traceHandle.trace8
t9 = traceHandle.trace9  # a potentially annoying volume of logs

# Obtain the ConnectionState enums from TACC (Xmpp.tac)
ConnectionState = Tac.Type( "Mgmt::Xmpp::Status::ConnectionState" )
# Obtain the VRF state enum (Ip.tac)
VrfState = Tac.Type( "Ip::VrfState" )
# Obtain the NeighborState
NeighborState = Tac.Type( "Mgmt::Xmpp::NeighborStatus::State" )
ipAddrZero = Tac.Value( "Arnet::IpAddr" ).ipAddrZero

# Tuple of VRF names in configuration that can correspond
# to the default VRF
DEFAULT_VRF_NAMES = ( "", IpLibConsts.DEFAULT_VRF )

# The limit on how many lines could be sent per
# message
LINE_LIMIT = 250

# Number of seconds to wait before timing out on XMPP server connect
CONNECT_TIMEOUT = 30

# Maximum number of seconds between connection retries
RECONNECT_MAX_TIMEOUT = 60

# Number of seconds between reconnection attempts
RECONNECT_DELAY_SECS = 3

# Period between DNS queries made by Aresolve for existing queries
DNS_RETRY_PERIOD = 60

# Attributes to ignore when dumping an entity
IGNORE_ATTRS = frozenset( ( "isNondestructing", "parent", "parentAttrName", "entity",
                            "fullName" ) )



def fibonacci( maxval ):
   """A fibonacci series generator used by the reconnection activity.

   This sequence never ends. After the sequence's current value is
   above maxval, maxval will be returned for each next() call.
   """
   x, y = 1, 1
   while True:
      yield x
      x, y = x + y, x
      if x > maxval:
         raise StopIteration

def checkXmppUserOrGroup( to ):
   assert xmppAgent is not None, "Not pycliented to the Xmpp agent?"
   xmppClient = xmppAgent.xmppClient()
   if not xmppClient:
      return "XMPP client not configured"

   # Basic sanity checks. Note that these are also validated as
   # part of running 'xmpp session <user-or-name>'
   # 1 - are we connected to our xmpp server
   if not xmppClient.connected():
      return "XMPP connection to server not established"

   # 2 - are we connected to this user / group
   if '@conference' in to:
      if not xmppClient.status_.group.get( to ):
         return "Not connected to group %s" % to
   else:
      xmppNeighbor = xmppClient.status_.neighbor.get( to )
      if not xmppNeighbor or xmppNeighbor.state != "present":
         return "Not connected to user %s" % to 

   return ""

def sendXmppMessage( to, message, group=False, captureResponse=True ):
   """Sends a message to a neighbor or group of neighbors over XMPP.

   If captureResponse is True, the XMPP client will wait for a response
   from a remote Arista EOS Xmpp agent (i.e., when executing a command
   from a remote client). If False, the method will return immediately
   after the message is delivered to the XMPP server.

   This is generally called by processes pyclient'ing into the Xmpp
   agent, so has to refer to variables at module scope.
   """
   assert xmppAgent is not None, "Not pycliented to the Xmpp agent?"

   xmppClient = xmppAgent.xmppClient()
   if not xmppClient:
      return "XMPP client not configured"

   if not xmppClient.connected():
      return "XMPP connection to server not established"

   mtype = 'groupchat' if group else 'chat'
   return xmppClient.sendMessage( to, message, mtype=mtype,
                                  captureResponse=captureResponse,
                                  group=group )

def getXmppReply( from_, id_ ):
   """ Return the responses we have regardless of them being
   complete or not and stop awaiting messages"""

   assert xmppAgent is not None, "Not pycliented to the Xmpp agent?"

   xmppClient = xmppAgent.xmppClient()
   if not xmppClient:
      return "XMPP client not configured"

   return xmppClient.getMessageResponse( from_, id_ )

def getXmppMessageStatus( from_, id_ ):
   """ Returns a dictionary the status of users which
   can be one of these values.

   '' - We have yet to recieve any acknowlegdeent of
   the message delivery

   'composing' - The message has been deliveried and is currently
   being processed.

   'active' - The message is complete
   """

   assert xmppAgent is not None, "Not pycliented to the Xmpp agent?"

   xmppClient = xmppAgent.xmppClient()
   if not xmppClient:
      return "XMPP client not configured"
   key = ( from_, id_ )
   return xmppClient.awaitingResponseStatus_.get( key, {} )

def stopAwaitingXmppReply( from_, id_ ):
   """ No longer store messages corresponding to the given
   arguments. """

   assert xmppAgent is not None, "Not pycliented to the Xmpp agent?"

   xmppClient = xmppAgent.xmppClient()
   if not xmppClient:
      return "XMPP client not configured"

   xmppClient.stopAwaitingResponse( from_, id_ )


class Conversation( object ):
   """ Base class for defining Xmpp Chat conversations.
   Derived classes must implement handlePresence,
   handleMessage, and help. See below for a description
   of what each of these functions does.
   """

   def handlePresence( self, username, presence ):
      """ Called when the presence of a user changes """
      raise NotImplementedError

   def handleMessage( self, username, message ):
      """ Called when a chat message is received by a user,
      returning a response or None if the message is not
      understood. """
      raise NotImplementedError

   def help( self ):
      """ Returns a list of the types of things this
      conversation knows about """
      raise NotImplementedError


class XmppClient( sleekxmpp.ClientXMPP ):
   """The CloudVision XMPP client."""

   def __init__( self, jid, password, hosts, config, status,
                 entityManager, disableAaa, configReactor ):

      # If we have an obscured password, unobscure it for SleekXMPP
      try:
         password = ReversibleSecretCli.decodeKey( password )
      except TypeError: # password is in (non-obscured) plain text
         pass
      super( XmppClient, self ).__init__( jid, password )
      # Supply an initial None for the SSL certificate, avoids a
      # a mostly harmless AttributeError.
      self._der_cert = None

      # We drive reconnection attempts from a TACC timed
      # notification. For that reason, we strictly disable the
      # SleekXMPP client doing its own reconnection during an initial
      # connection error, to better manage synchronous I/O for this
      # crazy TACC world.
      self.auto_reconnect = False

      self.register_plugin( "xep_0030" ) # Service Discovery
      self.register_plugin( "xep_0045" ) # Multi-User Chat
      self.muc_ = self.plugin[ "xep_0045" ]
      self.register_plugin( "xep_0060" ) # Pubsub
      self.register_plugin( "xep_0085" ) # Chat states
      self.register_plugin( "xep_0199" ) # XMPP Ping

      # Host is set by the XmppConfigReactor prior to calling connect()
      self.host_ = None
      self.port_ = config.port
      self.vrf_ = None  # A VrfStatusLocal object
      self.connected_ = False
      self.hosts_ = hosts[ : ]
      self.nextXmppServers_ = collections.deque( hosts )
      self.config_ = config
      self.status_ = status
      self.disableAaa_ = disableAaa
      self.configReactor_ = configReactor
      self.allVrfStatusLocal_ = configReactor.allVrfStatusLocal_
      self.ipStatus_ = configReactor.ipStatus_
      self.srcIntf_ = self.config_.srcIntfName
      self.srcIpAddr_ = None
      if self.srcIntf_:
         self.srcIpIntfStatus_ = self.ipStatus_.ipIntfStatus.get( self.srcIntf_ )
         self.srcIpAddr_ = self.srcIpIntfStatus_.activeAddrWithMask.address \
                           if self.srcIpIntfStatus_ else None
         self.srcIntfVrf_ = self.srcIpIntfStatus_.vrf \
                            if self.srcIpIntfStatus_ else None

      # Initialize source interface in status with default value
      self.status_.srcIntfName = Tac.Value( "Arnet::IntfId" )
      self.status_.srcIpAddr = Tac.Value( "Arnet::IpAddr" )

      # Dictionary of:
      #   to, id ->
      #         Dictionary of:
      #            username -> response
      # if a ( username, id ) is in this dictionary
      # this client will be waiting for a response
      # while waiting for a response this pair is
      # mapped to an dictionary of incomplete response(s)
      self.messagesAwaitingResponse_ = {}

      # Dictionary of:
      #   to, id ->
      #           Dictionary of:
      #              username -> status
      # determines the status of a response
      # values are either '', 'composing' or 'active'
      self.awaitingResponseStatus_ = {}

      self.fibonacci_ = fibonacci( RECONNECT_MAX_TIMEOUT )
      self.activity_ = Tac.ClockNotifiee()
      self.activity_.handler = self.handleTimeout
      self.activity_.timeMin = Tac.endOfTime

      self.conversation_ = {}
      self.entityManager_ = entityManager

      # Create a weak reference to ourselves and load the XMPP plugins,
      # giving each plugin that reference to access our attributes.
      t0( "Loading XMPP plugins" )
      proxy = weakref.proxy( self )
      Plugins.loadPlugins( "XmppPlugin", context=proxy )

      self.add_event_handler( "session_start",
                              Tac.WeakBoundMethod( self.start ), threaded=True )
      self.add_event_handler( "roster_received",
                              Tac.WeakBoundMethod( self.handleRosterReceived ) )
      self.add_event_handler( "connected",
                              Tac.WeakBoundMethod( self.handleConnected ) )
      self.add_event_handler( "disconnected",
                              Tac.WeakBoundMethod( self.handleDisconnected ) )
      self.add_event_handler( "failed_auth",
                              Tac.WeakBoundMethod( self.handleFailedAuth ) )
      self.add_event_handler( "no_auth",
                              Tac.WeakBoundMethod( self.handleNoAuth ) )
      self.add_event_handler( "socket_error",
                              Tac.WeakBoundMethod( self.handleSocketError ) )
      self.add_event_handler( "changed_status",
                              Tac.WeakBoundMethod(
                                 self.handleNeighborStatusChange ) )

      self.add_event_handler( "message", Tac.WeakBoundMethod( self.handleMessage ) )
      self.add_event_handler( "groupchat_message",
                              Tac.WeakBoundMethod( self.handleGroupChatMessage ) )
      self.add_event_handler( "groupchat_presence",
                              Tac.WeakBoundMethod( self.handleGroupChatPresence ) )
      self.add_event_handler( "groupchat_invite",
                              Tac.WeakBoundMethod( self.handleGroupChatInvite ) )
      self.add_event_handler( "chatstate_composing", lambda message :
                              proxy.handleChatState( message, "composing" ) )
      self.add_event_handler( "chatstate_active", lambda message :
                              proxy.handleChatState( message, "active" ) )
      self.add_event_handler( "ssl_invalid_cert", proxy.handleInvalidCert )

      # Add SASL <mechanisms> handler with lowest order to check socket
      # is suitable for use. i.e., immediately before authentication.
      assert self.override_feature( "mechanisms", self._preAuthHook ), \
             "Unable to find 'mechanisms' to override"

   def connected( self ):
      return self.connected_

   def entityManager( self ):
      return self.entityManager_

   def disableAaa( self ):
      return self.disableAaa_

   def registerConversation( self, name, conversation ):
      assert name not in self.conversation_
      self.conversation_[ name ] = conversation

   def handleInvalidCert( self, pem_cert, **unused_kwargs ):
      """Deals with invalid certificate events.

      For now, ignore them, as we cannot install CAs.
      """
      t0 ( "handleInvalidCert" )

   def configure_socket( self ):
      """Sets timeout and VRF for self.socket."""
      # If needed, configure the appropriate VRF for this connection.
      if self.config_.vrfName not in DEFAULT_VRF_NAMES:
         vrf = self._vrf()
         if vrf is not None:
            t8( "Using VRF", repr( vrf.vrfName ), "for next connection" )
            self.socket_class = functools.partial( Arnet.NsLib.socketAt,
                                                   ns=vrf.networkNamespace )
         else:
            t8( "No VRF named", repr( self.config_.vrfName ) )
      t8( "Setting Xmpp connect timeout to", CONNECT_TIMEOUT, "seconds." )
      self.socket.settimeout( CONNECT_TIMEOUT )

      # Bind socket to ip address of configured source interface
      if self.srcIpAddr_:
         try:
            self.socket.bind( ( self.srcIpAddr_, 0 ) )
            t8( "Using ip address %s of source interface %s "
                % ( self.srcIpAddr_, self.srcIntf_ ) )
            self.status_.srcIntfName = self.srcIntf_
            self.status_.srcIpAddr = self.srcIpAddr_
         except socket.error as err:
            t8( "Socket bind to ip address %s of source interface %s "
                "failed with errno:%s" % ( self.srcIpAddr_, self.srcIntf_,
                err.errno ) )

   def _emptyQueues( self ):
      count = 0
      for queue in ( self.event_queue, self.send_queue ):
         while not queue.empty():
            queue.get( True )
            count += 1
      t9( "Cleared", count, "items from {event,send}_queue" )

   def handleTimeout( self ):
      t0( "Handling reconnection timer" )
      assert not self.connected_, "timeout retry not a valid state while connected"
      # First make sure that all threads are stopped by this point
      assert self.stop.isSet(), "everything should be stopped"
      # Empty out the queues of any leftover actions
      self._emptyQueues()
      # Retry the connection
      self.connect()

   def _getNextServer( self ):
      """Returns the next server IP address as a str. IPv6 addresses come first."""
      def nextServer():
         try:
            return self.nextXmppServers_.popleft()
         except IndexError:
            self._refreshXmppServers()
            return None

      for _ in xrange( 2 ):
         server = nextServer()
         if server is not None:
            return server
      t0( "Could not repopulate server list. Mo servers available." )

   def _refreshXmppServers( self ):
      t8( "Refreshing next XMPP server queue" )
      self.nextXmppServers_.extend( self.hosts_ or [] )

   def _vrf( self ):
      return self.allVrfStatusLocal_.vrf.get( self.config_.vrfName )

   def _preAuthHook( self, features, originalFunc ):
      # If we're using STARTTLS and we don't allow unencrypted sessions,
      # confirm the socket is an SSL socket.
      t0( "Pre-SASL socket check hook" )
      if self.use_tls and not self.config_.starttlsPermitUnencrypted:
         if type( self.socket ) != ssl.SSLSocket:
            t0( "Pre-SASL socket check FAILED" )
            # We didn't get the encrypted socket we wished, disconnect
            self.disconnect( reconnect=False, send_close=False )
            self.status_.connectionState = ConnectionState.encryptionFailed
            return False
      t0( "Pre-SASL socket check OK" )
      return originalFunc( features )
   
   def connect( self, address=None, reattempt=True,
                use_tls=True, use_ssl=False):
      _ = reattempt, use_tls, use_ssl
      assert not self.auto_reconnect

      def vrfIsUsable( vrf ):
         return bool( vrf and vrf.state == VrfState.active )

      if ( self.config_.vrfName not in DEFAULT_VRF_NAMES and
           not vrfIsUsable( self._vrf() ) ):
         # We've configured a non-default VRF, but it's not usable
         t8( "VRF", repr( self.config_.vrfName ), "is unusable; connection halted" )
         self.shutdown()
         return

      self.host_ = self._getNextServer()
      if not self.host_:
         t8( "No hosts. Halting connection." )
         self.shutdown()
         return

      if self.srcIntf_:
         # Source interface does not exist or ip address not assigned to interface,
         # halt connection establishment
         if self.srcIpIntfStatus_ is None:
            t8( "Source Interface %s does not exist or ip address not assigned to "
                "source interface, halting connection" % self.srcIntf_ )
            self.shutdown()
            return

         # If souce interface has no ip address or zero ip address, halt connection
         # establishment and wait until interface gets valid ip address
         if self.srcIpAddr_ is None  or ( self.srcIpAddr_ == ipAddrZero ):
            t8( "Source Interface has no ip address: %s, halting connection"
                % self.srcIpAddr_ )
            self.shutdown()
            return

         # If source interface does not belong to configured vrf, halt connection
         # establishment and wait until interface is moved to the right vrf
         if self.config_.vrfName != self.srcIntfVrf_:
            t8( "Source Interface %s belongs to wrong vrf, halting connection. "
                "Configured VRF: %s, Interface VRF:%s." % ( self.srcIntf_,
                self.config_.vrfName, self.srcIntfVrf_ ) )
            self.shutdown()
            return

      t0( "Using XMPP server:", self.host_, "port:", self.port_ )
      # Prevent SleekXMPP doing AAAA/A queries on the IP address string
      self.default_domain = None
      connected = sleekxmpp.ClientXMPP.connect(
         self, address=( self.host_, self.port_ ), use_tls=True, reattempt=False )
      if not connected:
         # self.stop is not set as part of this path. We set
         # it manually here to simplify the expectations in
         # our handleTimeout method.
         self.stop.set()
         # Connection failed, retry shortly.
         self.scheduleConnectionRetry()
      else:
         # Process incoming events (start a read_thread)
         self.process( block=False )

   def scheduleConnectionRetry( self ):
      try:
         # No really, a generator does have a 'next()' method, pylint.
         # pylint:disable-msg=E1101
         timeout = self.fibonacci_.next()
      except StopIteration:
         timeout = RECONNECT_MAX_TIMEOUT
      t0( "Next connect retry in", timeout, "seconds" )
      self.activity_.timeMin = Tac.now() + timeout

   def start( self, unused_event ):
      """XMPP session start event handler."""
      t0( "XMPP session start" )
      self.status_.group.clear()
      self.status_.neighbor.clear()

      if self.bindfail:
         t0( "bindfail == True" )
         self.disconnect()
         return

      try:
         t0( "Getting XMPP roster" )
         self.get_roster()
      except ( sleekxmpp.exceptions.IqError,
               sleekxmpp.exceptions.IqTimeout ), e:
         t0( "Failed to get roster:", e )
         self.disconnect()
         return

      try:
         t0( "Setting XMPP status message:", repr( self.status_.presenceStatus ) )
         self.send_presence( pstatus=self.status_.presenceStatus )
      except ( sleekxmpp.exceptions.IqError,
               sleekxmpp.exceptions.IqTimeout ), e:
         t0( "Failed to set presence:", e )
         self.disconnect()
         return

      # We only set ourselves to connected after session startup is done
      self.joinChatRooms()
      if not self.status_.connectionState == ConnectionState.connected:
         Logging.log( XmppLogMsgs.XMPP_CLIENT_CONNECTED,
                      self.host_, self.port_, self.boundjid )
      self.status_.connectionState = ConnectionState.connected
      t0( "XMPP session start completed for", self.boundjid )

   def handleRosterReceived( self, unused_stanza ):
      """Synchronize the received XMPP roster with our neighbor list."""
      t0( "handleRosterReceived:", unused_stanza )
      for jid in self.client_roster.keys():
         t9( "jid:", jid, "conference?", "@conference" in jid )
         if "@conference" not in jid:
            self._newNeighbor( jid )

   def shutdown( self ):
      """Shuts down the XMPP client.

      This closes all threads and stops our TACC activity threads.
      """
      t0( "Shutting down XMPP client" )
      # Disconnect and then disable the reconnection timer activity
      self.disconnect( reconnect=False, wait=False, send_close=False )
      self.activity_.timeMin = Tac.endOfTime
      t8( "Finished shutting down XMPP client" )

   def sendMessage( self, mto, mbody, msubject=None, mtype=None, mhtml=None,
                    mfrom=None, mnick=None, captureResponse=False, group=False ):
      id_ = None
      if captureResponse:
         id_ = "arista-" + self.boundjid.user + "-" + self.new_id()
         t8( "sendMessage S2S id:", id_ )
         self.startAwaitingResponse( mto, id_, mbody, group )
      self.makeMessage(
         mto, mbody, mtype=mtype, mfrom=mfrom, mid=id_ ).send()
      if id_ is not None:
         # Convert the unicode ID value (which contains no non-ASCII
         # characters) to an ASCII string of the same number of
         # characters (as runes), so that we minimize any opportunity for
         # duplicate IDs.
         return id_.encode( "ascii", "replace" )

   def makeMessage( self, mto, mbody=None, msubject=None, mtype=None,
                    mhtml=None, mfrom=None, mnick=None, mid=None ):
      message = sleekxmpp.ClientXMPP.makeMessage( self, mto, mbody, mtype=mtype )
      if mid:
         message[ 'id' ] = mid
      return message

   def handleFailedAuth( self, unused_event ):
      """Handles XMPP client authentication failures.

      Connection failed due to non-existent username or bad password.
      """
      t0( "handleFailedAuth" )
      Logging.log( XmppLogMsgs.XMPP_CLIENT_AUTHEN_FAIL, self.host_, self.port_ )
      self.status_.connectionState = ConnectionState.authenticationFailed

   def handleNoAuth( self, unused_event ):
      """Handles when the client and server have no common authentication method.

      Does not set connectionState, as handleFailedAuth will also be called.
      """
      t0( "XMPP client and server cannot find a common authentication method" )

   def handleSocketError( self, serr ):
      t0( "SocketError", serr )
      # Nothing to do here - we'll go to the disconnected state and retry

   def handleDisconnected( self, unused_event ):
      t0( "Disconnecting from XMPP server", self.host_, "port:", self.port_ )

      # Only log a disconnect message if we're currently connected.
      if self.status_.connectionState == ConnectionState.connected:
         Logging.log( XmppLogMsgs.XMPP_CLIENT_DISCONNECTED,
                      self.host_, self.port_ )
      self.status_.connectionState = ConnectionState.notConnected
      self.connected_ = False
      # Clear group/neighbor datastructures as we're now disconnected.
      self.status_.group.clear()
      self.status_.neighbor.clear()
      # Retry the connection later.
      self.scheduleConnectionRetry()
         
   def handleConnected( self, unused_event ):
      isEncrypted = type( self.socket ) == ssl.SSLSocket
      suffix = "[SSL socket]" if isEncrypted else ""
      t0( "TCP connection made to XMPP server", self.host_,
          "port:", self.port_, suffix )
         
      self.status_.group.clear()
      self.status_.neighbor.clear()
      self.activity_.timeMin = Tac.endOfTime # reset timeout handler (if any)
      self.connected_ = True
      self.configReactor_.connectCount_ += 1
      # Remove the socket timeout to remove signal handlers to that
      # Aaa RPC calls complete without interrupted system calls.
      self.socket.settimeout( None )
      # Reset the fibonacci sequence to reset the retry timeout
      self.fibonacci_ = fibonacci( RECONNECT_MAX_TIMEOUT )

   def joinChatRooms( self ):
      """Joins all the multi user chatrooms (groups) in the config."""
      for room in self.config_.group:
         t0( "Joining MUC room:", room )
         self.joinGroupChatRoom( room.decode( "utf8" ) )

   def _updateNeighborStatus( self, neighbor, resource=None, remove=None ):
      """Handles a neighbors' status change."""
      # Ignore chatroom JIDs as neighbors
      utf8Neighbor = neighbor.encode( "utf8" )
      if ( utf8Neighbor in self.status_.group or
           utf8Neighbor in self.config_.group ):
         t9( "Ignoring chatroom:", utf8Neighbor )
         return

      # Remove clients unknown to the XMPP roster from sysdb, as well as
      # those who have unsubscribed to us.
      if remove is None:
         if not self.client_roster.has_jid( neighbor ):
            remove = True
         else:
            remove = bool(
               self.client_roster[ neighbor ][ "subscription" ] == "remove" )

      if remove:
         t8( "Removing", utf8Neighbor, "from neighbor list." )
         for c in self.conversation_.values():
            # BUG 85343: The CliChat plugin needs refactoring to clean these up.
            if hasattr( c, "_delSession" ):
               # Remove the AAA session
               c._delSession( neighbor )  # pylint: disable-msg=W0212
         del self.status_.neighbor[ utf8Neighbor ]

      elif resource is not None:
         # Obtain the neighbor status or create a new neighbor status entry.
         neighborStatus = self._newNeighbor( neighbor )
         # Change the user's status based on their resource
         # If the 'show' field is not set, the neighbor is available.
         # Any other value (e.g., 'dnd', 'away', 'xa') implies they're away.
         if not resource[ "show" ]:
            neighborStatus.state = NeighborState.present
         else:
            neighborStatus.state = NeighborState.notPresent
         neighborStatus.lastUpdateTime = Tac.now()
         status = resource[ "status" ]
         neighborStatus.presenceStatus = status.encode( "utf8" )

   def handleNeighborStatusChange( self, presence ):
      """Handles a neighbors' status change."""
      neighbor = presence[ "from" ].bare
      neighborUtf8 = neighbor.encode( "utf8" )
      statusFromGroup = bool( neighborUtf8 in self.config_.group )
      if statusFromGroup:
         # In a group chat, the neighbor JID is in the resource
         statusFromUs = bool( presence[ "from" ].resource.encode( "utf8" )
                              == self.config_.username )
      else:
         statusFromUs = bool( neighborUtf8 == self.config_.username )

      t8( "Presence from:", presence[ "from" ],
          "fromGroup:", statusFromGroup, "fromUs:", statusFromUs )
      if ( statusFromGroup or statusFromUs ):
         # Skip our status; groups are handled in handleGroupChatPresence
         return

      # Have plugins respond to the status change
      for c in self.conversation_.values():
         c.handlePresence( presence[ "from" ], presence )

      if presence[ "type" ] == "unavailable":
         self._updateNeighborStatus( neighbor, remove=True )
      else:
         # Get the neighbor status info, update it in Sysdb and run plugins
         neighborData = self.client_roster[ neighbor ]
         resource = neighborData.resources.get( presence[ "from" ].resource )
         if resource is not None:
            self._updateNeighborStatus( neighbor, resource=resource )

   def getResponse( self, message, username=None ):
      response = None
      errors = []
      username = username or message[ "from" ].bare
      body = message[ "body" ]
      length = len( body )
      t9( "getResponse command length:", length, "bytes" )

      # Avoid being DoS'd by copying massive request twice (strip, lower).
      if length < 100 and body.strip().lower() == "help":
         helpMsgs = []
         for c in self.conversation_.values():
            try:
               helpMsgs += c.help()
            except NotImplementedError:
               pass
         respData = [ "Here are some of the things I can help you with:" ]
         respData += [ "   - %s\n" % msg for msg in sorted( helpMsgs ) ]
         response = "\n".join( respData )
      else:
         for c in self.conversation_.values():
            try:
               response = c.handleMessage( message[ "from" ], body )
            except ( KeyboardInterrupt, SystemExit ):
               raise
            except Exception:
               # Error occurred. Print and continue
               sys.excepthook( *sys.exc_info() )  # pylint: disable-msg=W0702
               errors.append( "".join(
                  traceback.format_exception( *sys.exc_info() ) ) )
               continue
            if response is not None:
               break
         if response is None:
            # Message was not understood by any registered conversation
            if errors:
               response = "Here are the are the error(s) that have occurred.\n\n"
               response += "\n".join( errors )
            else:
               response = ( "Sorry, I didn't understand what you wrote. "
                            "Type 'help' for a list of things you can ask me." )
      assert response is not None
      return response

   def handleChatState( self, message, chatstate ):
      t0( "handleChatState", message, chatstate )
      id_ = message[ "id" ]
      if message[ "type" ] == "groupchat":
         username, room = message[ "mucnick" ], message[ "mucroom" ]
         if ( room, id_ ) in self.awaitingResponseStatus_:
            self.awaitingResponseStatus_[ ( room, id_ ) ][ username ] = chatstate
      else:
         username = message[ "from" ].bare
         if ( username, id_ ) in self.awaitingResponseStatus_:
            self.awaitingResponseStatus_[ ( username, id_ ) ][ username ] = chatstate

   def handleMessage( self, message ):
      t0( "Handling message:", message[ "from" ], "->", message[ "to" ] )
      mtype = message.get_type()
      if mtype == "groupchat":
         # groupchat messages are received by both handleMessage and
         # handleGroupChatMessage but we handle them in the second.
         return
      elif mtype != "chat":
         t0( "handleMessage recieve a non chat type which will be ignored:", mtype )
         return
      username = message[ "from" ].bare
      id_ = message[ "id" ]
      if id_.startswith( "Rarista-" + self.boundjid.user ):
         # This is a 'R'esponse to a command I sent
         # Take out the 'R' that was pre-pended to it
         id_ = id_[ 1: ]
         pendingMessage = self.messagesAwaitingResponse_.get( ( username, id_ ) )
         if pendingMessage:
            pendingMessage[ username ] += message[ "body" ]
         else:
            pass  # I am no longer expecting this response
      elif id_.startswith( "arista" ):
         # This is a command from another switch
         # prepend an 'R' and respond
         self.sendResponse( message, "R" + id_ )
      else:
         # This is a command comming from a chat client
         # response without modify id
         self.sendResponse( message, id_ )

   def sendResponse( self, message, id_ ):
      t0( "sendResponse", message, id_ )
      muser = message[ 'from' ]
      self._sendChatState( "composing", muser, id_ )
      message[ "body" ] = self.getResponse( message )
      response = message[ 'body' ].splitlines()
      for start in xrange( 0, len( response ), LINE_LIMIT ):
         self.makeMessage( muser,
                           "\n".join( response[ start : start + LINE_LIMIT ] ),
                           mtype="chat", mid=id_ ).send()
      self._sendChatState( "active", muser, id_ )

   def _sendChatState( self, chatstate, muser, id_=None, group=False ):
      replyMessage = self.make_message( muser )
      if id_.startswith( 'Rarista-' ):
         # take out the 'R'
         replyMessage[ 'id' ] = id_[ 1: ]
      replyMessage[ 'chat_state' ] = chatstate
      if group:
         replyMessage[ 'type' ] = 'groupchat'
      replyMessage.send()

   # -----------------------
   # Group-chat methods

   def handleGroupChatMessage( self, message ):
      username, room = message[ 'mucnick' ], message[ 'mucroom' ]
      id_ = message[ 'id' ]
      if id_.startswith( "Rarista-" + self.boundjid.user ):
         # This is a 'R'esponse to a command that I sent
         # Take out the 'R' that was pre-pended to it
         id_ = id_[ 1: ]
         if ( room, id_ ) in self.awaitingResponseStatus_:
            self.messagesAwaitingResponse_[ ( room, id_ ) ][ username ] \
                += message[ 'body' ]
         else:
            pass # I am no longer expecting this response
      elif not room:
         # Usually an error message; don't join, trace it.
         t0( "Empty room name in group chat message. Message body:",
             message[ "body" ] )
      elif not username:
         t0( "Empty user name in group chat message. Message body:",
             message[ "body" ] )
      elif id_.startswith( 'arista-' ):
         # This is a command from another switch
         # prepend an 'R' and respond
         self.sendGroupResponse( message, 'R' + id_, username )
      elif self.muc_.rooms[ room ][ username ][ 'status' ].startswith(
            XmppModelNameReactor.PRESENCE_PREFIX ):
         pass # This a response from an arista switch from a chat client, ignore
      else:
         # This is a command comming from a chat client
         # Respond without modifying the id
         self.sendGroupResponse( message, id_, username )

   def sendGroupResponse (self, message, id_, username ):
      ''' This message requires a response, we first check
      if it applies to us then respond appropiately. '''
      room = message[ 'mucroom' ]
      body = message[ 'body' ]
      desiredRecipients, message[ 'body' ] = extractUser( body )
      if not desiredRecipients or self.boundjid.user in desiredRecipients:
         self._sendChatState( 'composing', room, id_, True )
         response = self.getResponse( message, username )
         if len( response.splitlines() ) > LINE_LIMIT:
            # XXX_HUAN: Have yet to handle messages that are too large to send, which
            # needs to be broken up into pieces. I find that hard to read for a chat
            # client user as piecies of the message from the switches will be
            # recieved in an interveled manner. Maybe, we should just not allow long
            # messages to be sent in a chatroom, rather only allow it in a direction
            # chat.
            response = 'Response too long to send. ' \
                'Try sending the command to me directly.'
         m = message.reply( response )
         m[ 'id' ] = id_
         m.send()
         self._sendChatState( 'active', room, id_, True )

   def _logError( self, room, error ):
      """Logs the appropriate error message for a presence's error stanza."""
      errType = error[ "type" ]
      errCondition = error[ "condition" ]
      errText = error[ "text" ]
      t9( "logError type:", errType, "condition:", errCondition, "text:", errText )
      if errType == "auth":
         if errCondition == "not-authorized":
            Logging.log( XmppLogMsgs.XMPP_GROUP_NOT_AUTHORIZED, room )
         elif errCondition == "registration-required":
            Logging.log( XmppLogMsgs.XMPP_GROUP_REGISTRATION_REQUIRED, room )
         elif errCondition == "forbidden":
            Logging.log( XmppLogMsgs.XMPP_GROUP_FORBIDDEN,
                         room, self.host_, self.port_ )

   def handleGroupChatPresence( self, presence ):
      t9( "handleGroupChatPresence presence:", presence )
      user = presence[ "muc" ][ "nick" ]
      room = presence[ "muc" ][ "room" ]
      error = presence.getStanzaValues().get( "error|%s" % presence.get_lang() )
      owner = bool( presence[ "muc" ][ "affiliation" ] == "owner" )
      if error:
         return self._logError( room, error )
      if not "@" in user:
         # User is hiding its identity in the chat, so we can't friend
         # them directly. Arista switches never do this, so we know
         # that we'll at least be able to friend them (and prevent
         # recursively talking to each other).
         return
      # Make sure we're friends with this neighbor. We'll then update
      # their presence.
      # Note: client_roster is unsafe to iterate over.
      if not user in self.client_roster.keys():
         self.sendPresenceSubscription( user )
      # If presence is from myself and if the room is in config_.group
      # then update status_.group to reflect that I'm in the room
      roomUtf8 = room.encode( "utf8" )
      if user == self.boundjid.bare and roomUtf8 in self.config_.group:
         if owner:
            # Set the password on the group if it's set in our config.
            self._updateRoomConfig( user, room )
         self.status_.group[ roomUtf8 ] = True

   def _updateRoomConfig( self, user, room ):
      # If we're the owner and we think the room should have
      # a password, make sure the room is configured that way.
      roomConfig = self.config_.group.get( room.encode( "utf8" ) )
      if roomConfig and roomConfig.password:
         t0( "Checking configuration for switch-group:", room )
         try:
            form = reparseForm( self.muc_.getRoomConfig( room ) )
         except ( sleekxmpp.exceptions.IqError,
                  sleekxmpp.exceptions.IqTimeout ), e:
            t0( "Could not get group config for room:", room, "error:", str( e ) )
            return

         changed = False
         if not form.field[ "muc#roomconfig_passwordprotectedroom" ][ "value" ]:
            changed = True
            form.field[ "muc#roomconfig_passwordprotectedroom" ][ "value" ] = True
         password = ReversibleSecretCli.decodeKey( roomConfig.password )
         if not form.field[ "muc#roomconfig_roomsecret" ][ "value" ] == password:
            form.field[ "muc#roomconfig_roomsecret" ][ "value" ] = password
            changed = True
         try:
            if changed:
               t0( "Updating configuration for switch-group:", room )
               form.set_type( "submit" )
               self.muc_.setRoomConfig( room, form )
         except ( sleekxmpp.exceptions.IqError,
                  sleekxmpp.exceptions.IqTimeout ), e:
            t0( "Could not set group config for room:", room, "error:", str( e ) )

   def handleGroupChatInvite( self, message ):
      # Can't use message[ 'mucroom' ] because type of message is
      # 'normal', not 'groupchat'
      room = message[ "from" ].bare
      self.joinGroupChatRoom( room )

   def joinGroupChatRoom( self, room ):
      """Joins the switch-group named 'room' (str)."""
      config = self.config_.group.get( room.encode( "utf8" ) )
      password = ReversibleSecretCli.decodeKey( config.password ) if config else ""
      # Room will be coming from sysdb, so it'll be UTF8. decode it into unicode.
      self.muc_.joinMUC(
         room.decode( "utf8" ),
         self.boundjid.bare,  # Use bare JID as our nick so other switches find us
         password=password,
         pstatus=self.status_.presenceStatus )

   def leaveGroupChatRoom( self, room ):
      self.muc_.leaveMUC( room, self.boundjid.bare )
      del self.status_.group[ room.encode( "utf8" ) ]

   def isAristaSwitch( self, user ):
      """Returns True if the username supplied looks like an Arista switch."""
      return self.client_roster[ user ][ 'status' ].startswith(
         XmppModelNameReactor.PRESENCE_PREFIX )

   def startAwaitingResponse( self, from_ , id_, message_, group=False ):
      """ Start the capture of messages that correspond to the arguments.

      These messages are saved to 'messagesAwaitingResponse' dict as
      opposed to being processed normally by the Xmpp client. """

      t8( "startAwaitingResponse from:", from_, "id:", id_, "message:", message_ )
      if not group:
         # Not a groupchat, so only add the user chatting with us
         usernames = ( from_, )
      else:
         users, _ = extractUser( message_ )
         roster = self.muc_.rooms[ from_ ]
         usernames = [ username for username in roster
                       if roster[ username ][ 'status' ].startswith(
                          XmppModelNameReactor.PRESENCE_PREFIX ) ]
         if users:
            # this message is directed to only some users, filter them out..
            # user is a username without a domain, our current invariant is keeping
            # track of the full username.
            usernames =  [ username for username in usernames
                           if username.split( '@' )[ 0 ] in users ]
         if not usernames:
            return # There is no one here to recieve our message

      self.messagesAwaitingResponse_[ ( from_, id_ ) ] = dict(
         ( username, '' ) for username in usernames )
      self.awaitingResponseStatus_[ ( from_, id_ ) ] =  dict(
         ( username, '' ) for username in usernames )

   def stopAwaitingResponse( self, from_, id_  ):
      """ Stop the capture of the messages that correspond to
      the arguments by clearing them out of the dict 'messagesAwaitingResponse'
      and return the entry's value. """
      del self.messagesAwaitingResponse_[ ( from_, id_ ) ]
      del self.awaitingResponseStatus_[ ( from_, id_ ) ]

   def getMessageResponse( self, from_, id_ ):
      assert ( from_, id_ ) in self.messagesAwaitingResponse_
      assert ( from_, id_ ) in self.awaitingResponseStatus_
      ret = self.messagesAwaitingResponse_[ from_, id_ ]
      self.stopAwaitingResponse( from_, id_ )
      return ret

   def _newNeighbor( self, name ):
      return self.status_.newNeighbor( name.encode( "utf8" ) )


class XmppConfigReactor( Tac.Notifiee ):
   """A TACC notifiee that reacts to changes in the XMPP agent configuration."""

   notifierTypeName = "Mgmt::Xmpp::Config"

   def __init__( self, config, status, vrfStatusLocal, entityManager,
                 ipStatus, disableAaa=False ):
      t0( "XmppConfigReactor init" )
      Tac.Notifiee.__init__( self, config )
      self.entityManager_ = entityManager
      self.client_ = None
      self.status_ = status
      self.config_ = config
      self.disableAaa_ = disableAaa
      self.reconfigCount_ = 0
      self.connectCount_ = 0
      self.allVrfStatusLocal_ = vrfStatusLocal
      self.dns_ = Aresolve.Querier( self._dnsResponse, longTime=DNS_RETRY_PERIOD )
      self.lastNameResolved_ = ""
      self.lastXmppServers_ = []
      self.dnsRecords_ = {}
      self.dnsQueryInProgress_ = False
      self.ipStatus_ = ipStatus
      self.ipStatusReactor_ = None
      if self.config_.srcIntfName:
         self._registerIpIntfStatusReactor( self.config_.srcIntfName )
      # Users must call start() to pickup any pending DNS resolutions.

   def start( self ):
      self.handleIpAddrOrHostname()

   def client( self ):
      return self.client_

   def reconfigCount( self ):
      return self.reconfigCount_

   def connectCount( self ):
      return self.connectCount_

   @Tac.handler( "enabled" )
   def handleEnabled( self ):
      self._reconfigure()

   @Tac.handler( "ipAddrOrHostname" )
   def handleIpAddrOrHostname( self ):
      """Handles a configuration change of the address of the XMPP server."""
      n = self.notifier_
      version = isValidIp( n.ipAddrOrHostname )
      # Stop querying the old name.
      if self.lastNameResolved_ != n.ipAddrOrHostname:
         self._finishDns( self.lastNameResolved_ )
      if version is not None:
         t9( "ipAddrOrHostname", repr( n.ipAddrOrHostname ),
             "is an IPv%d address" % version )
         # No resolution required for IP addresses
         self.lastNameResolved_ = n.ipAddrOrHostname
         self.lastXmppServers_ = [ n.ipAddrOrHostname ]
      else:
         t9( "ipAddrOrHostname", repr( n.ipAddrOrHostname ), "is not an IP address" )
         # The configured IP or host is a hostname or the empty string (removed)
         # Start a new DNS query if needed
         if n.ipAddrOrHostname:
            # Don't increase the total number of queries for this name
            t8( "Querying DNS for", repr( n.ipAddrOrHostname ) )
            self.dns_.finishHost( n.ipAddrOrHostname )
            # We set the query in progress attribute before querying, in case
            # the callback (which ends inprogress status) is called before
            # query() returns.
            self.dnsQueryInProgress_ = True
            self.dns_.host( n.ipAddrOrHostname )
            # Set this now so that changes after this will cease querying this name
            self.lastNameResolved_ = n.ipAddrOrHostname
      self._reconfigure()

   def _finishDns( self, name ):
      """Finishes the outstanding DNS query for name if it exists."""
      if not name:
         return
      t9( "Finishing DNS query for", repr( name ) )
      self.dns_.finishHost( name )
      try:
         del self.dnsRecords_[ name ]
      except KeyError:
         pass

   @Tac.handler( "port" )
   def handlePort( self ):
      self._reconfigure()

   @Tac.handler( "username" )
   def handleUsername( self ):
      self._reconfigure()

   @Tac.handler( "password" )
   def handlePassword( self ):
      self._reconfigure()

   def _closeIpStatusReactors( self ):
      if self.ipStatusReactor_:
         self.ipStatusReactor_.close()
         self.ipStatusReactor_ = None

   def _registerIpIntfStatusReactor( self, srcIntfName ):
      reactorFilter = lambda x: x == srcIntfName
      self.ipStatusReactor_ = Tac.collectionChangeReactor(
         self.ipStatus_.ipIntfStatus, IpIntfStatusReactor,
         reactorArgs=( weakref.ref( self ), ), reactorFilter=reactorFilter )

   @Tac.handler( "srcIntfName" )
   def handleSrcIntfName( self ):
      t0( "Source interface configuration changed to :%s"
          % self.notifier_.srcIntfName )
      self._closeIpStatusReactors()
      srcIntf = self.notifier_.srcIntfName
      if srcIntf:
         self._registerIpIntfStatusReactor( srcIntf )

      """
      If the new interface has invalid IP address or if it is not in
      configured vrf, XmppClient.connect will halt connection establishment.
      """
      self._reconfigure()

   @Tac.handler( "vrfName" )
   def handleVrfChange( self ):
      t0( "Vrf configuration changed to %s" % self.notifier_.vrfName )
      if self.notifier_.vrfName in DEFAULT_VRF_NAMES:
         self._updateVrfAndReconfigure( None )
      else:
         # Non-default VRF case
         vrf = self.allVrfStatusLocal_.vrf.get( self.notifier_.vrfName )
         # Note that if the VRF state is not active, the connection won't
         # work, but we're doing what the user expected. If the VRF doesn't
         # exist, we'll switch to the default VRF
         self._updateVrfAndReconfigure( vrf )

   def vrfStateIs( self, vrfName, state ):
      """Handles an external change in VRF state.

      If the VRF is the one we've configured the XMPP subsystem to use,
      the new status of that VRF will be responded to.
      """
      t0( "Handling VRF", vrfName, "state change to", state )
      if vrfName == self.config_.vrfName:
         # We're interested in this VRF change
         vrf = self.allVrfStatusLocal_.vrf.get( vrfName )
         if vrf and vrf.state == VrfState.active:
            self._updateVrfAndReconfigure( vrf )
         else:
            # The VRF is either initializing or being deleted.
            self._shutdownXmppClient()

   def _updateVrfAndReconfigure( self, vrf ):
      if vrf is not None:
         self.status_.vrfName = vrf.vrfName
      else:
         # Default VRF
         self.status_.vrfName = IpLibConsts.DEFAULT_VRF
      """
      If the source interface is not in configured vrf, XmppClient.connect will
      halt connection establishment.
      """
      self._reconfigure()

   def _reconfigure( self ):
      self.reconfigCount_ += 1
      n = self.notifier_
      t8( "Reconfiguring XMPP Agent" )
      t9( "Configuration:", dumpEntity( n ) )
      # Shutdown any current XMPP client connection
      self._shutdownXmppClient()
      # If sufficiently configured and DNS is resolved, connect to the server
      self._maybeStartXmppClient()

   def restartClient( self ):
      self._reconfigure()

   def resetSourceIntf( self ):
      self._shutdownXmppClient()
      self.status_.srcIntfName = Tac.Value( "Arnet::IntfId" )

   def _shutdownXmppClient( self ):
      """Shuts down the XMPP client and sets the Sysdb state to notConnected."""
      if self.client_:
         try:
            self.client_.shutdown()
         finally:
            self.client_ = None

      self.status_.connectionState = ConnectionState.notConnected
      self.status_.enabled = False

   def agentConfigured( self ):
      """Returns True if the agent is sufficiently configured."""
      configured = bool( self.config_.enabled and
                         self.config_.domainName and
                         self.config_.username and
                         self.config_.password and
                         self.config_.ipAddrOrHostname and
                         self.config_.port )
      if not configured:
         t8( "Not starting Xmpp agent based on current configuration" )
      else:
         t0( "Sufficient configuration to start XMPP subsystem" )
      return configured

   def resolved( self ):
      """Returns True if the DNS state has resolved for the latest configuration."""
      resolved = bool( self.lastXmppServers_ and
                       self.config_.ipAddrOrHostname == self.lastNameResolved_ and
                       not self.dnsQueryInProgress_ )
      addnl = "" if resolved else "NOT "
      t8( "Xmpp agent has %sresolved XMPP server hostname" % addnl,
          repr( self.config_.ipAddrOrHostname ) )
      return resolved

   def _maybeStartXmppClient( self ):
      """Conditionally (re)starts the XMPP client if ready."""
      if self.agentConfigured() and self.resolved():
         self._buildNewClient()

   def _buildNewClient( self ):
      jid = '%s/%s' % ( self.notifier_.username,  # username contains the domainName
                        'Arista.%s' % str( uuid.uuid4().hex )[:8] )
      t0( "Creating XMPP client", jid )
      self.client_ = XmppClient( jid,
                                 self.notifier_.password,
                                 self.lastXmppServers_,
                                 self.notifier_,
                                 self.status_,
                                 self.entityManager_,
                                 self.disableAaa_,
                                 weakref.proxy( self ) )
      t0( "Created XMPP client", self.client_.boundjid.full )

      # This call must not block (for any appreciable period),
      # hence self.auto_reconnect must be False.
      self.client_.connect()
      # Even if we didn't succecsfully connect, the client
      # exists, so we have to do cleanup; thus we set status
      # enabled to allow us to cleanup later.
      self.status_.enabled = True

   def _dnsRecordEq( self, new, old ):
      """Compares two DNS records, 'new' and 'old'.

      If the lastRefresh attribute changed, it means that there's updates
      available, but we may not see them in this pass due to per-attribute
      updates. So we ignore that and just look at the data.
      """
      if ( new.valid != old.valid or
           new.lastError != old.lastError or  # A new error or error resolution
           sorted( new.ipAddress ) != sorted( old.ipAddress ) or
           sorted( new.ip6Address ) != sorted( old.ip6Address ) ):
         return False
      return True

   def _dnsResponse( self, record ):
      """Handles a DNS response. Called from Aresolve.DnsReactor.

      Args:
        record: a Aresolve.DnsRecord, the results from Aresolve.
      """
      self.dnsQueryInProgress_ = False
      t9( "DNS record:", record )
      update = True
      ourRecord = self.dnsRecords_.get( record.name )
      if not ourRecord:
         # DNS response for a new name
         t8( "Creating new DNS record for", repr( record.name ), "with lastRefresh:",
             record.lastRefresh )
      elif self._dnsRecordEq( record, ourRecord ):
         # No changes to report
         update = False

      if update:
         t8( "Updating DNS record for", repr( record.name ) )
         self.lastNameResolved_ = record.name
         if ourRecord is not None:
            t9( "Old record:", ourRecord )
         self.dnsRecords_[ record.name ] = record
         self._updateXmppServers()

      # We want to update the DNS record (and potentially reconnect) if
      # we've just started up (no client or not connected) or the DNS updated.
      if not self.client_ or not self.client_.connected() or update:
         t9( "DNS response triggering XMPP client restart" )
         self._shutdownXmppClient()
         self._maybeStartXmppClient()
      t9( "Completed DNS update for", record.name )

   def _updateXmppServers( self ):
      """Builds the client XMPP server list from all resolved record addresses."""
      hosts = []
      for record in self.dnsRecords_.values():
         hosts.extend( record.ip6Address )
         hosts.extend( record.ipAddress )
      # Set the servers to be configured on the client
      self.lastXmppServers_ = hosts
      t9( "lastXmppServers_ now:", hosts )

   @Tac.handler( "group" )
   def handleGroup( self, room ):
      if not self.status_.connectionState == ConnectionState.connected:
         return
      # check for diff between status and config and join/leave appropriately
      configSet = set( g.decode( "utf8" ) for g in self.config_.group )
      t0( "configSet:", configSet )
      statusSet = set( g.decode( "utf8" ) for g in self.status_.group )
      t0( "statusSet:", statusSet )
      # join room in config minus status
      configOnly = configSet - statusSet
      if room in configOnly:
         self.client_.joinGroupChatRoom( room )
      # leave room in status minus config
      statusOnly = statusSet - configSet
      if room in statusOnly:
         self.client_.leaveGroupChatRoom( room )


class XmppStatusReactor( Tac.Notifiee ):
   """A TACC notifiee that responds to Xmpp status changes."""

   notifierTypeName = "Mgmt::Xmpp::Status"

   def __init__( self, xmppStatus, configReactor ):
      t0( "XmppStatusReactor init" )
      Tac.Notifiee.__init__( self, xmppStatus )
      self.status_ = xmppStatus
      self.configReactor_ = configReactor
      t0( "XmppStatusReactor init end" )

   @Tac.handler( "presenceLastUpdateTime" )
   def handleState( self ):
      """Handles an XMPP presence update from the model name reactor."""
      t8( "presenceLastUpdateTime:", self.notifier_.presenceLastUpdateTime )
      # If we are connected, we can update our status.
      # If we haven't yet connected, we are yet to set our initial status.
      client = self.configReactor_.client()
      if client and self.status_.connectionState == ConnectionState.connected:
         client.send_presence( pstatus=self.status_.presenceStatus )


class VrfStatusLocalReactor( Tac.Notifiee ):
   """A TACC notifiee that responds to changes in VRF status.

   The XMPP client will be disabled if we're configured to use a VRF
   and it's not active or is deleted.
   """

   notifierTypeName = "Ip::VrfStatusLocal"

   def __init__( self, vrfStatusLocal, configReactor ):
      Tac.Notifiee.__init__( self, vrfStatusLocal )
      self.configReactor_ = configReactor
      self.vrfName = vrfStatusLocal.vrfName

   @Tac.handler( "state" )
   def handleState( self ):
      """Handles an explicit VRF state change."""
      t8( "vrfName:", self.vrfName, "state:", self.notifier_.state )
      self.configReactor_.vrfStateIs( self.vrfName, self.notifier_.state )

class IpIntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Ip::IpIntfStatus"
   def __init__( self, notifier, configReactorRef ):
      Tac.Notifiee.__init__( self, notifier )
      self.configReactor_ = configReactorRef()

   @Tac.handler( 'activeAddrWithMask' )
   def handleActiveAddrWithMask( self ):
      t8( "Source interface %s ip address changed to %s" % ( self.notifier_.intfId,
          self.notifier_.activeAddrWithMask ) )
      self.configReactor_.restartClient()

   def close( self ):
      self.configReactor_.resetSourceIntf()
      Tac.Notifiee.close( self )

xmppAgent = None
class Xmpp( Agent.Agent ):
   """The CLI over XMPP agent.

   The agent manages an XmppClient, instructed by the XmppConfigReactor.
   """

   def __init__( self, em, disableAaa=False ):
      self.config_ = self.status_ = self.vrfStatusLocal_ = self.configReactor_ = None
      self.allVrfStatusLocalReactor_ = self.modelNameReactor_ = None
      self.entityMibStatus_ = self.statusReactor_ = self.ipStatus_ = None
      self.warm_ = False
      self.disableAaa_ = disableAaa
      self.em_ = em

      global xmppAgent
      xmppAgent = self
      Agent.Agent.__init__( self, em )

   def doInit( self, em ):
      t0( "Starting Xmpp Agent" )
      mg = em.mountGroup()
      # Mount XMPP agent configuration and status
      self.config_ = mg.mount( "mgmt/xmpp/config", "Mgmt::Xmpp::Config", "r" )
      self.status_ = mg.mount( "mgmt/xmpp/status", "Mgmt::Xmpp::Status", "w" )
      # Mount the entity MIB status for hardware model name details
      self.entityMibStatus_ = mg.mount( "hardware/entmib", "EntityMib::Status", "r" )
      # Mount the local VRF state to react to VRF changes
      self.vrfStatusLocal_ = mg.mount( Cell.path( "ip/vrf/status/local" ),
                                       "Ip::AllVrfStatusLocal", "r" )
      Tac.Type( "Ira::IraIpStatusMounter" ).doMountEntities( mg.cMg_, True, False )
      self.ipStatus_ = mg.mount( "ip/status", "Ip::Status", "r" )
      # Run _mountsDone after the mount group finishes
      mg.close( self._mountsDone )

   def _mountsDone( self ):
      """Activities to perform once the mounts are complete."""
      # Create reactors for Mgmt::Xmpp::{Config,Status} changes
      self.configReactor_ = XmppConfigReactor(
         self.config_, self.status_, self.vrfStatusLocal_, self.em_,
         self.ipStatus_, disableAaa=self.disableAaa_ )
      self.statusReactor_ = XmppStatusReactor( self.status_, self.configReactor_ )
      # Create the model name reactor to set our status message.
      self.modelNameReactor_ = XmppModelNameReactor.EntityMibRootReactor(
         self.entityMibStatus_, self.configReactor_ )
      # Create reactor to respond to VRF status changes
      self.allVrfStatusLocalReactor_ = Tac.collectionChangeReactor(
         self.vrfStatusLocal_.vrf,
         VrfStatusLocalReactor,
         reactorArgs=( weakref.proxy( self.configReactor_ ), ) )
      self.warm_ = True
      t0( "Xmpp agent is now warm" )
      # Start the config reactor once warm, potentially creating a client connection
      self.configReactor_.start()

   def xmppClient( self ):
      return self.configReactor_.client()

   def warm( self ):
      return self.warm_


def extractUser( body ):
   body = body.strip()
   if not body:
      return None, None
   groupChatMultRe = re.compile( r"(\w+)[:;,] *(\w.*)" )
   if not groupChatMultRe.match( body ):
      return None, body
   users = set()
   while True:
      matchObj = groupChatMultRe.match( body )
      if not matchObj:
         break
      users.add( matchObj.group( 1 ) )
      body = matchObj.group( 2 )
   return users, body


def isValidIp( ip ):
   """Returns an int, 4 or 6 if ip is an IP address of that family, else None."""
   try:
      IpUtils.IpAddress( ip )
      return 4
   except ValueError:
      # Invalid or IPv6
      try:
         IpUtils.IpAddress( ip, addrFamily=socket.AF_INET6 )
         return 6
      except ValueError:
         return None


def dumpEntity( ent ):
   cmd = []
   for attr in sorted( filter( lambda a: a not in IGNORE_ATTRS, ent.attributes ) ):
      cmd += [ "%s=%s" % ( attr, repr( getattr( ent, attr, "__unset__" ) ) ) ]
   return ", ".join( cmd )


def reparseForm( form ):
   """Fix missing namespaces in the provided XML form."""
   xml = sleekxmpp.xmlstream.ET.fromstring( str( form ) )
   return sleekxmpp.plugins.xep_0004.Form( xml=xml )


def main():
   """Main Agent startup function, called from bin/Xmpp."""
   container = Agent.AgentContainer( [ Xmpp ], passiveMount=True )
   container.addOption( "", "--disable-aaa", dest="disableAaa",
                        action="store_true", default=False,
                        help="Do not communicate with the Aaa agent. This "
                        "disables authentication, authorization and accounting "
                        "of commands executed in this Xmpp instance.",
                        agentClass=Xmpp )
   container.runAgents()


# This script may be run as the agent binary from test frameworks,
# so start up the agent if so.
if __name__ == "__main__":
   main()
