#!/usr/bin/env python
# Copyright (c) 2008-2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import Plugins, Arnet
import re
import Cell
from TypeFuture import TacLazyType

def populateAttenuationMap( attnTable ):
   Plugins.loadPlugins( 'XcvrPlugin', context=attnTable )


def xcvrInfo( intfs, xcvrType=None, newXcvrNameIsModular=None ):
   # Returns a tuple of ( XcvrControllerName, [ XcvrNames ] )
   # If there is one intf in the collection, the xcvrCtrlName and xcvrName 
   # takes its name. In the case of RCx modules, xcvrCtrlName will take the common
   # root of the name.
   # For multiple intfs (e.g. QSFP) take the common root of the name (less the / )
   # e.g. Ethernet50/1 ... Ethernet50/4 would give a xcvr/controller name of 
   #      Ethernet50
   # e.g. Ethernet3/50/1 ... Ethernet3/50/12 would give a xcvr/controller name of 
   #      Ethernet3/50
   # For multiple roots the xcvr/Controller name is as follows:
   # e.g. Ethernet49/1 ... Ethernet50/12 would give xcvrCtrlName of Ethernet49-50
   # and xcvrNames of [ Ethernet49, Ethernet50 ]
   # newXcvrNameIsModular = None by default, True when modular, 
   #                        False when fixedSystem
   #                        Added for BUG207171, and used with the new
   #                        interface naming, where a single-speed only
   #                        port (think 40g-only, etc) now has a trailing
   #                        /1 on new platforms.

   if len( intfs ) == 1:
      if xcvrType == 'rcx':
         return ( intfs[ 0 ].rsplit( '/', 1 )[ 0 ], [ intfs[ 0 ] ] )
      if newXcvrNameIsModular is not None:
         modular = newXcvrNameIsModular
         numSlash = intfs[ 0 ].count( '/' )
         # A modular system should have the xcvrControllerName with a single '/'
         # A fixed system should have the xcvrControllerName with no '/'
         if ( modular and numSlash > 1 ) or ( not modular and numSlash > 0 ):
            lastSlash = intfs[ 0 ].rfind( '/' )
            return ( intfs[ 0 ][ : lastSlash ], [ intfs[ 0 ] ] )
      return ( intfs[ 0 ], [ intfs[ 0 ] ] )

   nms = set()
   for nm in intfs:
      nms.add( nm.rsplit( '/', 1 )[ 0 ] )

   nms = list( nms )
   xcvrNames = Arnet.sortIntf( nms )
   numIntfs = len( xcvrNames ) 
   assert ( numIntfs == 1 or numIntfs == 2 ), \
          "Currently max two XcvrSlots per XcvrController supported."
   xcvrCtrlName = xcvrNames[ 0 ]
   for i in range(1, numIntfs ):
      m = re.search( r'(\D*)(\d+)', xcvrNames[ i ].rsplit( '/', 1 )[ -1 ] )
      # verify that the intfs match the assumption that the name before the last
      # is the same on all of the interfaces.
      assert m 
      xcvrCtrlName += "-%s" % m.group( 2 )
   return ( xcvrCtrlName, xcvrNames )

def allStatusDirType():
   return 'Xcvr::AllStatusDir'

def getXcvrSlotName( name ):
   # Get the correct Xcvr slot name for an interface by removing lanes that
   # may be listed.
   fwdSlashIdx = [ i for i, ltr in enumerate( name ) if ltr == '/' ]
   if ( Cell.cellType() in ( 'fixed', 'generic' ) and len( fwdSlashIdx ) == 1 ) \
        or ( Cell.cellType() == 'supervisor' and len( fwdSlashIdx ) == 2 ) :
      return name[ 0:fwdSlashIdx[ -1 ] ]
   return name

def getLinecardName( intfName ):
   m = re.match( "Ethernet([0-9]+)/([0-9]+)", intfName )
   assert m, 'Could not parse interface: %s' % intfName
   return 'Linecard%s' % m.group( 1 )

def isManagementIntf( intfName ):
   return re.match( r"Management\d+/\d+", intfName ) != None

def getNumIntfs( status ):
   # each interface of cfp2-dco capable port represents 100G host interface
   # if non-coherent 100G CFP2 transceiver is inserted, set number of intfs to 1
   if status.xcvrType.lower() == "cfp2" and \
      status.cfp2SlotCapabilities.cfp2DcoSupported and \
      status.mediaTypeString != "100G-DWDM-DCO":
      numIntfs = 1
   else:
      numIntfs = len( status.xcvrConfig.intfName )
   return numIntfs

def getIntfNameByLane( laneId, status, maxChannels, intfNames ):
   numIntfs = getNumIntfs( status )
   if maxChannels > numIntfs:
      # Collection intfName is keyed by the first lane belonging to the
      # interface E.g.: a 4-lane port (QSFP) containing 2 interfaces will have
      # a /1 interface at key 0 and /3 interface at key 2.
      assert ( maxChannels % numIntfs ) == 0
      lanesPerIntf = maxChannels // numIntfs
      intfIdx = lanesPerIntf * ( laneId // lanesPerIntf )
   else:
      intfIdx = laneId
   intfName = status.xcvrConfig.intfName.get( intfIdx )

   if intfName not in intfNames:
      # XXX_LWR: we have to be careful that the names for
      #          interfaces that correspond to multi-lane xcvrs
      #          are handled properly by the interface range
      #          stuff.
      return ""

   if maxChannels > numIntfs:
      # Always append lane to the printable interface name, even if the
      # interface does not specify one (e.g. 40G-only interface).
      name = "%s/%d" % ( getXcvrSlotName( intfName ), laneId + 1 )
   else:
      name = intfName
   return name

MgmtIntfId = TacLazyType( "Arnet::MgmtIntfId" )
EthIntfId = TacLazyType( "Arnet::EthIntfId" )

def getLaneId( intfName ):
   if MgmtIntfId.isMgmtIntfId( intfName ):
      return 0
   else:
      return EthIntfId.lane( intfName )

def getLineSideChannels( status ):
   if status.mediaTypeString == "100G-DWDM-DCO":
      # In CFP2-DCO transceiver, there's only one network interface,
      # while maxChannels is currently representing host interfaces.
      # We only want to show information for interface /1
      # BUG370088: the current status.capabilities.lineSideChannels for
      #            CFP2-DCO is set to 2 which is wrong. CFP2-DCO only has
      #            1 optical channel
      lineSideChannels = 1
   else:
      lineSideChannels = status.capabilities.lineSideChannels
   return lineSideChannels

def getPrimaryIntfName( primaryIntfNames, inactiveIntfName ):
   """
   This function is used to get the primary interface that an inactive interface
   is subsumed to when it is known an inactive interface is being passed in.

   This works on the assumption that the primary interface can be determined
   by finding the nearest primary lane that is smaller than the current
   interface's lane. (e.g. et1/1 is the primary interface for et1/2,
   whereas et1/3 will never be et1/2's primary interface)
   """
   if primaryIntfNames and inactiveIntfName:
      curIntfLaneNum = inactiveIntfName.split( '/' )
      intfNamesMaxToMin = Arnet.sortIntf( primaryIntfNames )
      # Handle Agile which are SFP and don't have sub intf lanes
      if len( curIntfLaneNum ) == 1 or len( intfNamesMaxToMin ) == 1:
         return intfNamesMaxToMin[ 0 ]
      intfNamesMaxToMin.reverse()
      for intf in intfNamesMaxToMin:
         if int( intf.split( '/' )[ -1 ] ) > int( curIntfLaneNum[ -1 ] ):
            continue
         return intf
   # Could not find a primary intf
   return None

# Helper functions. Return true if this is the specified form-factor adapter
def isQsfpToSfpSwizzler( status ):
   return status and status.xcvrType.lower() == "qsfpplus" and status.sfpStatus

def isOsfpToQsfpSwizzler( status ):
   return status and status.xcvrType == "osfp" and status.qsfpStatus

def isQsfpDdToQsfpSwizzler( status ):
   return status and status.xcvrType == "qsfpDd" and status.qsfpStatus

def isQsfpCmisToQsfpAdapter( status ):
   return status and status.xcvrType == "qsfpCmis" and status.qsfpStatus

# Helper functions to return the appropriate xcvrStatus/xcvrConfig, whether
# the intf is swizzled or not.
def getXcvrStatus( status ):
   # Check if this is a form-factor adapter
   if isQsfpToSfpSwizzler( status ):
      return status.sfpStatus
   if isOsfpToQsfpSwizzler( status ):
      if isQsfpToSfpSwizzler( status.qsfpStatus ):
         return status.qsfpStatus.sfpStatus
      return status.qsfpStatus
   if isQsfpDdToQsfpSwizzler( status ):
      if isQsfpToSfpSwizzler( status.qsfpStatus ):
         return status.qsfpStatus.sfpStatus
      return status.qsfpStatus
   if isQsfpCmisToQsfpAdapter( status ):
      if isQsfpToSfpSwizzler( status.qsfpStatus ):
         return status.qsfpStatus.sfpStatus
      return status.qsfpStatus
   return status

def getXcvrConfig( status, config ):
   # Check if this is a form-factor adapter
   if isQsfpToSfpSwizzler( status ):
      return status.sfpConfig
   if isOsfpToQsfpSwizzler( status ):
      if isQsfpToSfpSwizzler( status.qsfpStatus ):
         return status.qsfpStatus.sfpConfig
      return status.qsfpConfig
   if isQsfpDdToQsfpSwizzler( status ):
      if isQsfpToSfpSwizzler( status.qsfpStatus ):
         return status.qsfpStatus.sfpConfig
      return status.qsfpConfig
   if isQsfpCmisToQsfpAdapter( status ):
      if isQsfpToSfpSwizzler( status.qsfpStatus ):
         return status.qsfpStatus.sfpConfig
      return status.qsfpConfig
   return config

# Check if the mediaType is valid in this slot
def validMediaType( mediaType, xcvrStatus ):
   if not xcvrStatus:
      return False
   return xcvrStatus.isSupportedMediaType( mediaType )

XcvrMediaType = TacLazyType( 'Xcvr::MediaType' )
sfpMediaTypeCliData = [
   ( XcvrMediaType.xcvr10GBaseAr, '10gbase-ar',
     'Override the media type to be 10GBASE-AR' ),
   ( XcvrMediaType.xcvr1000BaseLx, '1000base-lx',
     'Override the media type to be 1000BASE-LX' ),
   ( XcvrMediaType.xcvr100BaseLx10, '100base-lx10',
     'Override the media type to be 100BASE-LX10' ),
   ( XcvrMediaType.xcvr10GBaseDwdmZt, '10gbase-dwdm',
     'Override the media type to be 10GBASE-DWDM' ),
   ( XcvrMediaType.xcvr1000BaseBx10D, '1000base-bx10-d',
     'Override the media type to be 1000BASE-BX10-D' ),
   ( XcvrMediaType.xcvr25GBaseCrN, '25gbase-crn',
     'Override the media type to be 25GBASE-CR CA-N' ),
   ( XcvrMediaType.xcvr25GBaseArN, '25gbase-arn',
     'Override the media type to be 25GBASE-AR' ),
   ( XcvrMediaType.xcvr10GBaseEr, '10gbase-er',
     'Override the media type to be 10GBASE-ER' ),
   ( XcvrMediaType.xcvr25GBaseCr, '25gbase-cr',
     'Override the media type to be 25GBASE-CR CA-L' ),
   ( XcvrMediaType.xcvr1000BaseSx, '1000base-sx',
     'Override the media type to be 1000BASE-SX' ),
   ( XcvrMediaType.xcvr10GBaseDwdmZr, '10gbase-dwdm-zr',
     'Override the media type to be 10GBASE-DWDM-ZR' ),
   ( XcvrMediaType.xcvr25GBaseCrS, '25gbase-crs',
     'Override the media type to be 25GBASE-CR CA-S' ),
   ( XcvrMediaType.xcvr10GBaseZr, '10gbase-zr',
     'Override the media type to be 10GBASE-ZR' ),
   ( XcvrMediaType.xcvr1000BaseBx10U, '1000base-bx10-u',
     'Override the media type to be 1000BASE-BX10-U' ),
   ( XcvrMediaType.xcvr10GBaseDwdmEr, '10gbase-dwdm-er',
     'Override the media type to be 10GBASE-DWDM-ER' ),
   ( XcvrMediaType.xcvr10GBaseErbdU, '10gbase-erbd-u',
     'Override the media type to be 10GBASE-ERBD-U' ),
   ( XcvrMediaType.xcvr10GBaseSrLite, '10gbase-srl',
     'Override the media type to be 10GBASE-SRL' ),
   ( XcvrMediaType.xcvr25GBaseLr, '25gbase-lr',
     'Override the media type to be 25GBASE-LR' ),
   ( XcvrMediaType.xcvr10GBaseT, '10gbase-t',
     'Override the media type to be 10GBASE-T' ),
   ( XcvrMediaType.xcvr1000BaseT, '1000base-t',
     'Override the media type to be 1000BASE-T' ),
   ( XcvrMediaType.xcvr10GBaseErbdD, '10gbase-erbd-d',
     'Override the media type to be 10GBASE-ERBD-D' ),
   ( XcvrMediaType.xcvr10GBaseSr, '10gbase-sr',
     'Override the media type to be 10GBASE-SR' ),
   ( XcvrMediaType.xcvr10GBaseErlbdD, '10gbase-erlbd-d',
     'Override the media type to be 10GBASE-ERLBD-D' ),
   ( XcvrMediaType.xcvr10GBaseLr, '10gbase-lr',
     'Override the media type to be 10GBASE-LR' ),
   ( XcvrMediaType.xcvr25GBaseAr, '25gbase-ar',
     'Override the media type to be 25GBASE-AR' ),
   ( XcvrMediaType.xcvr10GBaseCr, '10gbase-cr',
     'Override the media type to be 10GBASE-CR' ),
   ( XcvrMediaType.xcvr25GBaseLrLite, '25gbase-lrl',
     'Override the media type to be 25GBASE-LRL' ),
   ( XcvrMediaType.xcvr10GBaseErlbdU, '10gbase-erlbd-u',
     'Override the media type to be 10GBASE-ERLBD-U' ),
   ( XcvrMediaType.xcvr25GBaseSr, '25gbase-sr',
     'Override the media type to be 25GBASE-SR' ),
   ( XcvrMediaType.xcvr25GBaseXsr, '25gbase-xsr',
     'Override the media type to be 25GBASE-XSR' ),
   ( XcvrMediaType.xcvr10GBaseLrLite, '10gbase-lrl',
     'Override the media type to be 10GBASE-LRL' ),
   ( XcvrMediaType.xcvr50GBaseCr, '50gbase-cr',
     'Override the media type to be 50GBASE-CR' ),
]

# The dictionary mediaTypeEnumToClis and mediaTypeCliToEnums are used in
# CliSave and related tests and ptests in XcvrCtrl.
sfpMediaTypeEnumToClis = { a : b for a, b, _ in sfpMediaTypeCliData }
sfpTokensToHelp = { a : b for _, a, b in sfpMediaTypeCliData }

qsfpMediaTypeCliData = [
   ( XcvrMediaType.xcvr40GBasePlr4, '40gbase-plr4',
     'Override the media type to be 40GBASE-PLR4' ),
   ( XcvrMediaType.xcvr50GBaseCr2N, '50gbase-cr2n',
     'Override the media type to be 50GBASE-CR2 CA-N' ),
   ( XcvrMediaType.xcvr50GBaseCr2L, '50gbase-cr2',
     'Override the media type to be 50GBASE-CR2 CA-L' ),
   ( XcvrMediaType.xcvr100GBaseLr, '100gbase-lr',
     'Override the media type to be 100GBASE-LR' ),
   ( XcvrMediaType.xcvr100GBaseCr4, '100gbase-cr4',
     'Override the media type to be 100GBASE-CR4 CA-L' ),
   ( XcvrMediaType.xcvr40GBaseEr4, '40gbase-er4',
     'Override the media type to be 40GBASE-ER4' ),
   ( XcvrMediaType.xcvr100GBaseXsr4, '100gbase-xsr4',
     'Override the media type to be 100GBASE-XSR4' ),
   ( XcvrMediaType.xcvr100GBaseCwdm4, '100gbase-cwdm4',
     'Override the media type to be 100GBASE-CWDM4' ),
   ( XcvrMediaType.xcvr200GBaseCr4, '200gbase-cr4',
     'Override the media type to be 200GBASE-CR4' ),
   ( XcvrMediaType.xcvr100GBaseCr2, '100gbase-cr2',
     'Override the media type to be 100GBASE-CR2' ),
   ( XcvrMediaType.xcvr100GEDwdm2, '100ge-dwdm2',
     'Override the media type to be 100GE-DWDM2' ),
   ( XcvrMediaType.xcvr100GBaseCrActive4, '100gbase-cra4',
     'Override the media type to be 100GBASE-CRA4' ),
   ( XcvrMediaType.xcvr100GBaseEr4, '100gbase-er4',
     'Override the media type to be 100GBASE-ER4' ),
   ( XcvrMediaType.xcvr40GBaseSrbd, '40gbase-srbd',
     'Override the media type to be 40GBASE-SRBD' ),
   ( XcvrMediaType.xcvr100GBaseAr4, '100gbase-ar4',
     'Override the media type to be 100GBASE-AR4 CA-L' ),
   ( XcvrMediaType.xcvr100GBaseLrl4, '100gbase-lrl4',
     'Override the media type to be 100GBASE-LRL4' ),
   ( XcvrMediaType.xcvr100GBaseAr4N, '100gbase-ar4n',
     'Override the media type to be 100GBASE-AR4 CA-N' ),
   ( XcvrMediaType.xcvr100GBasePsm4, '100gbase-psm4',
     'Override the media type to be 100GBASE-PSM4' ),
   ( XcvrMediaType.xcvr40GBaseUniv, '40gbase-univ',
     'Override the media type to be 40GBASE-UNIV' ),
   ( XcvrMediaType.xcvr40GBaseCr4, '40gbase-cr4',
     'Override the media type to be 40GBASE-CR4' ),
   ( XcvrMediaType.xcvr40GBasePlrl4, '40gbase-plrl4',
     'Override the media type to be 40GBASE-PLRL4' ),
   ( XcvrMediaType.xcvr40GBaseLr4, '40gbase-lr4',
     'Override the media type to be 40GBASE-LR4' ),
   ( XcvrMediaType.xcvr40GBaseLrl4, '40gbase-lrl4',
     'Override the media type to be 40GBASE-LRL4' ),
   ( XcvrMediaType.xcvr100GBaseSr4, '100gbase-sr4',
     'Override the media type to be 100GBASE-SR4' ),
   ( XcvrMediaType.xcvr100GBaseCr4S, '100gbase-cr4s',
     'Override the media type to be 100GBASE-CR4 CA-S' ),
   ( XcvrMediaType.xcvr100GBaseSwdm4, '100gbase-swdm4',
     'Override the media type to be 100GBASE-SWDM4' ),
   ( XcvrMediaType.xcvr200GBaseFr4, '200gbase-fr4',
     'Override the media type to be 200GBASE-FR4' ),
   ( XcvrMediaType.xcvr50GBaseCr2S, '50gbase-cr2s',
     'Override the media type to be 50GBASE-CR2 CA-S' ),
   ( XcvrMediaType.xcvr40GBaseSr4, '40gbase-sr4',
     'Override the media type to be 40GBASE-SR4' ),
   ( XcvrMediaType.xcvr100GBaseErl4, '100gbase-erl4',
     'Override the media type to be 100GBASE-ERL4' ),
   ( XcvrMediaType.xcvr100GBaseFr, '100gbase-fr',
     'Override the media type to be 100GBASE-FR' ),
   ( XcvrMediaType.xcvr100GBaseXcwdm4, '100gbase-xcwdm4',
     'Override the media type to be 100GBASE-XCWDM4' ),
   ( XcvrMediaType.xcvr40GBaseXsr4, '40gbase-xsr4',
     'Override the media type to be 40GBASE-XSR4' ),
   ( XcvrMediaType.xcvr100GBaseDr, '100gbase-dr',
     'Override the media type to be 100GBASE-DR' ),
   ( XcvrMediaType.xcvr100GBaseCr4N, '100gbase-cr4n',
     'Override the media type to be 100GBASE-CR4 CA-N' ),
   ( XcvrMediaType.xcvr40GBasePsm4, '40gbase-psm4',
     'Override the media type to be 40GBASE-PSM4' ),
   ( XcvrMediaType.xcvr100GBaseSrbd, '100gbase-srbd',
     'Override the media type to be 100GBASE-SRBD' ),
   ( XcvrMediaType.xcvr40GBaseAr4, '40gbase-ar4',
     'Override the media type to be 40GBASE-AR4' ),
   ( XcvrMediaType.xcvr200GBaseSr4, '200gbase-sr4',
     'Override the media type to be 200GBASE-SR4' ),
   ( XcvrMediaType.xcvr100GBaseLr4, '100gbase-lr4',
     'Override the media type to be 100GBASE-LR4' ),
]
qsfpMediaTypeEnumToClis = { a : b for a, b, _ in qsfpMediaTypeCliData }
qsfpTokensToHelp = { a : b for _, a, b in qsfpMediaTypeCliData }

cfp2MediaTypeCliData = [
   ( XcvrMediaType.xcvr100GBaseEr4, '100gbase-er4',
     'Override the media type to be 100GBASE-ER4' ),
   ( XcvrMediaType.xcvr100GDwdmCoherentE, '100g-dwdm-e',
     'Override the media type to be 100G-DWDM-E' ),
   ( XcvrMediaType.xcvr100GBaseSr10, '100gbase-sr10',
     'Override the media type to be 100GBASE-SR10' ),
   ( XcvrMediaType.xcvr100GBaseXsr10, '100gbase-xsr10',
     'Override the media type to be 100GBASE-XSR10' ),
   ( XcvrMediaType.xcvr100GBaseLr4, '100gbase-lr4',
     'Override the media type to be 100GBASE-LR4' ),
   ( XcvrMediaType.xcvr100GDwdmCoherent, '100g-dwdm',
     'Override the media type to be 100G-DWDM' ),
]
cfp2MediaTypeEnumToClis = { a : b for a, b, _ in cfp2MediaTypeCliData }
cfp2TokensToHelp = { a : b for _, a, b in cfp2MediaTypeCliData }

osfpMediaTypeCliData = [
   ( XcvrMediaType.xcvr200GBaseCr8, '200gbase-cr8',
     'Override the media type to be 200GBASE-CR8 CA-L' ),
   ( XcvrMediaType.xcvr400GBaseCr8N, '400gbase-cr8n',
     'Override the media type to be 400GBASE-CR8 CA-N' ),
   ( XcvrMediaType.xcvr400GBaseCrActive8, '400gbase-cra8',
     'Override the media type to be 400GBASE-CRA8' ),
   ( XcvrMediaType.xcvr200GBaseCr8S, '200gbase-cr8s',
     'Override the media type to be 200GBASE-CR8 CA-S' ),
   ( XcvrMediaType.xcvr400GBaseSr8, '400gbase-sr8',
     'Override the media type to be 400GBASE-SR8' ),
   ( XcvrMediaType.xcvr400GOptical, '400g-optical',
     'Override the media type to be Optical400G' ),
   ( XcvrMediaType.xcvr400GBaseAr8, '400gbase-ar8',
     'Override the media type to be 400GBASE-AR8' ),
   ( XcvrMediaType.xcvr200GBaseCr8N, '200gbase-cr8n',
     'Override the media type to be 200GBASE-CR8 CA-N' ),
   ( XcvrMediaType.xcvr400GBaseXdr4, '400gbase-xdr4',
     'Override the media type to be 400GBASE-XDR4' ),
   ( XcvrMediaType.xcvr400GBaseLr8, '400gbase-lr8',
     'Override the media type to be 400GBASE-LR8' ),
   ( XcvrMediaType.xcvr400GBase2Fr4, '400gbase-2fr4',
     'Override the media type to be 400GBASE-2FR4' ),
   ( XcvrMediaType.xcvr400GBaseCr8, '400gbase-cr8',
     'Override the media type to be 400GBASE-CR8 CA-L' ),
   ( XcvrMediaType.xcvr400GBaseDr4, '400gbase-dr4',
     'Override the media type to be 400GBASE-DR4' ),
   ( XcvrMediaType.xcvr400GBaseFr4, '400gbase-fr4',
     'Override the media type to be 400GBASE-FR4' ),
   ( XcvrMediaType.xcvr400GBaseCr8S, '400gbase-cr8s',
     'Override the media type to be 400GBASE-CR8 CA-S' ),
]
osfpMediaTypeEnumToClis = { a : b for a, b, _ in osfpMediaTypeCliData }
osfpTokensToHelp = { a : b for _, a, b in osfpMediaTypeCliData }

qsfpDdMediaTypeCliData = [
   ( XcvrMediaType.xcvr200GBaseCr8, '200gbase-cr8',
     'Override the media type to be 200GBASE-CR8 CA-L' ),
   ( XcvrMediaType.xcvr400GBaseCr8N, '400gbase-cr8n',
     'Override the media type to be 400GBASE-CR8 CA-N' ),
   ( XcvrMediaType.xcvr400GBaseCrActive8, '400gbase-cra8',
     'Override the media type to be 400GBASE-CRA8' ),
   ( XcvrMediaType.xcvr200GBaseCr8S, '200gbase-cr8s',
     'Override the media type to be 200GBASE-CR8 CA-S' ),
   ( XcvrMediaType.xcvr400GBaseSr8, '400gbase-sr8',
     'Override the media type to be 400GBASE-SR8' ),
   ( XcvrMediaType.xcvr400GOptical, '400g-optical',
     'Override the media type to be Optical400G' ),
   ( XcvrMediaType.xcvr400GBaseAr8, '400gbase-ar8',
     'Override the media type to be 400GBASE-AR8' ),
   ( XcvrMediaType.xcvr200GBaseCr8N, '200gbase-cr8n',
     'Override the media type to be 200GBASE-CR8 CA-N' ),
   ( XcvrMediaType.xcvr400GBaseXdr4, '400gbase-xdr4',
     'Override the media type to be 400GBASE-XDR4' ),
   ( XcvrMediaType.xcvr400GBaseLr8, '400gbase-lr8',
     'Override the media type to be 400GBASE-LR8' ),
   ( XcvrMediaType.xcvr400GBase2Fr4, '400gbase-2fr4',
     'Override the media type to be 400GBASE-2FR4' ),
   ( XcvrMediaType.xcvr400GBaseCr8, '400gbase-cr8',
     'Override the media type to be 400GBASE-CR8 CA-L' ),
   ( XcvrMediaType.xcvr400GBaseDr4, '400gbase-dr4',
     'Override the media type to be 400GBASE-DR4' ),
   ( XcvrMediaType.xcvr400GBaseFr4, '400gbase-fr4',
     'Override the media type to be 400GBASE-FR4' ),
   ( XcvrMediaType.xcvr400GBaseCr8S, '400gbase-cr8s',
     'Override the media type to be 400GBASE-CR8 CA-S' ),
]
qsfpDdMediaTypeEnumToClis = { a : b for a, b, _ in qsfpDdMediaTypeCliData }
qsfpDdTokensToHelp = { a : b for _, a, b in qsfpDdMediaTypeCliData }

mediaTypeEnumToClis = sfpMediaTypeEnumToClis.copy()
mediaTypeEnumToClis.update( qsfpMediaTypeEnumToClis )
mediaTypeEnumToClis.update( cfp2MediaTypeEnumToClis )
mediaTypeEnumToClis.update( osfpMediaTypeEnumToClis )
mediaTypeEnumToClis.update( qsfpDdMediaTypeEnumToClis )

mediaTypeCliToEnums = { v : k for k, v in mediaTypeEnumToClis.iteritems() }

# Used in the CLI - maps the media type token to the help for that token.
tokensToHelp = sfpTokensToHelp.copy()
tokensToHelp.update( qsfpTokensToHelp )
tokensToHelp.update( cfp2TokensToHelp )
tokensToHelp.update( osfpTokensToHelp )
tokensToHelp.update( qsfpDdTokensToHelp )


nonOverridableMediaTypeEnums = {
   'xcvr10GBaseLrm', 'xcvr10GBaseMfg','xcvr100BaseFx', 
   'xcvr10GBaseCrActive', 'xcvrUnknown', 'xcvr25GUnknown',
   'xcvr100GUnknown', 'xcvr400GCopperUnknown', 'xcvr400GOpticalUnknown',
   'xcvr40GBaseMfg', 'xcvr10GBaseT', 'xcvr5GBaseT', 'xcvr2p5GBaseT',
   'xcvr40GBaseSr4d', 'xcvr100GBaseClr4', 'xcvr100GDwdmDco', 'xcvr400GDwdmDco',
   'xcvr100GBaseMfg', 'xcvr2x100GBaseMfg', 'xcvr400GBaseZr', 'xcvrAmpZr'
   }

# existedMediaTypeEnums contains both overridable and non-overridable mediaType
existedMediaTypeEnums = nonOverridableMediaTypeEnums.copy()
existedMediaTypeEnums.update( mediaTypeEnumToClis.keys() )
