#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, print_function, division
import collections

import Tac

Speeds = Tac.Type( "Inventory::XcvrLaneMgr::PortSpeed" )

class XcvrLaneMgrFdlHelper( object ):
   """The port configuration descriptions can get complicated.  This
   class helps in both generating and reading back in a table of
   interface configurations. The 'genxxx' functions can generate the
   typical table entries for SFP, QSFP, MXP, and OSFP interfaces. The
   'configure' method then parses the table and fills in the inventory
   Tac model. We could have just directly populated the FDL but I
   wanted to keep the intermediate table format to allow unusual
   configurations to be described.

   This is the format expected by the laneMap parameter:
   [
   ( ethPortId,
   [ ( configId1,
   ( [speed1, speed2,...], [subordinateEthPortIds, ...],
   [ ( laneId, slotId, xcvrLane, used ), ... ] ) )
   ( configId2,
   ( [speed1, speed2,...], [subordinateEthPortIds, ...],
   [ ( laneId, slotId, xcvrLane, used ), ... ] ) ),
   ] ),
   ( next ethPort entry ), ...
   ]

   There is also a simpler table that basically encodes the parameters to the
   genxxx functions. See 'parseSimplePortTable' method.
   """

   def __init__( self, ethPortDir, xcvrSlotDir, portLaneConfigDir, laneMap=None ):
      self.ethPortDir = ethPortDir
      self.xcvrSlotDir = xcvrSlotDir
      self.portLaneConfigDir = portLaneConfigDir
      if laneMap:
         self.laneMap = laneMap
      else:
         self.laneMap = []

   @staticmethod
   def compressEthMapping( ethMapping ):
      """Takes an ethMapping of the form:

{
   relativeEthPortId1 : [
      ( port speed0, ( xcvrLane0, ... ), ( subordinate relative ethPortId0, ... ) ),
      ( port speed1, ... ),
      ...
   ],
   relativeEthPortId2 : [ ... ]
   ...
}

      and compresses it so that port speeds that have the same xcvrLane and
      subordinate ethPortId information are grouped.

      Return a format that is the same as ethMapping but with the port speed
      attribute as a list.
      """
      compressedEthMapping = {}
      for ethPortIdBase, speedInfoList in ethMapping.iteritems():
         cfgs = []
         commonSpeedData = collections.defaultdict( list )
         for speedInfo in speedInfoList:
            portSpeed = speedInfo[ 0 ]
            portSpeedData = speedInfo[ 1: ]
            commonSpeedData[ portSpeedData ].append( portSpeed )
         for speedData, portSpeeds in commonSpeedData.iteritems():
            cfgs.append( ( portSpeeds, ) + speedData )
         compressedEthMapping[ ethPortIdBase ] = cfgs
      return compressedEthMapping

   # DO NOT CHANGE ARGUMENTS. If something more complex is needed, please roll your
   # own function.
   def populateBasicEthMapping( self, ethPortIdBase, slotId, ethMapping ):
      """Populates self.laneMap given an ethMapping of the form

{
   relativeEthPortId1 : [
      ( port speed0, [ xcvrLane0, ... ], [ subordinate relative ethPortId0, ... ] ),
      ( port speed1, ... ),
      ...
   ],
   relativeEthPortId2 : [ ... ]
   ...
}

      It is assumed that only one slotId is involved, and ethernet port IDs are
      specified relative to ethPortIdBase.
      """
      for relativeEthPortId, speedInfoList in \
          XcvrLaneMgrFdlHelper.compressEthMapping( ethMapping ).iteritems():
         cfgs = []
         cfgId = 0
         for portSpeeds, xcvrLanes, subEthPortIds in speedInfoList:
            cfgs.append( ( cfgId, portSpeeds,
                           [ ethPortIdBase + subEthPortId
                             for subEthPortId in subEthPortIds ],
                           ( ( intfLane, slotId, xcvrLane, True )
                             for intfLane, xcvrLane in enumerate( xcvrLanes ) ) ) )
            cfgId += 1
         entry = ( ethPortIdBase + relativeEthPortId, cfgs )
         self.laneMap.append( entry )

   # The table data is complicated but there are really only
   # a few patterns. Given that, these functions can be used
   # to generate the full table from a much smaller table.
   # Then the FDL is not polluted unnecessarily.
   def genSfpSlotMap( self, ethPortId, slotId, slotLane=0, has1G=False,
                      has100M=False, has25G=False, has50G=False ):
      """Generates a SFP one lane configuration"""
      cfgs = [ self.gen10GCfg( slotId, 0, slotLane, has1G, has100M ) ]
      if has25G:
         cfgs.append( self.gen25GCfg( slotId, 0, slotLane ) )
      if has50G:
         cfgs.append( self.gen50GSfpCfg( slotId, 0, slotLane ) )
      entry = ( ethPortId, cfgs )
      self.laneMap.append( entry )

   def genQsfpSlotMap( self, ethPortId, slotId, is40gOnly=False, startLane=0,
                       has10G=True, has40G=True, has25G=False, has50G=False,
                       has100G=False, is100gOnly=False ):
      """Generate a 4 lane configuration like we would want for a QSFP slot. QSFP28
      also support 100G and potentially 4x25G and 2x50G as well as 40G and 4x10G, but
      may not be always true, for example optic 100G xcvrs.

      is40gOnly - if True we don't have any subordinate interfaces like the 40G
      only on Cloverdale.

      startLane - allows offsetting within the slot (say a 40G within an MXP)

      has10G - default True, allows 4x10G

      has40G - default True, allows 40G

      has25G - default False, allows 4x25G

      has50G - default False, allows 2x50G

      has100G - default False, allows 100G

      is100gOnly - if True we dont have any subordinate interfaces
      """

      # First create the 40G config and possibly 10G config for the ethPortId
      cfgs = []
      cfgId = 0
      if has40G:
         cfgs.append( self.gen40GCfg( ethPortId, slotId,
                                      not ( is40gOnly or is100gOnly ),
                                      startLane=startLane ) )
         cfgId += 1

      # If this is not 40G only we need to add the 3 10G ports.
      # This assumes that there are 3 sequencial ethPortIds assigned
      # to these 3 ports.
      if not is40gOnly:
         if has100G:
            numSubPorts = 4
            if is100gOnly:
               numSubPorts = 0
            cfgs.append( self.gen100GCfg( ethPortId,
                                          slotId,
                                          4,
                                          numSubPorts=numSubPorts,
                                          cfgId=cfgId ) )
            cfgId += 1
         # Gen sub port speeds at lane 0
         if has10G:
            cfgs.append( self.gen10GCfg( slotId, startLane, cfgId=cfgId ) )
            cfgId += 1
         if has50G:
            cfgs.append( self.gen25GCfg( slotId, startLane, cfgId=cfgId ) )
            cfgId += 1

         # Gen sub port speeds at lanes 2, 3, 4
         for lane in range( 1, 4 ):
            subPortCfgs = []
            subPortCfgId = 0
            if has10G:
               subPortCfgs.append( self.gen10GCfg( slotId,
                                                   slotLane=startLane + lane,
                                                   cfgId=subPortCfgId ) )
               subPortCfgId += 1
            if has25G:
               subPortCfgs.append( self.gen25GCfg( slotId,
                                                   slotLane=startLane + lane,
                                                   cfgId=subPortCfgId ) )
               subPortCfgId += 1
            if has50G and lane == 2:
               subPortCfgs.append( self.gen50GCfg( ethPortId,
                                                   slotId,
                                                   startLane=startLane + lane,
                                                   cfgId=subPortCfgId ) )
               subPortCfgId += 1
            subPortEntry = ( ethPortId + lane, subPortCfgs )
            self.laneMap.append( subPortEntry )

      assert cfgs, \
             "At least one portSpeed needs to be configured for port %s" % ethPortId
      entry = ( ethPortId, cfgs )
      self.laneMap.append( entry )

   def genQsfp100SlotMap( self, ethPortIdBase, slotId, singleLane=False ):
      """Generate a 4 lane configuration like we would want for a QSFP100 slot.
      A QSFP28 slot can support 100G and 4x25G and 2x50G as well as 40G and 4x10G.
      """

      if singleLane:
         qsfp100EthMapping = {
            0 : ( ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), () ),
                  ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), () ), ),
         }
      else:
         qsfp100EthMapping = {
            0 : ( ( Speeds.portSpeed1Gbps, ( 0, ), () ),
                  ( Speeds.portSpeed10Gbps, ( 0, ), () ),
                  ( Speeds.portSpeed25Gbps, ( 0, ), () ),
                  ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
                  ( Speeds.portSpeed50Gbps, ( 0, 1 ), ( 1, ) ),
                  ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ), ),
            1 : ( ( Speeds.portSpeed1Gbps, ( 1, ), () ),
                  ( Speeds.portSpeed10Gbps, ( 1, ), () ),
                  ( Speeds.portSpeed25Gbps, ( 1, ), () ), ),
            2 : ( ( Speeds.portSpeed1Gbps, ( 2, ), () ),
                  ( Speeds.portSpeed10Gbps, ( 2, ), () ),
                  ( Speeds.portSpeed25Gbps, ( 2, ), () ),
                  ( Speeds.portSpeed50Gbps, ( 2, 3 ), ( 3, ) ) ),
            3 : ( ( Speeds.portSpeed1Gbps, ( 3, ), () ),
                  ( Speeds.portSpeed10Gbps, ( 3, ), () ),
                  ( Speeds.portSpeed25Gbps, ( 3, ), () ),
            )
         }

      self.populateBasicEthMapping( ethPortIdBase, slotId, qsfp100EthMapping )

   def genQsfp200SlotMap( self, ethPortIdBase, slotId, singleLane=False,
                          has50G2Lane=True ):
      """Generate a 4 lane configuration like we would want for a QSFP100 slot.
      A QSFP28 slot can support 100G and 4x25G and 2x50G as well as 40G and 4x10G.
      """

      if singleLane:
         qsfp200EthMapping = {
            0 : ( ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), () ),
                  ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), () ),
                  ( Speeds.portSpeed200Gbps4Lane, ( 0, 1, 2, 3 ), () ), ),
         }
      elif not has50G2Lane:
         qsfp200EthMapping = {
            0 : ( ( Speeds.portSpeed10Gbps, ( 0, ), () ),
                  ( Speeds.portSpeed25Gbps, ( 0, ), () ),
                  ( Speeds.portSpeed50Gbps1Lane, ( 0, ), () ),
                  ( Speeds.portSpeed100Gbps2Lane, ( 0, 1 ), ( 1, ) ),
                  ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
                  ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
                  ( Speeds.portSpeed200Gbps4Lane, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ), ),
            1 : ( ( Speeds.portSpeed10Gbps, ( 1, ), () ),
                  ( Speeds.portSpeed25Gbps, ( 1, ), () ),
                  ( Speeds.portSpeed50Gbps1Lane, ( 1, ), () ), ),
            2 : ( ( Speeds.portSpeed10Gbps, ( 2, ), () ),
                  ( Speeds.portSpeed25Gbps, ( 2, ), () ),
                  ( Speeds.portSpeed50Gbps1Lane, ( 2, ), () ),
                  ( Speeds.portSpeed100Gbps2Lane, ( 2, 3 ), ( 3, ) ), ),
            3 : ( ( Speeds.portSpeed10Gbps, ( 3, ), () ),
                  ( Speeds.portSpeed25Gbps, ( 3, ), () ),
                  ( Speeds.portSpeed50Gbps1Lane, ( 3, ), () ), )
         }
      else:
         qsfp200EthMapping = {
            0 : ( ( Speeds.portSpeed10Gbps, ( 0, ), () ),
                  ( Speeds.portSpeed25Gbps, ( 0, ), () ),
                  ( Speeds.portSpeed50Gbps1Lane, ( 0, ), () ),
                  ( Speeds.portSpeed50Gbps, ( 0, 1 ), ( 1, ) ),
                  ( Speeds.portSpeed100Gbps2Lane, ( 0, 1 ), ( 1, ) ),
                  ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
                  ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
                  ( Speeds.portSpeed200Gbps4Lane, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ), ),
            1 : ( ( Speeds.portSpeed10Gbps, ( 1, ), () ),
                  ( Speeds.portSpeed25Gbps, ( 1, ), () ),
                  ( Speeds.portSpeed50Gbps1Lane, ( 1, ), () ), ),
            2 : ( ( Speeds.portSpeed10Gbps, ( 2, ), () ),
                  ( Speeds.portSpeed25Gbps, ( 2, ), () ),
                  ( Speeds.portSpeed50Gbps1Lane, ( 2, ), () ),
                  ( Speeds.portSpeed50Gbps, ( 2, 3 ), ( 3, ) ),
                  ( Speeds.portSpeed100Gbps2Lane, ( 2, 3 ), ( 3, ) ), ),
            3 : ( ( Speeds.portSpeed10Gbps, ( 3, ), () ),
                  ( Speeds.portSpeed25Gbps, ( 3, ), () ),
                  ( Speeds.portSpeed50Gbps1Lane, ( 3, ), () ), )
         }

      self.populateBasicEthMapping( ethPortIdBase, slotId, qsfp200EthMapping )

   def genPam4Qsfp100SlaveSlotMap( self, ethPortIdBase, slotId ):
      """Generate a QSFP100 port that has a conversion from PAM4 to QSFP100 somewhere
      on the NIF, that is slaved to another QSFP100 port.

      This slaved port only supports 50G and 100G, and only has 2 Ethernet
      interfaces.
      """
      pam4Qsfp100EthMapping = {
         0 : ( ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), ( 1, ) ),
               ( Speeds.portSpeed50Gbps, ( 0, 1 ), () ),
               ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), ( 1, ) ), ),
         1 : ( ( Speeds.portSpeed50Gbps, ( 2, 3 ), () ), ),
      }
      self.populateBasicEthMapping( ethPortIdBase, slotId, pam4Qsfp100EthMapping )

   def genMxpSlotMap( self, ethPortId, slotId, has100g=True ):
      """
      Generates an MXP (Arista Multi-speed Port) configuration.
      has100g - if True the first lane can be configured for 100G. The
      T2 boxes can't do 100G.
      """
      cfgs = []

      # First EthPortId has 100G, 40G and 10G
      cfgId = 0
      if has100g:
         cfgs.append(
            self.gen100GCfg( ethPortId, slotId, numLanes=12, numSubPorts=12,
                             cfgId=cfgId, unusedLanes=[0,11] ) )
         cfgId += 1

      # Now add in the 40G and 10G
      for xlge in range( 0, 3 ):
         startLane = 4 * xlge
         cfgs.append( self.gen40GCfg( ethPortId, slotId, True, cfgId=cfgId,
                                      startLane=startLane ) )
         cfgId += 1
         cfgs.append( self.gen10GCfg( slotId, slotLane=startLane, cfgId=cfgId ) )
         # Add the above configs for the first ethPortId
         self.laneMap.append( ( ethPortId, cfgs ) )
         cfgs = []

         # Move on to the next ethPort
         ethPortId += 1
         cfgId = 0
         # Then we generate the 3 10G only lanes.
         for x in range( 1, 4 ):
            cfgs = [ self.gen10GCfg( slotId, slotLane=startLane+x, cfgId=cfgId ) ]
            self.laneMap.append( ( ethPortId, cfgs ) )
            cfgs = []
            ethPortId += 1

   def genRj45SlotMap( self, ethPortId, slotId, slotLane=0, is10gOnly=False ):
      cfgs = []
      if is10gOnly:
         cfgs.append( self.gen10GCfg( slotId, 0, slotLane ) )
      else:
         cfgs.append( self.gen10MCfg( slotId, 0, slotLane ) )
         cfgs.append( self.gen5GCfg( slotId, 0, slotLane ) )
         cfgs.append( self.gen10GCfg( slotId, 0, slotLane, True, True ) )
      entry = ( ethPortId, cfgs )
      self.laneMap.append( entry )

   def gen8LaneCfg( self, ethPortId, slotId, cfgId, numSubPorts=0 ):
      speedCfg = [ Speeds.portSpeed400Gbps8Lane, Speeds.portSpeed200Gbps8Lane ]
      if numSubPorts:
         subPorts = [ x for x in range( ethPortId + 1, ethPortId + numSubPorts ) ]
      else:
         subPorts = []

      # [ ( laneId, slotId, xcvrLane, used ), ... ] ) )
      laneCfg = [ ( x, slotId, x, True ) for x in range( 0, 8 ) ]
      return [ ( cfgId, speedCfg, subPorts, laneCfg ) ]

   def gen4LaneCfg( self, ethPortId, slotId, cfgId, numSubPorts=0, startLane=0 ):
      # For now, we put all speeds in a single list.
      # If required, we can build different functions in case speed configurations
      # turn out to have different lane maps.
      # All generated configs then can be combined into a single list and
      # returned from this function
      speedCfg = [ Speeds.portSpeed200Gbps4Lane, Speeds.portSpeed100Gbps,
                   Speeds.portSpeed40Gbps ]
      if numSubPorts:
         subPorts = [ x for x in range( ethPortId + 1, ethPortId + numSubPorts ) ]
      else:
         subPorts = []

      # [ ( laneId, slotId, xcvrLane, used ), ... ] ) )
      laneCfg = [ ( x, slotId, x + startLane, True ) for x in range( 0, 4 ) ]
      return [ ( cfgId, speedCfg, subPorts, laneCfg ) ]

   def gen2LaneCfg( self, ethPortId, slotId, cfgId, numSubPorts=0, startLane=0 ):
      # For now, we put all speeds in a single list.
      # If required, we can build different functions in case speed configurations
      # turn out to have different lane maps.
      # All generated configs then can be combined into a single list and
      # returned from this function
      speedCfg = [ Speeds.portSpeed100Gbps2Lane, Speeds.portSpeed50Gbps ]
      if numSubPorts:
         subPorts = [ x for x in range( ethPortId + 1, ethPortId + numSubPorts ) ]
      else:
         subPorts = []

      # [ ( laneId, slotId, xcvrLane, used ), ... ] ) )
      laneCfg = [ ( x, slotId, x + startLane, True ) for x in range( 0, 2 ) ]
      return [ ( cfgId, speedCfg, subPorts, laneCfg ) ]

   def gen1LaneCfg( self, slotId, cfgId, slotLane=0, has1G=False, has100M=False ):
      # For now, we put all speeds in a single list.
      # If required, we can build different functions in case speed configurations
      # turn out to have different lane maps.
      # All generated configs then can be combined into a single list and
      # returned from this function
      speedCfg = [ Speeds.portSpeed50Gbps1Lane, Speeds.portSpeed25Gbps,
                   Speeds.portSpeed10Gbps ]
      if has1G:
         speedCfg.append( Speeds.portSpeed1Gbps )
      if has100M:
         speedCfg.append( Speeds.portSpeed100Mbps )

      subPorts = []
      # [ ( laneId, slotId, xcvrLane, used ), ... ] ) )
      laneCfg = [ ( 0, slotId, slotLane, True ) ]
      return [ ( cfgId, speedCfg, subPorts, laneCfg ) ]

   def genCfp2SlotMap( self, ethPortIdBase, slotId ):
      cfp2EthMapping = {
         0 : ( ( Speeds.portSpeed10Gbps, ( 0, ), () ),
               ( Speeds.portSpeed25Gbps, ( 0, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 0, ), () ),
               ( Speeds.portSpeed50Gbps, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed200Gbps4Lane, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ), ),
         1 : ( ( Speeds.portSpeed10Gbps, ( 1, ), () ),
               ( Speeds.portSpeed25Gbps, ( 1, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 1, ), () ), ),
         2 : ( ( Speeds.portSpeed10Gbps, ( 2, ), () ),
               ( Speeds.portSpeed25Gbps, ( 2, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 2, ), () ),
               ( Speeds.portSpeed50Gbps, ( 2, 3 ), ( 3, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 2, 3 ), ( 3, ) ), ),
         3 : ( ( Speeds.portSpeed10Gbps, ( 3, ), () ),
               ( Speeds.portSpeed25Gbps, ( 3, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 3, ), () ), )
      }

      self.populateBasicEthMapping( ethPortIdBase, slotId, cfp2EthMapping )

   # OSFPs have a new model for generating lane configurations.
   # The idea is to collapse multiple same number of lanes configurations into
   # one if we can. This avoids an explosion of the number of cfgIds for
   # identical configurations such as 10G and 25G, or 50G and 2 lane 100G.
   def genOsfpSlotMap( self, ethPortIdBase, slotId, is400gOnly=False,
                       singleLane=False ):
      """Generate a 8 lane configuration like we would want for a OSFP or QSFP-DD
      slot.
      """
      osfpEthMapping = {
         0 : ( ( Speeds.portSpeed10Gbps, ( 0, ), () ),
               ( Speeds.portSpeed25Gbps, ( 0, ), () ),
               ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed50Gbps1Lane, ( 0, ), () ),
               ( Speeds.portSpeed50Gbps, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed200Gbps4Lane, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed400Gbps8Lane, ( 0, 1, 2, 3, 4, 5, 6, 7 ),
                 ( 1, 2, 3, 4, 5, 6, 7 ) ) ),
         1 : ( ( Speeds.portSpeed10Gbps, ( 1, ), () ),
               ( Speeds.portSpeed25Gbps, ( 1, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 1, ), () ), ),
         2 : ( ( Speeds.portSpeed10Gbps, ( 2, ), () ),
               ( Speeds.portSpeed25Gbps, ( 2, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 2, ), () ),
               ( Speeds.portSpeed50Gbps, ( 2, 3 ), ( 3, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 2, 3 ), ( 3, ) ), ),
         3 : ( ( Speeds.portSpeed10Gbps, ( 3, ), () ),
               ( Speeds.portSpeed25Gbps, ( 3, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 3, ), () ), ),
         4 : ( ( Speeds.portSpeed10Gbps, ( 4, ), () ),
               ( Speeds.portSpeed25Gbps, ( 4, ), () ),
               ( Speeds.portSpeed40Gbps, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ),
               ( Speeds.portSpeed50Gbps1Lane, ( 4, ), () ),
               ( Speeds.portSpeed50Gbps, ( 4, 5 ), ( 5, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 4, 5 ), ( 5, ) ),
               ( Speeds.portSpeed100Gbps, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ),
               ( Speeds.portSpeed200Gbps4Lane, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ), ),
         5 : ( ( Speeds.portSpeed10Gbps, ( 5, ), () ),
               ( Speeds.portSpeed25Gbps, ( 5, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 5, ), () ), ),
         6 : ( ( Speeds.portSpeed10Gbps, ( 6, ), () ),
               ( Speeds.portSpeed25Gbps, ( 6, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 6, ), () ),
               ( Speeds.portSpeed50Gbps, ( 6, 7 ), ( 7, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 6, 7 ), ( 7, ) ), ),
         7 : ( ( Speeds.portSpeed10Gbps, ( 7, ), () ),
               ( Speeds.portSpeed25Gbps, ( 7, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 7, ), () ), ),
      }

      if singleLane:
         osfpEthMapping = {
            0 : ( ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), () ),
                  ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), () ),
                  ( Speeds.portSpeed200Gbps4Lane, ( 0, 1, 2, 3 ), () ),
                  ( Speeds.portSpeed400Gbps8Lane, ( 0, 1, 2, 3, 4, 5, 6, 7 ),
                    () ) ),
         }
      elif is400gOnly:
         osfpEthMapping = {
            0 : ( ( Speeds.portSpeed400Gbps8Lane, ( 0, 1, 2, 3, 4, 5, 6, 7 ),
                    () ), ),
         }

      self.populateBasicEthMapping( ethPortIdBase, slotId, osfpEthMapping )

   # Could have
   # MXP port with 12 lanes and 2 not used.
   # CFP2 4 lanes all used
   # CFP2 10 lanes all used at 100G and can be used at 10x10G.
   # QSFP28 4 lanes at 100G, or 4 lanes at 40G or 4x10G
   def gen100GCfg( self, ethPortId, slotId, numLanes, numSubPorts=0,
                   cfgId=0,
                   startLane=0,
                   unusedLanes=[] ):

      speedCfg = [ Speeds.portSpeed100Gbps ]
      if numSubPorts:
         subPorts = [ x for x in range( ethPortId + 1, ethPortId + numSubPorts ) ]
      else:
         subPorts = []
      # Init to all lanes used...
      laneUsed = [ True for x in range( 0, numLanes ) ]
      # Then mark some as unused
      for lane in unusedLanes:
         laneUsed[ lane ] = False

      laneCfg = [ ( x, slotId, x + startLane, laneUsed[ x ] )
                  for x in range( 0, numLanes ) ]
      return ( cfgId, speedCfg , subPorts, laneCfg )

   # QSFP28 2 lanes at 50G
   def gen50GCfg( self, ethPortId, slotId, genSubPorts=True, cfgId=0,
                  startLane=0 ):
      speedCfg = [ Speeds.portSpeed50Gbps ]
      if genSubPorts:
         subPorts = [ x for x in range( ethPortId + 1, ethPortId + 2 ) ]
      else:
         subPorts = []
      laneCfg = [ ( x, slotId, x + startLane, True ) for x in range( 0, 2 ) ]
      return ( cfgId, speedCfg , subPorts, laneCfg )

   def gen40GCfg( self, ethPortId, slotId, genSubPorts=True, cfgId=0,
                  startLane=0 ):
      """
      Generates one 40G config entry assuming that the lanes are sequencial
      """
      speedCfg = [ Speeds.portSpeed40Gbps ]
      if genSubPorts:
         subPorts = [ x for x in range( ethPortId + 1, ethPortId + 4 ) ]
      else:
         subPorts = []
      laneCfg = [ ( x, slotId, x + startLane, True ) for x in range( 0, 4 ) ]
      return ( cfgId, speedCfg , subPorts, laneCfg )

   def gen50GSfpCfg( self, slotId, slotLane=0, cfgId=0 ):
      speedCfg = [ Speeds.portSpeed50Gbps1Lane ]
      subPorts = []
      laneCfg = [ ( 0, slotId, slotLane, True ), ]
      return ( cfgId, speedCfg, subPorts, laneCfg )

   def gen25GCfg( self, slotId, slotLane=0, cfgId=0 ):
      speedCfg = [  Speeds.portSpeed25Gbps ]
      subPorts = []
      laneCfg = [ ( 0, slotId, slotLane, True ), ]
      return ( cfgId, speedCfg , subPorts, laneCfg )

   def gen10GCfg( self, slotId, slotLane=0, cfgId=0, has1G=False, has100M=False ):
      speedCfg = [  Speeds.portSpeed10Gbps ]
      if has1G:
         speedCfg.append( Speeds.portSpeed1Gbps )
      if has100M:
         speedCfg.append( Speeds.portSpeed100Mbps )

      subPorts = [ ]
      laneCfg = [ ( 0, slotId, slotLane, True ), ]
      return ( cfgId, speedCfg , subPorts, laneCfg )

   def gen5GCfg( self, slotId, slotLane=0, cfgId=0 ):
      speedCfg = [ Speeds.portSpeed2p5Gbps, Speeds.portSpeed5Gbps ]

      subPorts = []
      laneCfg = [ ( 0, slotId, slotLane, True ), ]
      return ( cfgId, speedCfg , subPorts, laneCfg )

   def gen10MCfg( self, slotId, slotLane=0, cfgId=0 ):
      speedCfg = [ Speeds.portSpeed10Mbps ]

      subPorts = []
      laneCfg = [ ( 0, slotId, slotLane, True ), ]
      return ( cfgId, speedCfg, subPorts, laneCfg )

   def parseSimplePortTable( self, portTable ):
      """
      Parses a mimimalistic table which can be used to
      configure most ports in a system using this helper
      class.
      [ ( type, { kwargs } ), ... ]
      # where type is 'mxp', 'sfp', 'qsfp'

      # WARNING: only osfp and mxp are currently supported
      """
      for portParams in portTable:
         if portParams[0] == 'mxp':
            self.genMxpSlotMap( **portParams[1] )
         elif portParams[0] == 'qsfp':
            self.genQsfpSlotMap( **portParams[ 1 ] )
         elif portParams[ 0 ] == 'qsfp200':
            self.genQsfp200SlotMap( **portParams[ 1 ] )
         elif portParams[ 0 ] == 'qsfp100':
            self.genQsfp100SlotMap( **portParams[ 1 ] )
         elif portParams[ 0 ] == 'pam4Qsfp100Slave':
            self.genPam4Qsfp100SlaveSlotMap( **portParams[ 1 ] )
         elif portParams[0] == 'sfp':
            self.genSfpSlotMap( **portParams[1] )
         elif portParams[0] in ( 'osfp', 'qsfpDd', ):
            self.genOsfpSlotMap( **portParams[1] )
         elif portParams[0] == 'rj45':
            self.genRj45SlotMap( **portParams[1] )
         elif portParams[ 0 ] == 'cfp2':
            self.genCfp2SlotMap( **portParams[ 1 ] )
         else:
            assert( 0 )

   def configure( self ):
      """
      Parses the contents of self.laneMap and populates the FDL.
      """
      for ethPortId, configs in self.laneMap:
         for config in configs:
            configId, speeds, subordinatePorts, laneCfgs = config
            portLaneConfig = self.portLaneConfigDir.newPortLaneConfig( ethPortId )
            portLaneConfig.port = self.ethPortDir.port[ ethPortId ]
            portSpeedConfig = portLaneConfig.newPortSpeedConfig( configId )

            for speed in speeds:
               portSpeedConfig.supportedSpeed[ speed ] = True

            for subPortId in subordinatePorts:
               portSpeedConfig.subordinatePort[ subPortId ] = \
                   self.ethPortDir.port[ subPortId ]

            for (ethLane, slotId, xcvrLane, used) in laneCfgs:
               slot = self.xcvrSlotDir.xcvrSlot[ slotId ]
               portXcvrConfig = portSpeedConfig.newPortXcvrConfig(
                  ethLane, slot, xcvrLane, used )
