#!/usr/bin/python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
import Tac
import CliMatcher
import MultiRangeRule
from CliParser import Completion
from CliParserCommon import MatchResult, noMatch, ParseError, safeInt

# pylint: disable-msg=W1401

maxVni = Tac.Type( 'Vxlan::VniExt' ).max
invalidVni = Tac.Type( 'Vxlan::VniExtOrNone' ).invalidVni

class VniDottedDoParse( MultiRangeRule.DoParse ):
   '''
   This do parse class is intended for a range of VNIs as opposed to
   ASNs like the original DoParse. To accomplish this, we implemented our
   own matchNumber and convertToNumber method with our own regex.

   Use this class as a parameter in MultiRangeMatcher to match on
   multiple VNIs. dottedRange should be set to true if this class
   is being used.
   '''

   def __init__( self, mode, buff, noSingletons, explicitIntfTypes,
                 genericRange=False, dottedRange=False, maxRanges=0,
                 noHoleRange=False, dollarCompHelpText=None,
                 mergeRange=False ):
      self.mergeRange_ = mergeRange
      super( VniDottedDoParse, self ).__init__( mode, buff, noSingletons,
            explicitIntfTypes, genericRange, dottedRange, maxRanges, noHoleRange,
            dollarCompHelpText, mergeRange )


   numberDottedRe_ = re.compile( r"([0-9]+\.){0,3}[0-9]*" )

   def matchNumber( self ):
      '''Specifies the criteria/regex to match for a number'''
      return self._need( self.numberDottedRe_ )

   def convertToNumber( self, soFar ):
      '''Specifies the criteria/regex to convert the matched string
         to an integer value'''
      if '.' not in soFar:
         soFar = safeInt( soFar )
      else:
         soFarSpl = soFar.split( '.' )
         numSplit = len( soFarSpl )
         if numSplit < 3:
            raise ParseError( "%r invalid format" % soFar )
         else:
            val = 0
            for s in soFarSpl:
               if s == '':
                  raise ParseError( "%r invalid format" % soFar )
               tmp = safeInt( s )
               val = val << 8 | tmp
            soFar = val
      return soFar

# Better a free function than a @static method of VniFormat class
def vniToString( vniNum, dottedNotation ):
   # blind conversion from string or int
   vniNum = int( vniNum )
   if vniNum == invalidVni:
      return 'invalid'
   elif dottedNotation:
      if vniNum & 0xFF000000 == 0:
         return "{}.{}.{}".format( ( vniNum >> 16 & 0xFF ),
                                   ( vniNum >> 8 & 0xFF ),
                                   ( vniNum & 0xFF ) )
      else:
         return "{}.{}.{}.{}".format( ( vniNum >> 24 & 0xFF ),
                                      ( vniNum >> 16 & 0xFF ),
                                      ( vniNum >> 8 & 0xFF ),
                                      ( vniNum & 0xFF ) )
   else:
      return str( vniNum )
      
# Convert Vni ranges to a string representation. vniRanges is a list of
# VniRange() objects
def vniRangesToString( vniRanges, dottedNotation ):
   sList = []
   for vniRange in vniRanges:
      if vniRange.minVni != vniRange.maxVni:
         vniString = ( vniToString( vniRange.minVni, dottedNotation ) +
                       ( "--" if dottedNotation else "-" ) +
                       vniToString( vniRange.maxVni, dottedNotation ) )
      else:
         vniString = vniToString( vniRange.minVni, dottedNotation ) 
      sList.append( vniString )
   return ", ".join( sList )
   
# Convert Vni ranges to list of Vnis. vniRanges is a list of VniRange() objects
def vniRangesToVnis( vniRanges ):
   vnis = []
   for vniRange in vniRanges:
      vnis += range( vniRange.minVni, vniRange.maxVni + 1 )
   return vnis
   
class VniFormat( object ):
   vni24DottedRe_ = re.compile( '(\d+)\.(\d+)\.(\d+)$' )
   vni32DottedRe_ = re.compile( '(\d+)\.(\d+)\.(\d+)\.(\d+)$' ) 

   def __init__( self, vniNum, dottedNotation=False ):
      self.dottedNotation = dottedNotation
      if type( vniNum ) in [ int, long ]:
         self.vniNum = vniNum
      elif type( vniNum ) == str:
         vniNum = vniNum.strip()
         if vniNum.isdigit():
            self.vniNum = int( vniNum )
         else:
            # Match 24-bit dotted notation
            m = VniFormat.vni24DottedRe_.match( vniNum )
            if m:
               self.vniNum = ( int( m.group( 1 ) ) << 16 ) | \
                  ( int( m.group( 2 ) ) << 8 ) | \
                  int( m.group( 3 ) )
            else:
               # Match 32-bit dotted notation
               m = VniFormat.vni32DottedRe_.match( vniNum )
               if m:
                  self.vniNum = ( int( m.group( 1 ) ) << 24 ) | \
                     ( int( m.group( 2 ) ) << 16 ) | \
                     ( int( m.group( 3 ) ) << 8 ) | \
                     int( m.group( 4 ) )
               elif vniNum == 'invalid':
                  self.vniNum = invalidVni
               else:
                  raise ValueError( '%s is not a valid VNI' % vniNum )
      else:
         self.vniNum = invalidVni

   def __str__( self ):
      return vniToString( self.vniNum, self.dottedNotation )

   def __lt__( self, other ):
      return self.vniNum < other.vniNum

   def toNum( self ):
      return self.vniNum

   def isValid( self ):
      # invalidVni is outside the range from 0 - maxVni
      return 0 < self.vniNum < invalidVni

class VniMatcher( CliMatcher.Matcher ):
   '''Match VNI in dotted format or 24-bit/32-bit number'''

   # Regex for VNI completion
   vniCompletionRe_ = re.compile( '(\d+)(\.(\d+)?){0,3}?$' )

   def __init__( self, helpdesc, vxlanCntrlCfgBox, **kwargs ):
      super( VniMatcher, self ).__init__( helpdesc=helpdesc, **kwargs )
      self.completionDotted_ = Completion( '   A.B.C.D', helpdesc, False )
      self.completion_ = Completion( '   <1-4294967294>', helpdesc, False )
      self.vxlanCntrlCfgBox = vxlanCntrlCfgBox

   def match( self, mode, context, token ):
      try:
         vni = VniFormat( token )
      except ValueError:
         return noMatch
      # Validate VNI range
      if not vni.isValid():
         return noMatch
      result = str( vni ) 
      return MatchResult( result, result )

   def completions( self, mode, context, token ):
      # Ignore the completion if vxlanCntrlCfgBox is not populated yet
      if not self.vxlanCntrlCfgBox:
         return []

      # Display completion help for dotted notation when configured
      if self.vxlanCntrlCfgBox[ 0 ].vniInDottedNotation:
         completion = self.completionDotted_
      else:
         completion = self.completion_

      if not token:
         return [ completion ]

      m = VniMatcher.vniCompletionRe_.match( token )
      if m is None:
         return []
      # If matched string can construct VNI, validate that VNI is valid before 
      # returning completion
      try:
         vni = VniFormat( m.group( 0 ) )
         if not vni.isValid():
            return []
      except ValueError:
         pass
      return [ completion ]

   def __str__( self ):
      return '<Vni>'
