#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from socket import AF_INET6, AF_INET
import Tac, PyClient, AgentDirectory, Arnet
from Arnet.Verify import VerifyException
from IpLibConsts import DEFAULT_VRF
from FibUtils import mountSmashTables, unmountSmashTables
from KernelFibUtils import kernelIpRoutes

class Verifier:
   def __init__( self, entityManager ):
      sysname = AgentDirectory.agentList()[ 0 ][ 'system' ]
      self.entityManager = entityManager
      self.sysdb = PyClient.PyClient( sysname, 'Sysdb' ).agentRoot()
      self.kfRoot = PyClient.PyClient( sysname, 'KernelFib' ).root()[ sysname ]
      self.ipVrfStatus = self.sysdb.entity.get( "ip/vrf/status/global" )
      self.intfStatusKernel = self.kfRoot.entity.get(
         "localAgentPlugin/interface/status/all" )
      self.vrfNames = [ DEFAULT_VRF ]
      for vrf in self.ipVrfStatus.vrf:
         self.vrfNames.append( vrf )
      self.fibRoutingStatus = {}
      self.fibRouting6Status = {}
      self.forwardingStatus = {}
      self.forwarding6Status = {}
      self.routeV4DataInKernel = {}
      self.routeV6DataInKernel = {}
      self.mountSmashTables()

   def mountSmashTables( self ):
      for vrfName in self.vrfNames:
         ( self.fibRoutingStatus[ vrfName ],
               self.forwardingStatus[ vrfName ] ) = \
            mountSmashTables( self.entityManager, vrfName=vrfName,
                              ipVersion=4 )
         ( self.fibRouting6Status[ vrfName ],
               self.forwarding6Status[ vrfName ] ) = \
            mountSmashTables( self.entityManager, vrfName=vrfName,
                              ipVersion=6 )
   
   def unmountSmashTables( self ):
      for vrfName in self.vrfNames:
         unmountSmashTables( self.entityManager, vrfName=vrfName,
                             ipVersion=4 )
         unmountSmashTables( self.entityManager, vrfName=vrfName,
                           ipVersion=6 )

   def getRouteDataInKernel( self, af=AF_INET ):
      routeDataInKernel = {}
      ipv6 = ( af == AF_INET6 )
      for vrfName in self.vrfNames:
         routeDataInKernel[ vrfName ] = kernelIpRoutes( vrfName, ipv6=ipv6,
                                                        getMetric=True )
      return routeDataInKernel

   def cleanup( self ):
      self.sysdb = None
      self.kfRoot = None
      self.ipVrfStatus = None
      self.intfStatusKernel = None
      self.unmountSmashTables()
      self.fibRoutingStatus = {}
      self.forwardingStatus = {}
      self.fibRouting6Status = {}
      self.forwarding6Status = {}
      self.routeV4DataInKernel = {}
      self.routeV6DataInKernel = {}
      self.forwardingStatus = None

   def verifyRouteInKernel( self, prefix, vrf=DEFAULT_VRF, af=AF_INET ):
      fingerprint = ''
      data = ''
      routeDataInSmash = {}
      if af == AF_INET:
         fibRoutingStatus = self.fibRoutingStatus[ vrf ]
         forwardingStatus = self.forwardingStatus[ vrf ]
         routeDataInKernel = self.routeV4DataInKernel[ vrf ]
      else:
         fibRoutingStatus = self.fibRouting6Status[ vrf ]
         forwardingStatus = self.forwarding6Status[ vrf ]
         routeDataInKernel = self.routeV6DataInKernel[ vrf ]
         
      def routeTypeToProto( routeType ):
         ignoredRouteTypes = [ 'invalid', 'receive', 'receiveBcast',
                               'attached' ]
         if routeType == 'kernel' or routeType == 'connected':
            proto = 'kernel'
         elif routeType not in ignoredRouteTypes:
            proto = 'gated'
         else:
            proto = 'ignored'
         return proto
      
      route = fibRoutingStatus.route[ prefix ]
      proto = routeTypeToProto( route.routeType )
      
      # Return for ignored route types as KernelFib doesn't program them
      if proto == 'ignored':
         return ( fingerprint, data )

      unprogrammedPrefixes = [ "0.0.0.0/8", "::/96", "::1/128" ]
      if str( prefix ) in unprogrammedPrefixes:
         return ( fingerprint, data )

      found = str( prefix ) in routeDataInKernel.keys()
      
      # Verify if the prefix was after all programmed in kernel
      if not found:
         fingerprint = '%s was not programmed in Kernel in vrf %s'
         data = ( prefix.stringValue, vrf )
         return ( fingerprint, data )

      routeDataInSmash[ 'proto' ] = proto
      if route.metric == 0 and af == AF_INET6:
         routeDataInSmash[ 'metric' ] = 1024
      else:
         routeDataInSmash[ 'metric' ] = route.metric

      ( hop, dev, proto, metric ) = routeDataInKernel[ str( prefix ) ][ 0 ]
      
      # Modify hop, dev and metric found from the output of ip route show to match
      # the one found in smash
      if not hop:
         hop = "0.0.0.0" if af == AF_INET else None
      if dev == "lo":
         dev = None
      if metric:
         metric = int( metric )
      else:
         metric = 0

      # Verify that correct via is programmed by KernelFib.
      viaFound = True
      fecId = route.fecId
      fec = forwardingStatus.fec.get( fecId )
      for viaId in fec.via:
         viaFound = False
         via = fec.via.get( viaId )
         viaData = { 'dev': None, 'hop': None }
         if via.intfId:
            viaData[ 'dev' ] = self.intfStatusKernel[ via.intfId ].deviceName
         if via.hop:
            viaData[ 'hop' ] = str( via.hop )
         if ( viaData[ 'dev' ], viaData[ 'hop' ] ) == ( dev, hop ):
            viaFound = True
            break
      if not viaFound:
         fingerprint = '%s programmed in kernel for %s in vrf %s is incorrect'
         data = ( 'via', str( prefix ) , vrf )
      
      # Verify other route properties i.e. proto and metric
      if routeDataInSmash[ 'proto' ] != proto:
         # If the routeDataInSmash[ 'proto' ] == kernel, it would imply that 
         # the route was learnt from kernel (kernel itself could have learnt the
         # route from bash/ bird or any other source). Hence the kernel can have
         # a routetype None (bash added), bird etc, where as EOS does not know
         # this. Hence for routes learnt from kernel just check that the source 
         # is not gated in kernel.
         if routeDataInSmash[ 'proto' ] == 'kernel':
            if proto == 'gated':
               fingerprint = '%s programmed in kernel for %s in vrf %s is incorrect'
               data = ( 'route protocol', str( prefix ), vrf )
               return ( fingerprint, data )
         else:
            fingerprint = '%s programmed in kernel for %s in vrf %s is incorrect'
            data = ( 'route protocol', str( prefix ), vrf )
            return ( fingerprint, data )
      
      if routeDataInSmash[ 'metric' ] != metric:
         fingerprint = '%s programmed in kernel for %s in vrf %s is incorrect'
         data = ( 'metric', str( prefix ), vrf )

      return ( fingerprint, data )

   def verifyPrefixInVrf( self, prefix, vrf=DEFAULT_VRF, af=AF_INET ):
      fibRoutingStatus = self.fibRoutingStatus if af == AF_INET \
            else self.fibRouting6Status
      if prefix in fibRoutingStatus[ vrf ].route:
         ( fingerprint, data ) = self.verifyRouteInKernel( prefix, vrf=vrf, af=af )
         if fingerprint != '':
            raise VerifyException( fingerprint, data )

   def verify( self, routeAndVrf ):
      route = routeAndVrf[ 0 ]
      vrf = routeAndVrf[ 1 ]
      if ":" in route:
         if not len( self.routeV6DataInKernel ):
            self.routeV6DataInKernel = \
                     self.getRouteDataInKernel( af=AF_INET6 )
         prefix = Arnet.Ip6Prefix( route )
         af = AF_INET6
      else:
         if not len( self.routeV4DataInKernel ):
            self.routeV4DataInKernel = \
                        self.getRouteDataInKernel( af=AF_INET )
         prefix = Arnet.Prefix( route )
         af = AF_INET
      self.verifyPrefixInVrf( prefix, vrf=vrf, af=af )
