#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, PyClient, AgentDirectory, SmashLazyMount
import PyClientBase
from Arnet.NsLib import runMaybeInNetNs, DEFAULT_NS
from Arnet.Verify import VerifyException, NotApplicableException, runningAgents
import Arnet, re
from IpLibConsts import DEFAULT_VRF

NEIGHBOR_VERIFY_MAX = 100 
class Verifier( object ):
   def __init__( self, entityManager ):
      if 'Arp' not in runningAgents():
         raise NotApplicableException
      sysname = AgentDirectory.agentList()[ 0 ][ 'system' ]
      self.entityManager = entityManager
      self.sysdb = PyClient.PyClient( sysname, 'Sysdb' ).agentRoot()
      self.arpRoot = PyClient.PyClient( sysname, 'Arp' ).root()[ sysname ]
      self.allIntfStatusDir = self.arpRoot.entity.get(
         "localAgentPlugin/interface/status/all" )
      self.staticArpConfigDir = None 
      self.staticArpStatusDir = None
      self.arpSmash_ = SmashLazyMount.mount( entityManager, 'arp/status',
            'Arp::Table::Status', SmashLazyMount.mountInfo( 'reader' ) )
      self.vrfSmash_ = SmashLazyMount.mount( entityManager, 'vrf/vrfIdMapStatus',
            'Vrf::VrfIdMap::Status', SmashLazyMount.mountInfo( 'reader' ) )
      self.allVrfConfig = self.sysdb.entity.get( "ip/vrf/config" )
      self.vrfNameToId = {}
      for vrfId, vrfEntry in self.vrfSmash_.vrfIdToName.iteritems():
         self.vrfNameToId[ vrfEntry.vrfName ] = vrfId

   def cleanup( self ):
      self.sysdb = None
      self.arpRoot = None
      self.allIntfStatusDir = None 
      self.staticArpConfigDir = None 
      self.staticArpStatusDir = None
      self.allVrfConfig = None
      self.arpSmash_ = None
      self.vrfSmash_ = None 
      self.vrfNameToId = None

   def readKernelIp6NeighborTable( self, netNs=DEFAULT_NS ):
      # Build a dict containing the kernel's ipv6 Neighbor table.
      karp = {}

      arpRe = re.compile( 
             r"^([A-Fa-f0-9:.]{0,46})\s+\S+\s+(\S+)\s+\S+\s+(\S+)\s+(router\s+)*" +
             r"(\S+)$" )

      # store just the ip addresses that we have seen
      ips = {}
      output = runMaybeInNetNs( netNs, [ "ip", "-6", "neigh", "show" ],
                                stdout=Tac.CAPTURE, asRoot=True )

      lines = output.strip().split( '\n' )

      for line in lines:
         m = arpRe.match( line )
         if m:
            ip = m.group( 1 )
            deviceName = m.group( 2 )
            eth = m.group( 3 )
            static = bool( m.group( 4 ) == "PERMANENT" )
         
            karp.setdefault( deviceName, {} )[ ip ] = ( eth, static )
            ips[ ip ] = ips.get( ip, [] ) + [ deviceName ]
      return karp, ips

   def arpSmashToNeighborDir( self ):
      '''
      Convert arp smash into Arp::NeighborDir.

      Return None if arpSmash is None.
      '''
      if not self.arpSmash_:
         return None

      nd = Tac.newInstance( "Arp::NeighborDir", "temp" )
      for entry in self.arpSmash_.neighborEntry.itervalues():
         intfName = entry.intfId
         if not intfName in nd.arpIntfStatus:
            nd.arpIntfStatus.newMember( intfName )
         ais = nd.arpIntfStatus[ intfName ]
         neighborEntry = Tac.newInstance( "Arp::NeighborEntryStatus",
               entry.addr.v6Addr, entry.ethAddr, entry.isStatic )
         neighborEntry.source = entry.source
         ais.ip6NeighborEntry.addMember( neighborEntry )
      return nd

   def compareIp6NeighborsAgainstKernel( self, neighborDirByIntfName,
                                        intfStatusKernel,
                                        netNs=DEFAULT_NS ):

      karp, _ = self.readKernelIp6NeighborTable( netNs )
      d = neighborDirByIntfName
      if d is None:
         return

      for key in d.arpIntfStatus:
         deviceName = intfStatusKernel[ key ].deviceName
         if 'Management' in key or 'Internal' in key:
            karp.pop( deviceName, None )
            continue
         arpIntfStatus = d.arpIntfStatus[ key ]
         neighEntries = arpIntfStatus.ip6NeighborEntry

         if deviceName not in karp:
            continue
      
         karpDeviceNameDict = karp[ deviceName ]

         for ipAddr, neighEntry in neighEntries.iteritems():
            if ipAddr.stringValue not in karpDeviceNameDict:
               message = ( "dir contains an ip address for %s "
                           "that is not in the "
                           "kernel neigh table: '%s'" % ( deviceName, ipAddr ) )
               fingerprint = message
               data = ipAddr
               raise VerifyException( fingerprint, data )

            ( ethAddr, _ ) = karpDeviceNameDict[ ipAddr.stringValue ]
            if ethAddr != neighEntry.ethAddr:
               fingerprint = "Mismatch MAC address %s %s" % ( ethAddr, \
                                                              neighEntry.ethAddr )
               data = ipAddr
               raise VerifyException( fingerprint, data )
         
            # remove this ip address so only those that aren't tracked will
            # be in the dictionary
            #print 'Verified neighbor %s in kernel' % ipAddr
            del karpDeviceNameDict[ ipAddr.stringValue ]
      
         # scrub invalid entries that the kernel still shows
         for ipAddr, ( ethAddr, _ ) in karpDeviceNameDict.items():
            if ethAddr == '00:00:00:00:00:00':
               del karpDeviceNameDict[ ipAddr ]
         
         if len( karpDeviceNameDict ) != 0:
            # there were some arp entries that we did not notice for this device
            message = ( "dir for %s does not contain neigh entries "
                        "for %s" % ( deviceName, karpDeviceNameDict.items() ) )
            fingerprint = message
            data = ipAddr
            raise VerifyException( fingerprint, data )

         del karp[ deviceName ]

      if len( karp ) != 0:
         # there were some devices that we did not notice
         message = ( "dir did not contain neigh entries for "
                     "%s" % ( karp.keys() ) )
         fingerprint = message
         data = ipAddr
         raise VerifyException( fingerprint, data )

   def neighKey( self, ip6Addr, intf ):
      key = Tac.Value( "Arp::NeighborEntryKey", ip6Addr=Arnet.Ip6Addr( ip6Addr ), 
                       intfId=intf )
      return key

   # check that state is as we expect
   def neighborConfigEntryPresent( self, ip6Address, vrf=DEFAULT_VRF ):
      for intf in self.allIntfStatusDir.intfStatus:
         key = self.neighKey( ip6Address, intf )
         neighborEntry = self.staticArpConfigDir.ipv6.get( key )
         if neighborEntry:
            break
      if neighborEntry is None:
         return False
      else:
         return True

   def neighStatusEntryPresent( self, ip6Addr, vrf=DEFAULT_VRF ):
      for intfName in self.staticArpStatusDir.ipv6:
         neighIntf = self.staticArpStatusDir.ipv6.get( intfName )
         if neighIntf:
            neighStatusEntry = neighIntf.neighborEntry.get(
               Arnet.Ip6Addr( ip6Addr ) )
            if neighStatusEntry:
               return ( neighStatusEntry, intfName )
      return ( None, None )

   def arpSmashLookup( self, intfName, ipAddrStr, vrfName=DEFAULT_VRF ):
      '''
      Lookup arp entry

      Parameters
      ----------
      ipAddrStr : str
         String representing the IPv[46] address (e.g. '10.0.0.1' or '10::1')
      vrfName : str, optional
         Name of vrf where to install the arp/neighbor entry.
         If omitted, use default vrf.

      Returns
      -------
      entry : Arp::Table::ArpEntry or None
         Full arp entry, or None if entry not found.
      ''' 
      vrfId = self.vrfNameToId[ vrfName ] 
      if vrfId is None:
         return None
      genIpAddr = Tac.Value( 'Arnet::IpGenAddr', ipAddrStr )
      key = Tac.Value( 'Arp::Table::ArpKey', vrfId, genIpAddr, intfName )
      table = self.arpSmash_.arpEntry if genIpAddr.af == 'ipv4' \
                 else self.arpSmash_.neighborEntry
      return table.get( key )

   def kernelEntryPresent( self, ip, vrf=DEFAULT_VRF ):
      netNs = "ns-%s" % vrf if vrf != DEFAULT_VRF else DEFAULT_NS
      return( runMaybeInNetNs( netNs, [ "ip", "-6", "neigh", "show", "to", ip ],
                       stdout=Tac.CAPTURE ) )

   def verifyNeighborIp6( self, ip6Addr, vrf=DEFAULT_VRF ):
      fingerprint = ''
      data = None
      try:
         self.staticArpConfigDir = self.sysdb.entity.get(
               'arp/input/config/cli' ).vrf[ vrf ]
         self.staticArpStatusDir = self.sysdb.entity.get(
               'arp/input/status' ).vrf[ vrf ]
      except PyClientBase.RpcError as e:
         if not self.staticArpConfigDir or not self.staticArpStatusDir:
            fingerprint = 'Vrf %s configuration does not exist'
            data = vrf
            raise VerifyException( fingerprint, data )
         else:
            raise e
      try:
         neighConfig = self.neighborConfigEntryPresent( ip6Addr, vrf=vrf )

         if not neighConfig:
            fingerprint = 'Neighbor %s configuration does not exist'
            data = ip6Addr
            raise VerifyException( fingerprint, data )

         ( neighStatusEntry, intfName ) = self.neighStatusEntryPresent( ip6Addr,
                                                                        vrf=vrf )
         neighKernelEntry = self.kernelEntryPresent( ip6Addr, vrf=vrf )
         if ( intfName is not None ):
            arpSmashEntry = self.arpSmashLookup( intfName, ip6Addr, vrfName=vrf )
         else:
            arpSmashEntry = None

         if not neighStatusEntry:
            fingerprint = 'Neighbor %s status does not exist'
            data = ip6Addr
            raise VerifyException( fingerprint, data )
         if not arpSmashEntry:
            fingerprint = 'Neighbor %s does not exist in smash'
            data = ip6Addr
            raise VerifyException( fingerprint, data )
         if neighKernelEntry == '':
            fingerprint = 'Neighbor %s  does not exist in kernel'
            data = ip6Addr 
            raise VerifyException( fingerprint, data )
         #print 'Verified neighbor %s in VRF %s' % ( ip6Addr, vrf )
      except VerifyException as e:
         raise e

   def verify( self, addrAndVrf ):
      ip6Addr = addrAndVrf[ 0 ]
      vrf = addrAndVrf[ 1 ]
      if vrf == None:
         vrf = DEFAULT_VRF
      try:
         if ( ip6Addr == 'ALL' ):
            totalEntries = len( self.arpSmash_.neighborEntry )
            if totalEntries > NEIGHBOR_VERIFY_MAX:
               print 'Skipped Neighbor Verification for %s entries' % totalEntries
               raise NotApplicableException
            neighborDirByIntfName = self.arpSmashToNeighborDir()

            for vrf in [ DEFAULT_VRF ] + self.allVrfConfig.vrf.keys():
               self.staticArpConfigDir = self.sysdb.entity.get(
                     'arp/input/config/cli' ).vrf[ vrf ]
               for ( neighborEntry ) in self.staticArpConfigDir.ipv6:
                  self.verifyNeighborIp6( str( neighborEntry.ip6Addr ), vrf=vrf )

               netNs = "ns-%s" % vrf if vrf != DEFAULT_VRF else DEFAULT_NS
               self.compareIp6NeighborsAgainstKernel( neighborDirByIntfName,
                                           self.allIntfStatusDir, netNs )
            neighborDirByIntfName = None
         else: 
            self.verifyNeighborIp6( ip6Addr, vrf=vrf )
      except:
         raise
