# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, PyClient, AgentDirectory
from Arnet.Verify import VerifyException, NotApplicableException, runningAgents

PortChannelIntfId = Tac.Type( "Arnet::PortChannelIntfId" )

class Verifier:
   def __init__( self, entityManager ):
      if 'Lag' not in runningAgents():
         raise NotApplicableException
      sysname = AgentDirectory.agentList()[ 0 ][ 'system' ]
      self.sysdb = PyClient.PyClient( sysname, 'Sysdb' ).agentRoot()
      self.lagRoot = PyClient.PyClient( sysname, 'Lag' ).root()[ sysname ]
      self.ethLagIntfConfigDir = self.sysdb.entity.get( 'interface/config/eth/lag' )
      self.ethLagIntfStatusDir = self.lagRoot.entity.get( 
               'localAgentPlugin/interface/status/eth/lag' )
      self.allIntfStatusDir = self.sysdb.entity.get( 'interface/status/all' )
      self.lagInputIntf = self.sysdb.entity.get( 'lag/input/interface' )
      self.lagConfig = self.sysdb.entity.get( 'lag/config' )
      self.lagStatus = self.lagRoot.entity.get( 'localAgentPlugin/lag/status' )

   def cleanup( self ):
      self.sysdb = None
      self.lagRoot = None
      self.ethLagIntfConfigDir = None
      self.ethLagIntfStatusDir = None
      self.allIntfStatusDir = None
      self.lagInputIntf = None
      self.lagConfig = None
      self.lagStatus = None

   def verifyPoIntf( self, intfName ):
      fingerprint = ''
      data = None

      config = self.ethLagIntfConfigDir.intfConfig.get( intfName )
      status = self.ethLagIntfStatusDir.intfStatus.get( intfName )
      lagInputIntfDir = self.lagInputIntf
      lagConfigDir = self.lagConfig

      if not config:
         fingerprint = 'Port-Channel interface %s configuration does not exist'
         data = intfName
         raise VerifyException( fingerprint, data )

      # if config present, then check for status
      if status:
         if status.operStatus != 'intfOperUp':
            fingerprint = 'Port-Channel interface %s'\
                           ' is operationally down - %s'
            data = ( intfName, status.operStatus )
            if not status.membersAreActive:
               fingerprint = 'Port-Channel interface %s'\
                     ' is operationally down - %s due to LACP stay in standby'
               data = ( intfName, status.operStatus )
         # check that correct input status is chosen
         curHighestPriority = -1
         intfStatus = None
         for key in lagInputIntfDir:
            elisd = lagInputIntfDir[ key ]
            if elisd and intfName in elisd:
               if elisd.priority > curHighestPriority:
                  curHighestPriority = elisd.priority
                  intfStatus = elisd.intfStatus.get( intfName )
         if intfStatus != status:
            fingerprint = 'Port-Channel interface %s status (%s) '\
                           'comes from wrong input - %s'
            data = ( intfName, status, intfStatus )
         # check fallback consistency
         allowedFallbackMap = { # config : [ possible fallback type in status ]
            'fallbackNone' : [ 'fallbackNone' ],
            'fallbackStatic' : [ 'fallbackNone', 'fallbackStatic' ],
            'fallbackIndividual' : [ 'fallbackNone', 'fallbackIndividual' ] }
         if status.fallbackEnabled not in\
               allowedFallbackMap[ config.fallback ]:
            fingerprint = 'Port-Channel interface %s fallback '\
                           'config/status mismatch. Config: %s, Status: %s'
            data = ( intfName, config.fallback, status.fallbackEnabled )
         # verify that member ports are actually configured in this LAG
         for port in status.member.iterkeys():
            epilc = lagConfigDir.phyIntf.get( port )
            if epilc.lag != config:
               fingerprint = 'Port-Channel interface %s contains member port'\
                              'not configured in this port-channel - %s'
               data = ( intfName, port )
      else:
         fingerprint = 'Port-Channel interface %s status does not exist'
         data = intfName

      if fingerprint != '':
         raise VerifyException( fingerprint, data )

   def verifyPhyIntf( self, intfName ):
      fingerprint = ''
      data = None

      lagConfigDir = self.lagConfig
      lagStatusDir = self.lagStatus
      allIntfStatusDir = self.allIntfStatusDir
      ethLagIntfStatusDir = self.ethLagIntfStatusDir

      config = lagConfigDir.phyIntf.get( intfName )
      status = lagStatusDir.phyIntf.get( intfName )

      if not( config and config.lag ):
         # This physical port is not configured as part of a port-channel
         raise NotApplicableException()

      # check that EthPhyIntfLagStatus exists
      if not status:
         fingerprint = 'Physical interface %s is configured in a Port-Channel'\
                        ' but missing EthPhyIntfLagStatus'
         data = intfName
         raise VerifyException( fingerprint, data )

      # check forwarding model
      elis = ethLagIntfStatusDir.intfStatus.get( config.lag.intfId )
      forwardingModel = \
         allIntfStatusDir.intfStatus.get( intfName ).forwardingModel
      if elis.fallbackEnabled != 'fallbackIndividual':
         if forwardingModel != 'intfForwardingModelDataLink':
            fingerprint = 'Physical interface %s is configured in a Port-Channel'\
               ' but forwarding model is %s'
            data = ( intfName, forwardingModel )
      else:
         if forwardingModel != 'intfForwardingModelBridged':
            fingerprint = 'Physical interface %s is configured in a Port-Channel'\
               ' and fallback individual is enabled but forwarding model is %s'
            data = ( intfName, forwardingModel )

      if fingerprint != '':
         raise VerifyException( fingerprint, data )

   def verify( self, intfName ):
      try:
         if PortChannelIntfId.isPortChannelIntfId( intfName ):
            self.verifyPoIntf( intfName )
         elif 'ethernet' in intfName.lower():
            self.verifyPhyIntf( intfName )
      except: # pylint: disable-msg=W0702
         raise
