#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from socket import AF_INET, AF_INET6
import Tac, PyClient, AgentDirectory, Arnet
from Arnet.Verify import VerifyException
from FibUtils import mountSmashTables, unmountSmashTables
from KernelFibUtils import kernelIpRoutes

class Verifier:
   def __init__( self, entityManager ):
      sysname = AgentDirectory.agentList()[ 0 ][ 'system' ]

      self.sysdb = PyClient.PyClient( sysname, 'Sysdb' ).agentRoot()
      self.ipStatusDir = self.sysdb.entity.get( 'ip/status' )
      self.ip6StatusDir = self.sysdb.entity.get( 'ip6/status' )
      self.entityManager = entityManager
      self.fibRoutingStatus = None

   def cleanup( self ):
      self.sysdb = None
      self.ipStatusDir = None
      self.ip6StatusDir = None
      self.fibRoutingStatus = None
   
   def checkConnectedRoute( self, prefix, prefSrc, intfName, vrf, af=AF_INET ):
      fingerprint = ''
      data = ''
      ipv6 = True if af == AF_INET6 else False
      kernelRoutes = kernelIpRoutes( vrf, ipv6=ipv6, connectedRoute=True,
                                     getMetric=True )
      found = False
      if prefix.stringValue in kernelRoutes:
         found = True
      if found:
         ( _, _, proto, scope, src, metric ) = \
               kernelRoutes[ prefix.stringValue ][ 0 ]

      def getFingerprint( intfName, variable, expectedValue, variableString ):
         fingerprint = ''
         data = ''
         if variable != expectedValue:
            fingerprint = 'connected route for %s does not have correct %s \
                           programmed in kernel'
            data = ( intfName, variableString )
         return ( fingerprint, data )
         
      # check prefix
      if found == False:
         fingerprint = 'connected route for %s is not present in kernel'
         data = intfName
         return ( fingerprint, data )

      # check preferred src
      ( fingerprint, data ) = getFingerprint( intfName, src, prefSrc,
                                    "preferred source address" )
      if fingerprint:
         return ( fingerprint, data )

      # check proto
      ( fingerprint, data ) = getFingerprint( intfName, proto, "kernel", "proto" )
      if fingerprint:
         return ( fingerprint, data )

      # check scope
      if af == AF_INET:
         ( fingerprint, data ) = getFingerprint( intfName, scope, "link", "scope" )
         if fingerprint:
            return ( fingerprint, data )

      # check metric
      expectedMetric = self.fibRoutingStatus.route[ prefix ].metric
      ( fingerprint, data ) = getFingerprint( intfName, metric,
                                              str( expectedMetric ),
                                              "metric" )
      if fingerprint:
         return ( fingerprint, data )

      return ( fingerprint, data )

   def intfOk( self, intfName, af=AF_INET ):
      fingerprint = ''
      data = ''
      addrs = None
      ipVersion = 4 if af == AF_INET else 6
      if af == AF_INET:
         status = self.ipStatusDir.ipIntfStatus.get( intfName )
         if status:
            addrs = [ status.activeAddrWithMask ] + \
                  status.activeSecondaryWithMask.keys()
      else:
         status = self.ip6StatusDir.intf.get( intfName )
         if status:
            addrs = status.addr.keys()
      if status:
         # Mount routing/status
         ( self.fibRoutingStatus, _ ) = mountSmashTables( self.entityManager, 
                                              vrfName=status.vrf,
                                              ipVersion=ipVersion )
         for addr in addrs:
            prefSrc = str( addr.address )
            prefix = Arnet.Subnet( prefSrc, addr.len, af ).toValue()
            if prefix in self.fibRoutingStatus.route:
               ( fingerprint, data ) = self.checkConnectedRoute( prefix, prefSrc,
                                                 intfName, status.vrf, af=af )
         # unmount routing/status
         unmountSmashTables( self.entityManager, vrfName=status.vrf,
                             ipVersion=ipVersion )
      return ( fingerprint, data )

   def verifyIntf( self, intfName ):
      # Verify both v4 and v6 connected routes
      for af in [ AF_INET, AF_INET6 ]:
         ( fingerprint, data ) = self.intfOk( intfName, af=af )
         if fingerprint != '':
            raise VerifyException( fingerprint, data )

   def verify( self, intfName ):
      try:
         self.verifyIntf( intfName )
      except: # pylint: disable-msg=W0702
         raise


