#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, PyClient, Cell, AgentDirectory
import SmashLazyMount
from Arnet.Verify import VerifyException, NotApplicableException, runningAgents
import Arnet
from socket import AF_INET, AF_INET6

class Verifier:
   def __init__( self, entityManager ):
      if 'Ira' not in runningAgents():
         raise NotApplicableException

      sysname = AgentDirectory.agentList()[ 0 ][ 'system' ]

      self.sysdb = PyClient.PyClient( sysname, 'Sysdb' ).agentRoot()
      self.ira = PyClient.PyClient( sysname, 'Ira' ).agentRoot()

      self.ipConfigDir = self.sysdb.entity.get( 'ip/config' )
      self.ipStatusDir = self.sysdb.entity.get( 'ip/status' )
      self.ip6ConfigDir = self.sysdb.entity.get( 'ip6/config' )
      self.ip6StatusDir = self.sysdb.entity.get( 'ip6/status' )
      self.vrfStatusGlobal = self.sysdb.entity.get( 'ip/vrf/status/global' )
      self.vrfStatusLocal = \
           self.sysdb.entity.get( Cell.path( "ip/vrf/status/local" ) )
      self.routingVrfCfgDir = self.sysdb.entity.get( 'routing/vrf/config' )
      self.routing6VrfCfgDir = self.sysdb.entity.get( 'routing6/vrf/config' )
      self.entityManager = entityManager
      self.ribMessage = ' Check if Rib::setup has been called for the vrf'
      self.fibRoutingStatus = None
      self.fibRouting6Status = None
      self.fibRoutingStatusCache = {}
      self.fibRouting6StatusCache = {}

   def cleanup( self ):
      self.sysdb = None
      self.ira = None

      self.ipConfigDir = None
      self.ip6ConfigDir = None
      self.ipStatusDir = None
      self.ip6StatusDir = None
      self.vrfStatusGlobal = None 
      self.vrfStatusLocal = None 
      self.routingVrfCfgDir = None 
      self.routing6VrfCfgDir = None 
      self.entityManager = None 
      self.ribMessage = None 
      self.fibRoutingStatus = None
      self.fibRouting6Status = None
      self.fibRoutingStatusCache = {}
      self.fibRouting6StatusCache = {}

      SmashLazyMount.smashProxies.clear()

   def checkVrfState( self, config ):
      fingerprint = ''
      vrfState = Tac.Type( "Ip::VrfState" )
      if self.vrfStatusGlobal.vrf.has_key( config.vrf ) != True:
         fingerprint = "vrfRoot NULL for %s due to missing " \
                       "ip/vrf/status/global/vrf entry"
      elif self.vrfStatusLocal.vrf.has_key( config.vrf ) != True:
         fingerprint = "vrfRoot NULL for %s due to missing " \
                       "ip/vrf/status/local entry"
      elif self.vrfStatusLocal.vrf[ config.vrf ].state != \
                vrfState.active:
         fingerprint = "vrfRoot NULL for %s due to " \
                       "ip/vrf/status/local/vrf/state not active"
      return fingerprint

   def checkV4ConnectedRoute( self, connectedRoute, intfName, vrf ) :
      fingerprint = ''
      data = None
      if connectedRoute.stringValue != '0.0.0.0/0' and \
         self.fibRoutingStatus.route.has_key( \
              connectedRoute ) == False:
         fingerprint = "connected route %s is not present " \
                       "for intf %s in vrf %s: %s"
         data = ( connectedRoute, 
                  intfName,
                  vrf,
                  self.ribMessage )
      return ( fingerprint, data )

   def getFibRoutingStatus( self, vrfName ):
      self.fibRoutingStatus = self.fibRoutingStatusCache.get( vrfName )
      if not self.fibRoutingStatus:
         path = 'routing/status' if vrfName == 'default' else \
                'routing/vrf/status/%s' % vrfName
         self.fibRoutingStatus = SmashLazyMount.mount( 
                                        self.entityManager, path,
                                        'Smash::Fib::RouteStatus',
                                        SmashLazyMount.mountInfo( 'reader' ) )
         self.fibRoutingStatusCache[ vrfName ] = self.fibRoutingStatus
      return self.fibRoutingStatus

   def getFibRouting6Status( self, vrfName ):
      self.fibRouting6Status = self.fibRouting6StatusCache.get( vrfName )
      if not self.fibRouting6Status:
         path = 'routing6/status' if vrfName == 'default' else \
                'routing6/vrf/status/%s' % vrfName
         self.fibRouting6Status = SmashLazyMount.mount( 
                                        self.entityManager, path,
                                        'Smash::Fib6::RouteStatus',
                                        SmashLazyMount.mountInfo( 'reader' ) )
         self.fibRouting6StatusCache[ vrfName ] = self.fibRouting6Status
      return self.fibRouting6Status

   def intfV4Ok( self, intfName ):
      fingerprint = ''
      data = None
      routedConfig = None 

      # get the config, status and routedIpConfig
      config = self.ipConfigDir.ipIntfConfig.get( intfName )
      status = self.ipStatusDir.ipIntfStatus.get( intfName )
      if config:
         try:
            self.fibRoutingStatus = self.getFibRoutingStatus( config.vrf )
         except Exception as e: # pylint: disable-msg=W0703
            fingerprint = 'Mounting Fib routing status from Smash failed: %s' % e
         else:
            if config.vrf == 'default':
               ipIntfSm = self.ira[ 'Ira' ].iraRoot.ipIntfSm
            else:
               if self.ira[ 'Ira' ].vrfRootSm.vrfRoot.has_key( config.vrf ):
                  ipIntfSm = self.ira[ 'Ira' ].vrfRootSm.\
                                  vrfRoot[ config.vrf ].ipIntfSm
               else:
                  fingerprint = self.checkVrfState( config )
                  if fingerprint == '' and \
                                 self.routingVrfCfgDir.vrf.has_key( config.vrf ):
                     fingerprint = "vrfRoot NULL for %s due to missing " \
                                   "routing/vrf/config"
                  data = config.vrf
                  ipIntfSm = None
               if ipIntfSm and ipIntfSm.routedIpIntfConfig.has_key( intfName ):
                  routedConfig = ipIntfSm.routedIpIntfConfig[ intfName ]

      if fingerprint != '': 
         return ( fingerprint, data )

      # if the config is present then check for the status
      if config:
         if status:
            # Make sure the config and the status have the same vrf
            # otherwise, fingerprint it
            if status.vrf == config.vrf:
               # Make sure all the addresses in the config
               # are present in the status. Otherwise, 
               # fingerprint it
               if ( ( [ status.activeAddrWithMask ] + 
                        status.activeSecondaryWithMask.keys() ) != 
                    ( [ config.addrWithMask ] + 
                        config.secondaryWithMask.keys() ) ):

                  fingerprint = "ip address %s not present in the " \
                                "status for the intf %s in vrf %s" 
                  allAddr = [ config.addrWithMask.address ]
                  for key in config.secondaryWithMask.keys():
                     allAddr = allAddr + [ key.address ]
                  addr = str( allAddr ).strip('[]')
                  data =   ( addr, 
                             intfName,
                             status.vrf )
                  return ( fingerprint, data )
               # status is intact with addresses, so check
               # the connected routes
               connectedRoute = Arnet.Subnet( 
                                      status.activeAddrWithMask.address, 
                                      status.activeAddrWithMask.len, 
                                      AF_INET ).toValue()
               ( fingerprint, data ) = self.checkV4ConnectedRoute( 
                                            connectedRoute, 
                                            intfName,
                                            status.vrf )
               for key in status.activeSecondaryWithMask.keys():
                  connectedRoute = Arnet.Subnet( 
                                         key.address, 
                                         key.len, 
                                         AF_INET ).toValue()
                  ( fingerprint, data ) = self.checkV4ConnectedRoute( 
                                               connectedRoute,
                                               intfName,
                                               status.vrf )

               # verify kernel related items for the intf
               Arnet.Verify.verifyKernelDevice( intfName, self.sysdb )
            else:
               fingerprint = "ip status vrf %s is not same as " \
                             "the config vrf %s for the intf %s"
               data = ( status.vrf, config.vrf, intfName )
         else:
            # verify kernel device here - this condition typically happens if
            # kernel device is missing
            Arnet.Verify.verifyKernelDevice( intfName, self.sysdb )

            # if the status is not present then check if the routedConfig
            # is present. If it does then for v4 fingerprint the error.
            # For v6, make sure the config has addresses before
            # fingerprinting the error. 
            if routedConfig:
               fingerprint = "ip status not present but routedConfig " \
                             "is present for the intf %s"
               data = intfName
            else:
               # config itself is missing, most probably not a problem.
               # just log it, don't mark it as a problem.
               print "ip routedConfig/status is not present for the intf %s" % \
                     intfName

      return ( fingerprint, data )

   def intfV6Ok( self, intfName ):
      fingerprint = ''
      data = None
      routedConfig = None 

      # get the config, status and routedIpConfig
      config = self.ip6ConfigDir.intf.get( intfName )
      status = self.ip6StatusDir.intf.get( intfName )
      if config:
         try:
            self.fibRouting6Status = self.getFibRouting6Status( config.vrf )
         except Exception as e: # pylint: disable-msg=W0703
            fingerprint = 'Mounting Fib6 routing status from Smash failed %s' % e
         else:
            if config.vrf == 'default':
               ip6IntfSm = self.ira[ 'Ira' ].iraRoot.ip6IntfSm
            else:
               if self.ira[ 'Ira' ].vrfRootSm.vrfRoot.has_key( config.vrf ):
                  ip6IntfSm = self.ira[ 'Ira' ].vrfRootSm.\
                                   vrfRoot[ config.vrf ].ip6IntfSm
               else:
                  fingerPrint = self.checkVrfState( config )
                  if fingerPrint == '' and \
                                 self.routing6VrfCfgDir.vrf.has_key( config.vrf ):
                     fingerprint = "vrfRoot NULL for %s due to " \
                                   "missing routing6/vrf/config"
                  data = config.vrf
                  ip6IntfSm = None
            if ip6IntfSm and ip6IntfSm.routedIp6IntfConfig.has_key( intfName ):
               routedConfig = ip6IntfSm.routedIp6IntfConfig[ intfName ]

      if fingerprint != '': 
         return ( fingerprint, data )

      # if the config is present then check for the status
      if config:
         if status:
            # Make sure the config and the status have the same vrf
            # otherwise, fingerprint it
            if status.vrf == config.vrf:
               # Make sure all the addresses in the config
               # are present in the status. Otherwise, 
               # fingerprint it
               for i in config.addr:
                  if not status.addr.has_key( i ):
                     fingerprint = "ipv6 address %s not present in " \
                                   "the status for the intf %s in vrf %s"
                     data = ( i, intfName, status.vrf )
                     return ( fingerprint, data )
               # status is intact with addresses, so check
               # the connected routes
               for addr in status.addr:
                  connectedRoute = Arnet.Subnet( addr.address, 
                                                 addr.len, 
                                                 AF_INET6 ).toValue()
                  # skip link local routes
                  if connectedRoute.stringValue == 'fe80::/64':
                     continue
                  if self.fibRouting6Status.route.has_key( \
                          connectedRoute ) == False:
                     fingerprint = "connected route %s is not present " \
                                   "for intf %s in vrf %s: %s"
                     data = ( connectedRoute, 
                              intfName,
                              status.vrf,
                              self.ribMessage )
               # verify kernel related items for the intf
               Arnet.Verify.verifyKernelDevice( intfName, self.sysdb )
            else:
               fingerprint = "ipv6 status vrf %s is not same as " \
                             "the config vrf %s for the intf %s"
               data = ( status.vrf, config.vrf, intfName )
         else:
            # verify kernel device here - this condition typically happens if
            # kernel device is missing
            Arnet.Verify.verifyKernelDevice( intfName, self.sysdb )

            # if the status is not present then check if the routedConfig
            # is present. If it does then for v4 fingerprint the error.
            # For v6, make sure the config has addresses before
            # fingerprinting the error. 
            if routedConfig:
               if len( config.addr ):
                  fingerprint = "ipv6 status not present but routedConfig " \
                                "is present for the intf %s"
                  data = intfName
            else:
               # config itself is missing, most probably not a problem.
               # just log it, don't mark it as a problem.
               print "ipv6 routedConfig/status is not present for the intf %s" % \
                     intfName

      return ( fingerprint, data )

   def verifyIntf( self, intfName ):
      # Verify both v4 and v6 config/status pairs
      ( fingerprint, data ) = self.intfV4Ok( intfName )
      if fingerprint != '':
         raise VerifyException( fingerprint, data )

      ( fingerprint, data ) = self.intfV6Ok( intfName )
      if fingerprint != '':
         raise VerifyException( fingerprint, data )

   def verify( self, intfName ):
      try:
         self.verifyIntf( intfName )
      except: # pylint: disable-msg=W0702
         raise
