#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import PyClient, Cell, AgentDirectory
from Arnet.Verify import VerifyException, NotApplicableException, runningAgents

class Verifier:
   def __init__( self, entityManager ):
      if 'Ebra' not in runningAgents():
         raise NotApplicableException

      sysname = AgentDirectory.agentList()[ 0 ][ 'system' ]

      self.sysdb = PyClient.PyClient( sysname, 'Sysdb' ).agentRoot()
      self.ebra = PyClient.PyClient( sysname, 'Ebra' ).agentRoot()

      self.intfHwCapability = self.sysdb.entity.get(
                                                'interface/hardware/capability' )
      self.subIntfHwStatusDir = self.sysdb.entity.get(
                                                'interface/hardware/status/subintf' )
      self.subIntfConfigDir = self.sysdb.entity.get( 'interface/config/subintf' )
      self.allIntfConfigDir = self.sysdb.entity.get( 'interface/config/all' )
      self.allIntfStatusDir = self.sysdb.entity.get( 'interface/status/all' )
      self.allIntfStatusLocalDir = self.sysdb.entity.get(
                                             Cell.path( 'interface/status/local' ) )
      self.allIntfNamespaceConfigDir = self.sysdb.entity.get(
                                             Cell.path( 'interface/nsconfig' ) )
      self.bridgingConfig = self.sysdb.entity.get( 'bridging/config' )

   def cleanup( self ):
      self.sysdb = None
      self.ebra = None

      self.intfHwCapability = None
      self.subIntfHwStatusDir = None
      self.subIntfConfigDir = None
      self.allIntfConfigDir = None
      self.allIntfStatusDir = None
      self.allIntfStatusLocalDir = None
      self.allIntfNamespaceConfigDir = None
      self.bridgingConfig = None

   def verifySubIntf( self, subIntfName ):
      # Verifies the following:
      # * Config/Status/StatusLocal/SwitchIntfConfig etc are present
      # * Interface is enabled
      # * BridgeMac is set
      # * encap dot1q tag is configured properly and without duplication
      # * Parent interface is up, in routed mode and has the right forwardingModel
      # * Hardware programming is successful
      # * Interface is operationally up
      # * Kernel device state is fine (should be up, be in the right namespace etc)
      fingerprint = ''
      data = None
      rethrow = True  # rethrow the exception to the caller

      def parentIntfOk( subIntfName ):
         parentIntfName = subIntfName.split( '.' )[ 0 ]
         status = self.allIntfStatusDir.intfStatus.get( parentIntfName )
         if not status:
            return ( False, 'Parent interface status is not available' )
         elif status.operStatus != 'intfOperUp':
            return ( False, 'Parent interface is not operationally up' )
         elif status.forwardingModel != 'intfForwardingModelRouted':
            return ( False, 'Parent interface is not a routed port' )
         else:
            return ( True, '' )

      def parent( subIntf ):
         assert '.' in subIntf.intfId
         return subIntf.intfId.split( '.' )[ 0 ]

      assert '.' in subIntfName
      try:
         config = self.subIntfConfigDir.intfConfig.get( subIntfName )
         status = self.allIntfStatusDir.intfStatus.get( subIntfName )
         statusLocal = self.allIntfStatusLocalDir.intfStatusLocal.get( subIntfName )
         sic = self.bridgingConfig.switchIntfConfig.get( subIntfName )
         hwStatus = self.subIntfHwStatusDir.hardwareStatus.get( subIntfName )

         if not self.intfHwCapability.l3SubintfSupported:
            fingerprint = 'Subinterfaces are not supported or is not enabled'
            raise VerifyException( fingerprint )
         if not config:
            fingerprint = 'Subinterface %s configuration does not exist'
            data = subIntfName
            raise VerifyException( fingerprint, data )
         if not config.adminEnabled:
            fingerprint = 'Subinterface %s is administratively down'
            data = subIntfName
            raise VerifyException( fingerprint, data )
         if not config.enabled:
            fingerprint = 'Subinterface %s is not enabled'
            data = subIntfName
            raise VerifyException( fingerprint, data )
         if not status:
            fingerprint = 'Subinterface %s status does not exist'
            data = subIntfName
            raise VerifyException( fingerprint, data )
         if not statusLocal:
            fingerprint = 'Subinterface %s statusLocal does not exist'
            data = subIntfName
            raise VerifyException( fingerprint, data )
         if not sic:
            fingerprint = 'Subinterface %s switchIntfConfig is missing'
            data = subIntfName
            raise VerifyException( fingerprint, data )
         if self.bridgingConfig.bridgeMacAddr == '00:00:00:00:00:00':
            fingerprint = 'Bridge Mac Addr is zero'
            raise VerifyException( fingerprint )
         if config.encapType != 'dot1q':
            fingerprint = 'Unknown encap type for subinterface %s'
            data = subIntfName
            raise VerifyException( fingerprint, data )
         if config.vlanId == 0:
            fingerprint = 'No dot1q tag configured for subinterface %s'
            data = subIntfName
            raise VerifyException( fingerprint, data )
         if len( [ s for s in self.subIntfConfigDir.values() \
                     if parent( s ) == parent( config ) and \
                     s.vlanId == config.vlanId ] ) > 1:
            fingerprint = 'Invalid configuration for %s - duplicate dot1q tag'
            data = subIntfName
            raise VerifyException( fingerprint, data )
         ( ok, fingerprint ) = parentIntfOk( subIntfName )
         if not ok:
            fingerprint = fingerprint + ' (for subinterface %s)'
            data = subIntfName
            raise VerifyException( fingerprint, data )
         if not hwStatus:
            fingerprint = 'Subinterface %s hardware status does not exist'
            data = subIntfName
            raise VerifyException( fingerprint, data )
         if not hwStatus.hardwareSuccess:
            fingerprint = 'Subinterface %s not successfully programmed in hardware'
            data = subIntfName
            rethrow = False  # stop the exception, continue evaluating other actors
            raise VerifyException( fingerprint, data )
         if status.operStatus != 'intfOperUp':
            fingerprint = 'Subinterface %s is operationally down - %s'
            data = ( subIntfName, status.operStatus )
            raise VerifyException( fingerprint, data )
         # verificiation of kernel device happens in Ira plugin
      except VerifyException as e:
         if rethrow:
            raise e

   def verifyVlanIntf( self, vlanIntfName ):
      # Verifies the following:
      # * Config/Status/StatusLocal/SwitchIntfConfig etc are present
      # * Interface is enabled
      # * BridgeMac is set
      # * Interface is operationally up
      # * Kernel device state is fine (should be up, be in the right namespace etc)
      fingerprint = ''
      data = None
      try:
         config = self.allIntfConfigDir.intfConfig.get( vlanIntfName )
         status = self.allIntfStatusDir.intfStatus.get( vlanIntfName )
         statusLocal = self.allIntfStatusLocalDir.intfStatusLocal.get( vlanIntfName )

         if not config:
            fingerprint = 'Vlan interface %s configuration does not exist'
            data = vlanIntfName
            raise VerifyException( fingerprint, data )
         if not config.adminEnabled:
            fingerprint = 'Vlan interface %s is administratively down'
            data = vlanIntfName
         if not config.enabled:
            fingerprint = 'Vlan interface %s is not enabled'
            data = vlanIntfName
            raise VerifyException( fingerprint, data )
         if not status:
            fingerprint = 'Vlan interface %s status does not exist'
            data = vlanIntfName
            raise VerifyException( fingerprint, data )
         if not statusLocal:
            fingerprint = 'Vlan interface %s statusLocal does not exist'
            data = vlanIntfName
            raise VerifyException( fingerprint, data )
         if self.bridgingConfig.bridgeMacAddr == '00:00:00:00:00:00':
            fingerprint = 'Bridge Mac Addr is zero'
            raise VerifyException( fingerprint )
         if status.operStatus != 'intfOperUp':
            fingerprint = 'Vlan interface %s is operationally down - %s'
            data = ( vlanIntfName, status.operStatus )
            raise VerifyException( fingerprint, data )
         # verificiation of kernel device happens in Ira plugin
      except VerifyException as e:
         raise e

   def isSvi( self, intfName ):
      return 'vlan' in intfName.lower()

   def isSubIntf( self, intfName ):
      return '.' in intfName

   def isLoopback( self, intfName ):
      return 'loopback' in intfName.lower()

   def verifyInternalVlan( self, intfName ):
      fingerprint = None
      data = None

      config = self.allIntfConfigDir.intfConfig.get( intfName )
      status = self.allIntfStatusDir.intfStatus.get( intfName )
      sic = self.bridgingConfig.switchIntfConfig.get( intfName )
      if config and status and config.adminEnabled and \
                               status.forwardingModel == 'intfForwardingModelRouted':
         if self.isSvi( intfName ) or self.isLoopback( intfName ):
            pass # no 'sic' verification needed
         elif not sic:
            fingerprint = 'SwitchIntfConfig missing for %s'
            data = intfName
         elif not sic.switchportMode == 'routed':
            fingerprint = 'switchportMode for %s is %s instead of being routed'
            data = ( intfName, sic.switchportMode )
         elif not sic.nativeVlan in xrange( 1, 4095 ):
            fingerprint = '%s internal vlan is %s (invalid)'
            data = ( intfName, str( sic.nativeVlan ) )
         if fingerprint:
            raise VerifyException( fingerprint, data )
      elif status:
         #print 'Ebra: %s operStatus=%s, forwardingModel=%s' % \
         #      ( intfName, status.operStatus, status.forwardingModel )
         raise NotApplicableException

   def verify( self, intfName ):
      try:
         self.verifyInternalVlan( intfName )
      except NotApplicableException:
         if not self.isSvi( intfName ) and not self.isSubIntf( intfName ):
            raise
      except: # pylint: disable-msg=W0702
         raise

      if self.isSvi( intfName ):
         self.verifyVlanIntf( intfName )
      elif self.isSubIntf( intfName ):
         self.verifySubIntf( intfName )
