#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, PyClient, AgentDirectory, SmashLazyMount
import PyClientBase
from Arnet.NsLib import runMaybeInNetNs, DEFAULT_NS
from Arnet.Verify import VerifyException, NotApplicableException, runningAgents
import re
from IpLibConsts import DEFAULT_VRF

ARP_VERIFY_MAX = 100
class Verifier( object ):
   def __init__( self, entityManager ):
      if 'Arp' not in runningAgents():
         raise NotApplicableException
      sysname = AgentDirectory.agentList()[ 0 ][ 'system' ]
      self.entityManager = entityManager
      self.sysdb = PyClient.PyClient( sysname, 'Sysdb' ).agentRoot()
      self.arpRoot = PyClient.PyClient( sysname, 'Arp' ).root()[ sysname ]
      self.intfStatusKernel = self.arpRoot.entity.get(
         "localAgentPlugin/interface/status/all" )
      self.staticArpConfigDir = None
      self.staticArpStatusDir = None
      self.arpSmash_ = SmashLazyMount.mount( entityManager, 'arp/status',
            'Arp::Table::Status', SmashLazyMount.mountInfo( 'reader' ) )
      self.vrfSmash_ = SmashLazyMount.mount( entityManager, 'vrf/vrfIdMapStatus',
            'Vrf::VrfIdMap::Status', SmashLazyMount.mountInfo( 'reader' ) )
      self.allVrfConfig = self.sysdb.entity.get( "ip/vrf/config" )
      self.vrfNameToId = {}
      for vrfId, vrfEntry in self.vrfSmash_.vrfIdToName.iteritems():
         self.vrfNameToId[ vrfEntry.vrfName ] = vrfId

   def cleanup( self ):
      self.sysdb = None
      self.arpRoot = None
      self.intfStatusKernel = None
      self.staticArpConfigDir = None 
      self.staticArpStatusDir = None
      self.allVrfConfig = None
      self.arpSmash_ = None
      self.vrfSmash_ = None
      self.vrfNameToId = None

   def readKernelArpTable( self, netNs=DEFAULT_NS ):
      # Build a dict containing the kernel's ARP table.
      karp = {}

      arpRe = re.compile( 
         r"(\d+\.\d+\.\d+\.\d+)\s+\S+\s+(\S+)\s+(\S+)\s+\S+\s+(\S+)$" )
      # From include/linux/if_arp.h
      ATF_COM = 0x02
      ATF_PERM = 0x04

      # store just the ip addresses that we have seen
      ips = {}

      #if netNs is DEFAULT_NS:
      #   fo = file( '/proc/net/arp', 'r' )
      #else:
      fo = runMaybeInNetNs( netNs, [ 'cat', '/proc/net/arp' ], 
                            stdout=Tac.CAPTURE, asRoot=True )
      fo = fo.split( '\n' )

      for line in fo:
         if line == '':
            continue
         m = arpRe.match( line )
         if m:
            ip = m.group( 1 )
            flags = int( m.group( 2 ), 16 )
            static = bool( flags & ATF_PERM )
            if flags & ATF_COM:
               eth = m.group( 3 )
            else:
               # Entries that have been manually deleted from the ARP table will be
               # incomplete, but the old Ethernet address still appears in
               # /proc/net/arp.  We need to ignore that old address.
               eth = '00:00:00:00:00:00'
               continue
            deviceName = m.group( 4 )
         
            karp.setdefault( deviceName, {} )[ ip ] = ( eth, static )
            ips[ ip ] = ips.get( ip, [] ) + [ deviceName ]
      return karp, ips

   def arpSmashToNeighborDir( self ):
      '''
      Convert arp smash into Arp::NeighborDir.

      Return None if arpSmash is None.
      '''
      if not self.arpSmash_:
         return None

      nd = Tac.newInstance( "Arp::NeighborDir", "temp" )
      for entry in self.arpSmash_.arpEntry.itervalues():
         intfName = entry.intfId
         if not intfName in nd.arpIntfStatus:
            nd.arpIntfStatus.newMember( intfName )
         ais = nd.arpIntfStatus[ intfName ]
         aes = Tac.newInstance( "Arp::ArpEntryStatus", entry.addr.v4Addr )
         aes.ethAddr = entry.ethAddr
         aes.isStatic = entry.isStatic
         aes.source = entry.source
         ais.arpEntry.addMember( aes )

      return nd

   def compareNeighborsAgainstKernel( self, neighborDirByIntfName,
                                      intfStatusKernel,
                                      netNs=DEFAULT_NS ):

      karp, _ = self.readKernelArpTable( netNs )
      directory = neighborDirByIntfName
      if directory is None:
         return 

      for key in directory.arpIntfStatus:
         deviceName = intfStatusKernel[ key ].deviceName
         if 'Management' in key or 'Internal' in key:
            karp.pop( deviceName, None )
            continue

         arpIntfStatus = directory.arpIntfStatus[ key ]
         arpEntries = arpIntfStatus.arpEntry

         if deviceName not in karp:
            continue
      
         karpDeviceNameDict = karp[ deviceName ]

         for ipAddr, arpEntry in arpEntries.iteritems(): # pylint: disable-msg=W0621
            if ipAddr not in karpDeviceNameDict:
               message = ( "dir contains an ip address for %s "
                           "that is not in the "
                           "kernel arp table: '%s'" % ( deviceName, ipAddr ) )
               fingerprint = message
               data = ipAddr
               raise VerifyException( fingerprint, data )

            ( ethAddr, _ ) = karpDeviceNameDict[ ipAddr ]
            assert ethAddr == arpEntry.ethAddr
            if ethAddr != arpEntry.ethAddr:
               fingerprint = "Mismatch MAC address %s %s" % ( ethAddr, \
                                                              arpEntry.ethAddr )
               data = ipAddr
               raise VerifyException( fingerprint, data )
         
            # remove this ip address so only those that aren't tracked will
            # be in the dictionary
            #print 'Verified ARP %s in kernel' % ipAddr
            del karpDeviceNameDict[ ipAddr ]
      
         # scrub invalid entries that the kernel still shows
         for ipAddr, ( ethAddr, _ ) in karpDeviceNameDict.items():
            if ethAddr == '00:00:00:00:00:00':
               del karpDeviceNameDict[ ipAddr ]
      
         if len( karpDeviceNameDict ) != 0:
            # there were some arp entries that we did not notice for this device
            message = ( "dir for %s does not contain arp entries "
                        "for %s" % ( deviceName, karpDeviceNameDict.items() ) )
            fingerprint = message
            data = deviceName
            raise VerifyException( fingerprint, data )

         del karp[ deviceName ]

      if len( karp ) != 0:
         error = False
         for devName in karp.keys():
            # ignore arp entries on the internal intf between active and
            # standby, since Arp agent ignores them.
            if not devName.lower().startswith( "internal" ):
               error = True

         if error:
            # there were some devices that we did not notice
            message = ( "dir did not contain arp entries for "
                        "%s" % ( karp.keys() ) )
            fingerprint = message
            data = karp.keys()
            raise VerifyException( fingerprint, data )

   def arpSmashLookup( self, intfName, ipAddrStr, vrfName=DEFAULT_VRF ):
      '''
      Lookup arp entry

      Parameters
      ----------
      ipAddrStr : str
         String representing the IPv[46] address (e.g. '10.0.0.1' or '10::1')
      vrfName : str, optional
         Name of vrf where to install the arp/neighbor entry.
         If omitted, use default vrf.

      Returns
      -------
      entry : Arp::Table::ArpEntry or None
         Full arp entry, or None if entry not found.
      ''' 
      vrfId = self.vrfNameToId[ vrfName ]
      if vrfId is None:
         return None
      genIpAddr = Tac.Value( 'Arnet::IpGenAddr', ipAddrStr )
      key = Tac.Value( 'Arp::Table::ArpKey', vrfId, genIpAddr, intfName )
      table = self.arpSmash_.arpEntry if genIpAddr.af == 'ipv4' \
                 else self.arpSmash_.neighborEntry
      return table.get( key )

   def kernelEntryPresent( self, ip, vrf=DEFAULT_VRF ):
      netNs = "ns-%s" % vrf if vrf != DEFAULT_VRF else DEFAULT_NS
      return( runMaybeInNetNs( netNs, [ "ip", "neigh", "show", "to", ip ],
                       stdout=Tac.CAPTURE ) )

   def debugKernelEntry( self, kentry ):
      arpRe = re.compile(
         r"^([A-Fa-f0-9:.]{0,46})\s+\S+\s+(\S+)\s+(\S+)$" )

      lines = kentry.strip().split( '\n' )
      for line in lines:
         m = arpRe.match( line )
         if m:
            _ = m.group( 1 )
            deviceName = m.group( 2 )
            state = m.group( 3 )
            return ( deviceName, state )
      return ( None, None )
         
   def arpStatusEntryPresent( self, ipAddr ):
      for intfName in self.staticArpStatusDir.ipv4:
         arpIntf = self.staticArpStatusDir.ipv4.get( intfName )
         if arpIntf:
            arpStatusEntry = arpIntf.arpEntry.get( ipAddr )
            if arpStatusEntry:
               return ( arpStatusEntry, intfName )
      return ( None, None )
            
   def verifyArpIp( self, ipAddr, vrf=DEFAULT_VRF ):
      fingerprint = ''
      data = None
      try:
         self.staticArpConfigDir = self.sysdb.entity.get(
               'arp/input/config/cli' ).vrf[ vrf ]
         self.staticArpStatusDir = self.sysdb.entity.get(
               'arp/input/status' ).vrf[ vrf ]
      except PyClientBase.RpcError as e:
         if not self.staticArpConfigDir or not self.staticArpStatusDir:
            fingerprint = 'Vrf %s configuration does not exist'
            data = vrf
            raise VerifyException( fingerprint, data )
         else:
            raise e
      try:
         arpConfig = self.staticArpConfigDir.ipv4.get( ipAddr )

         ( arpStatusEntry, intfName ) = self.arpStatusEntryPresent( ipAddr ) 
         arpKernelEntry = self.kernelEntryPresent( ipAddr, vrf=vrf )
         if ( intfName is not None ):
            arpSmashEntry = self.arpSmashLookup( intfName, ipAddr, vrfName=vrf )
         else:
            arpSmashEntry = None

         if not arpConfig:
            fingerprint = 'Arp %s configuration does not exist'
            data = ipAddr
            raise VerifyException( fingerprint, data )
         if not arpStatusEntry:
            fingerprint = 'Arp %s status does not exist'
            data = ipAddr
            raise VerifyException( fingerprint, data )
         if not arpSmashEntry:
            fingerprint = 'Arp %s does not exist in smash'
            data = ipAddr
            raise VerifyException( fingerprint, data )
         if arpKernelEntry == '':
            fingerprint = 'Arp %s  does not exist in kernel'
            data = ipAddr 
            raise VerifyException( fingerprint, data )
         if not 'PERMANENT' in arpKernelEntry:
            ( dev, state )  = self.debugKernelEntry( arpKernelEntry )
            fingerprint = 'Arp %s  failed to resolve in kernel on %s ( state=%s )' \
                          % ( ipAddr, dev, state )
            data = None
            raise NotApplicableException
         #print 'Verified ARP %s in VRF %s' % ( ipAddr, vrf )
      except VerifyException as e:
         raise e

   def verify( self, addrAndVrf ):
      ipAddr = addrAndVrf[ 0 ]
      vrf = addrAndVrf[ 1 ]
      if vrf == None:
         vrf = DEFAULT_VRF
      try:
         if ( ipAddr == 'ALL' ):
            totalEntries = len( self.arpSmash_.arpEntry )
            if totalEntries > ARP_VERIFY_MAX:
               print 'Skipped Arp Verification for %s entries' % totalEntries
               raise NotApplicableException
            neighborDirByIntfName = self.arpSmashToNeighborDir()

            # iterate over all ARP entries in each vrf.
            for vrf in [ DEFAULT_VRF ] + self.allVrfConfig.vrf.keys():
               self.staticArpConfigDir = self.sysdb.entity.get(
                     'arp/input/config/cli' ).vrf[ vrf ]

               for ipAddr in self.staticArpConfigDir.ipv4:
                  self.verifyArpIp( ipAddr, vrf=vrf )

               netNs = "ns-%s" % vrf if vrf != DEFAULT_VRF else DEFAULT_NS
               self.compareNeighborsAgainstKernel( neighborDirByIntfName,
                                      self.intfStatusKernel, netNs )
            neighborDirByIntfName = None
         else:
            self.verifyArpIp( ipAddr, vrf=vrf )
      except:
         raise
