#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os

AWS = 'AWS'
AZURE = 'Azure'
GCP = 'GCP'
KVM = 'KVM'
ESXI = 'ESXi'
BAREMETAL = 'BareMetal'
CEOSR = 'cEOSR'

## check if EOS is running inside docker container
def runningInDocker_():
   with open( '/proc/1/cgroup', 'r' ) as ifh:
      cgroupData = ifh.read()
      return os.path.exists( '/.dockerenv' ) and \
         any( v in cgroupData for v in [ 'docker', 'lxc', 'kubepods' ] )

isDocker_ = runningInDocker_()
def runningInDocker():
   return isDocker_

def getDmiInfo( propName ):
   propPath = os.path.join( '/', 'sys', 'class', 'dmi', 'id', propName )
   if not os.path.exists( propPath ):
      return None
   try:
      with open( propPath ) as f:
         return str( f.read() ).rstrip()
   except IOError:
      # It's possible to run in namespace dut using ssh in some stest scenerio.
      # Handle gracefully as we may not have permissions to access /sys subdir.
      return None

def getSysVendorInfo():
   return getDmiInfo( 'sys_vendor' )

def getProductUuid():
   return getDmiInfo( 'product_uuid' )

def getProductName():
   return getDmiInfo( 'product_name' )

def platformAWS():
   return getPlatform() == AWS

def platformAzure():
   return getPlatform() == AZURE

def platformGCP():
   return getPlatform() == GCP

def platformKVM():
   return getPlatform() == KVM

def platformESXi():
   return getPlatform() == ESXI

def platformBareMetal():
   return getPlatform() == BAREMETAL

def platformCeosR():
   return getPlatform() == CEOSR

# platformCloud() function returns True for cloud platforms and baremetal.
# To check only for cloud platforms use function platformCloudOnly().
def platformCloud():
   p = getPlatformCloudList() + [ BAREMETAL ]
   return getPlatform() in p

def platformCloudOnly():
   return getPlatform() in getPlatformCloudList()

def getPlatformCloudList():
   return [ AWS, AZURE, GCP ]

# Case insensitive compare sysVendor/productName/Uuid,
# since productUUid in 4.19 kernel on Aws is lowercase, while
# in older kernel is uppercase.
def getPlatform_():
   if runningInDocker():
      return CEOSR

   sysVendor = getSysVendorInfo()
   productName = getProductName()
   productUuid = getProductUuid()

   if sysVendor is not None:
      sysVendor = sysVendor.lower()

   if productName is not None:
      productName = productName.lower()

   if sysVendor == 'xen' or sysVendor == 'amazon ec2':
      if productUuid is not None and productUuid.lower()[ :3 ] == 'ec2':
         return AWS
   elif sysVendor == 'microsoft corporation':
      return AZURE
   # Nutanix uses KVM as its hypervisor.
   elif sysVendor == 'qemu' or productName == 'kvm' or sysVendor == 'nutanix':
      return KVM
   elif sysVendor == 'vmware, inc.':
      return ESXI
   elif sysVendor == 'google':
      return GCP
   elif sysVendor == 'advantech':
      return BAREMETAL
   return None

platform_ = getPlatform_()
def getPlatform():
   return os.environ.get( 'VEOS_SIM_ENVIRONMENT', platform_ )
