# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing

traceHandle = Tracing.Handle( 'UwsgiAaaCacheLib' )
warn = traceHandle.trace1
info = traceHandle.trace2
trace = traceHandle.trace3
debug = traceHandle.trace4

class _CacheEntry( object ):
   def __init__( self, obj ):
      self.obj_ = obj
      self.expiryTime_ = Tac.endOfTime
      self.usageCounter_ = 0

   def setExpiryTime( self, expiryTime ):
      ''' Sets the expiration time of this entry '''
      self.expiryTime_ = expiryTime

   def getExpiryTime( self ):
      ''' Gets the time expiration time of this entry '''
      return self.expiryTime_

   def incrUsageCount( self ):
      ''' Marks this entry as 'in use' so the object won't cleanup '''
      trace( '_CacheEntry.incrUsageCount Marking', self, 'as in use' )
      self.usageCounter_ += 1

   def isInUse( self ):
      ''' Returns True if this entry is currently being used. '''
      return self.usageCounter_ > 0

   def decrUsageCount( self ):
      ''' Releases the entry, so it might be considered for cleanup '''
      trace( '_CacheEntry.decrUsageCount Marking', self, 'as no longer used' )
      assert self.usageCounter_ > 0
      self.usageCounter_ -= 1
      trace( '_CacheEntry.decrUsageCount exiting', self, 'usage cnt',
             self.usageCounter_ )

   def getObject( self ):
      ''' Return the object in the cache '''
      trace( '_CacheEntry.getObject', self.obj_ )
      return self.obj_


class Cache( object ):
   ''' Generic Cache object. '''

   def __init__( self, entryCleanupFn=None, timeout=0, extendTimeout=False ):
      self.entries_ = {}
      self.entryCleanupFn_ = entryCleanupFn
      self.timeoutNotifiee_ = Tac.ClockNotifiee( handler=self._applyTimeouts,
                                                 timeMin=Tac.endOfTime )
      self.timeout_ = timeout
      self.extendTimeout_ = extendTimeout

   def cleanup( self ):
      ''' Uncondiontally releases all of the entries from the cache. This
          should only really be used for testing'''
      trace( 'Cache.cleanup entry' )
      # we need to call all of our entries cleanup funcs
      for key, entry in self.entries_.iteritems():
         trace( 'Entry', entry, 'in use:', entry.isInUse() )
         if self.entryCleanupFn_:
            trace( 'Cleaning up entry', entry )
            self.entryCleanupFn_( key, entry.getObject() )
      self.entries_.clear()

      # Update the timer now that we have no entries
      self.timeoutNotifiee_.timeMin = Tac.endOfTime
      trace( 'Cache.cleanup exit' )

   def get( self, key, incrementUsageCnt=True ):
      ''' Returns the Object corresponding to the given key.
          None if no Entry exists '''
      trace( 'Cache.get entry key:', key )
      if key not in self.entries_:
         return None
      
      entry = self.entries_[ key ]
      if incrementUsageCnt:
         entry.incrUsageCount()
      if self.extendTimeout_:
         entry.setExpiryTime( Tac.now() + self.timeout_ )
      trace( 'Cache.get exit', entry.getObject() )
      return entry.getObject()

   def insert( self, key, obj ):
      ''' Insert a new object in the cache '''
      trace( 'Cache.insert entry', obj )
      assert key not in self.entries_, '%s duplicate keys!' % key
      entry = _CacheEntry( obj )
      entry.setExpiryTime( Tac.now() + self.timeout_ )
      self.entries_[ key ] = entry
      trace( 'Cache.insert exit', obj )

   def release( self, key ):
      ''' Releases an entry in the cache '''
      trace( 'Cache.release entry', key )
      entry = self.entries_.get( key )
      if entry is None:
         return

      entry.decrUsageCount()
      self._maybeCleanupEntry( key )

      if self.hasKey( key ) and not entry.isInUse():
         # this means that this entry this entry was not cleaned up
         # but that it no longer has any users, that means we have
         # to adjust the timeout to be the min of what currently and
         # and this entry
         dueTime = min( self.timeoutNotifiee_.timeMin, entry.getExpiryTime() )
         self.timeoutNotifiee_.timeMin = dueTime
      trace( 'Cache.release exit', key )

   def cleanupEntry( self, key ):
      ''' Releases an entry in the cache '''
      trace( 'cleanupEntry entry', key )
      assert key in self.entries_
      entry = self.entries_[ key ]

      debug( 'Deleting entry for', entry )
      if self.entryCleanupFn_:
         self.entryCleanupFn_( key, entry.getObject() )
      del self.entries_[ key ]
  
      if not self.entries_:
         self.timeoutNotifiee_.timeMin = Tac.endOfTime
      trace( 'Cache.cleanupEntry exit', key )

   def getExpiryTime( self, key ):
      if key not in self.entries_:
         return None

      return self.entries_[ key ].getExpiryTime()

   def hasKey( self, key ):
      return key in self.entries_

   def _maybeCleanupEntry( self, key ):
      entry = self.entries_[ key ]
      if entry.isInUse():
         return

      if Tac.now() < entry.getExpiryTime():
         return

      # Only delete if it has expired from the cache, and the
      # entry is no longer in use. Otherwise we may be
      # deleting an entry for a user that is still running
      # commands (i.e. the request took longer than
      # gracePeriod)
      self.cleanupEntry( key )

   @Tac.withActivityLock
   def _applyTimeouts( self ):
      '''Timer callback. Discards stale entries and computes next callback
      time.'''
      trace( 'Cache._applyTimeouts entry' )
      for key in self.entries_.keys():
         self._maybeCleanupEntry( key )

      if not self.entries_:
         debug( 'No entries found in cache, skipping _updateTimer' )
         # There are no entries, so cancel the timer for now
         self.timeoutNotifiee_.timeMin = Tac.endOfTime
         return

      expiryTimes = tuple( e.getExpiryTime() for e in self.entries_.itervalues()
                           if not e.isInUse() )
      dueTime = min( expiryTimes ) if expiryTimes else Tac.endOfTime
      # We do not have to worry about dueTime being in the past.
      # In those cases, _applyTimeouts() will be called immediately.
      trace( 'Cache._updateTimer() setting dueTime=', dueTime )
      self.timeoutNotifiee_.timeMin = dueTime
      trace( 'Cache._applyTimeouts exit' )
