#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# Contains functions and objects relevant to the programming of a
# customer-programmable fpga

import Tac
import sys, os, fcntl

sysname = os.environ.get( "SYSNAME", "ar" )
MAGIC = 0xABCD0F35 # magic value to read and write back from scratch reg

class PidFileLock:
   ''' Context that, when entered, locks a pid file so that no other instances of
   this process can run.'''
   FILENAMEFMT = "/var/run/%s.pid"
   def __init__( self, utilName ):
      self.utilName_ = utilName
      self.pidFd_ = -1
   def __enter__( self ):
      self.pidFd_ = os.open( PidFileLock.FILENAMEFMT % self.utilName_,
                             os.O_CREAT | os.O_RDWR )
      try:
         fcntl.flock( self.pidFd_, fcntl.LOCK_EX | fcntl.LOCK_NB )
      except IOError:
         print "%s with pid %s already running" % \
             ( self.utilName_, os.read( self.pidFd_, 50 ) )
         sys.exit( 1 )
      os.write( self.pidFd_, "%d" % os.getpid() )
      os.fsync( self.pidFd_ )
   def __exit__( self, exc_type, exc_value, traceback ):
      os.close( self.pidFd_ )

class FpgaProgrammerConfig( object ):
   def __init__( self, fpgaImgPath, fpgaId, fpgaName, scdSystem, scdConfig=None ):
      '''Takes a path to the image, the fpga id that selects it in hardware,
      the name of the fpga (in the scdConfig), the name of the scd, and an
      optional hardware scd config.'''
      self.fpgaImgPath_ = fpgaImgPath
      self.fpgaSelect_ = fpgaId
      self.fpgaName_ = fpgaName
      self.scdSystem_ = scdSystem
      if scdConfig is None:
         import EntityManager, Cell
         em = EntityManager.Sysdb( os.environ.get( 'SYSNAME', 'ar' ) )
         mg = em.mountGroup()
         config = mg.mount( "hardware/cell/%d/scd/config" % Cell.cellId(),
                            "Hardware::Scd::Config", 'r' )
         mg.mount( 'cell/%d/hardware/pciDeviceStatusDir' % Cell.cellId(),
                   'Hardware::PciDeviceStatusDir', 'r' )
         mg.close( blocking=True )
         scdConfig = config.scdConfig[ scdSystem ]
      self.scdConfig_ = scdConfig
      self.fpgaConfig_ = self.scdConfig_.userFpgaConfig[ self.fpgaName_ ]

   def fpgaImgPath( self ):
      return self.fpgaImgPath_
   def fpgaSelect( self ):
      return self.fpgaSelect_
   def fpgaName( self ):
      return self.fpgaName_
   def scdConfig( self ):
      return self.scdConfig_
   def fpgaConfig( self ):
      return self.fpgaConfig_

def doProgramFpga( config ):
   ''' Programs the user fpga. Returns whether or not the fpga is programmed.'''
   if not os.path.exists( config.fpgaImgPath() ):
      print 'Error: cannot find specified fpga image %s' % config.fpgaImgPath()
      return False

   print 'Beginning Programming.'
   hardwareAccessMethod = Tac.newInstance(
      'Fru::Ham', 'userFpgaHam', 'hamTypeRecursive', config.scdConfig().ham,
      config.fpgaConfig().programmingBlock.ctrlRegAddr, 0, '' )
   programmer = Tac.newInstance( 'Hardware::Fpga::ProgrammerSm',
                                 None, 
                                 Tac.activityManager.clock,
                                 hardwareAccessMethod,
                                 config.fpgaImgPath(),
                                 int( config.fpgaSelect() ),
                                 'FpgaProgrammer' )
   Tac.waitFor( lambda: programmer.done != 'running', 60.0,
                description='programming complete' )
   if programmer.done == 'failed':
      print 'Programming failed'
      return False
   # Verifying Result
   if config.fpgaConfig().programmingBlock.programmingDone.asserted:
      print 'Image verification successful'
   else:
      print 'Image verification failed'
      return False
   return True

def validateFpgaSetup( config ):
   '''Returns true if the fpga was set up properly.'''
   ham = config.scdConfig().ham
   # Read and print revision
   rev = ham.hamImpl.data[ config.fpgaConfig().revisionRegOffset ]
   # if the revision is 0 or all f's, it is an error
   if rev == 0 or rev == 0xffffffff:
      print "Bad FPGA revision number: %d" % rev
      return False
   print '%s FPGA Revision: %d' % ( config.fpgaName(), rev )
   ham.hamImpl.data[ config.fpgaConfig().scratchRegOffset ] = 0x0
   ham.hamImpl.data[ config.fpgaConfig().scratchRegOffset ] = MAGIC
   if ham.hamImpl.data[ config.fpgaConfig().scratchRegOffset ] != MAGIC:
      print "Failed to validate fpga register interface. Is it present?"
      return False
   return True
