# Copyright (c) 2005-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""This module implements the "system" URL scheme."""

import errno
import os
import sys

import BasicCliModes
import CliSave
from FlashUrl import LocalFilesystem, LocalUrl
import Url

def _getRunningConfig( entityManager, dstFn,
                       saveAll, saveAllDetail, showSanitized, showJson,
                       showNoSeqNum,
                       secureMonitor=False,
                       showProfileExpanded=False,
                       intfFilter=None ):
   # _getRunningConfig() is called only from SysdbUrl which is a UrlPlugin
   # loaded by ConfigAgent through the Cli. Since the ConfigAgent is the one
   # processing the 'show running*' cli command, all the state that is necessary
   # is present locally. So just call the CliSave.saveRunningConfig() to
   # generate the running config.
   with file( dstFn, 'wb' ) as dstOutput:
      CliSave.saveRunningConfig( entityManager, dstOutput,
                                 saveAll, saveAllDetail,
                                 showSanitized, showJson,
                                 showNoSeqNum,
                                 secureMonitor=secureMonitor,
                                 showProfileExpanded=showProfileExpanded,
                                 intfFilter=intfFilter )

class SystemUrl( LocalUrl ):
   def listdir( self ):
      if not self.isdir():
         raise EnvironmentError( errno.ENOTDIR, os.strerror( errno.ENOTDIR ) )
      assert self.pathname == '/'

      return [ 'running-config' ]

   def _options( self ):
      '''Return all options as a namedtuple if the syntax is valid, else None.'''
      if not self.pathname.startswith( '/running-config' ):
         return None
      paths = set( self.pathname[ len( '/running-config:' ): ].split( ':' ) )

      showAll = 'all' in paths
      showAllDetail = showAll and 'detail' in paths
      showNoSeqNum = 'noseqnum' in paths
      showSanitized = 'sanitized' in paths
      showJson = 'json' in paths
      secureMonitor = 'secure-monitor' in paths
      showProfileExpanded = 'profile-expanded' in paths
      intfFilter = None

      if secureMonitor and not ( self.context.cliSession and
                                 self.context.cliSession.secureMonitor() ):
         # normal user cannot view secure-monitor configuration
         return None

      for p in paths:
         if p.startswith( 'interface@' ):
            intfFilter = set( p[ len( 'interface@' ): ].split( ',' ) )
            break

      return CliSave.ShowRunningConfigOptions( showAll, showAllDetail, showSanitized,
                                               showJson, showNoSeqNum,
                                               secureMonitor,
                                               showProfileExpanded,
                                               intfFilter )

   def exists( self ):
      return self.pathname == '/' or self._options() is not None

   def isdir( self ):
      return self.pathname == '/'

   def islink( self ):
      return False

   def historyEntryName( self ):
      if self.pathname.startswith( '/running-config' ):
         return "running"
      return None

   def size( self ):
      if self.isdir():
         return 0
      f = self.open()
      try:
         return len( f.read() )
      finally:
         f.close()

   def date( self ):
      if not self.exists():
         raise EnvironmentError( errno.ENOENT, os.strerror( errno.ENOENT ) )
      return 0

   def permission( self ):
      if not self.exists():
         raise EnvironmentError( errno.ENOENT, os.strerror( errno.ENOENT ) )
      if self.isdir():
         return ( True, True, True, True )
      return ( False, True, True, False )

   def get( self, dstFn ):
      options = self._options()
      if options is None:
         raise EnvironmentError( errno.ENOENT, os.strerror( errno.ENOENT ) )
      if self.isdir():
         raise EnvironmentError( errno.EISDIR, os.strerror( errno.EISDIR ) )

      # make sure we create the file( if not already created ) from this
      # process so the file permissions are correct ( uid of this process )
      fd = file( dstFn, 'wb' )
      fd.close()

      _getRunningConfig( self.context.entityManager, dstFn,
                         options.saveAll, options.saveAllDetail,
                         options.showSanitized, options.showJson,
                         options.showNoSeqNum,
                         secureMonitor=options.secureMonitor,
                         showProfileExpanded=options.showProfileExpanded,
                         intfFilter=options.intfFilter )

   def put( self, srcFn, append=False ):
      srcFile = file( srcFn, 'rb' )
      try:
         if not self.exists():
            raise EnvironmentError( errno.EACCES, os.strerror( errno.EACCES ) )
         assert not self.isdir()
         assert self.pathname == '/running-config'
         import Cli
         Cli.loadConfigFromFile( srcFile, self.context.entityManager,
                                 disableAaa=self.context.disableAaa,
                                 initialModeClass=BasicCliModes.GlobalConfigMode,
                                 disableGuards=True,
                                 secureMonitor=\
                                 self.context.cliSession.secureMonitor(),
                                 aaaUser=self.context.cliSession.aaaUser(),
                                 privLevel=self.context.cliSession.privLevel_ )
      finally:
         srcFile.close()

   renameto = LocalUrl._notSupported
   delete = LocalUrl._notSupported
   mkdir = LocalUrl._notSupported
   rmdir = LocalUrl._notSupported

   def ignoreTrailingWhitespaceInDiff( self ):
      return True

class TerminalUrl( LocalUrl ):

   def get( self, dstFn ):
      print "enter input line by line; when done enter one or more control-d"
      sys.stdout.flush()
      data = sys.stdin.read()
      dstFd = file( dstFn, 'wb' )
      dstFd.write( data )
      dstFd.close()

   def put( self, srcFn, append=False ):
      srcFd = file( srcFn, 'rb' )
      data = srcFd.read()
      srcFd.close()
      print data

   # this is needed in spite of supportsListing() = False
   def listdir( self ):
      return []

   def exists( self ):
      return True

   def isdir( self ):
      return self.pathname == '/'

   def islink( self ):
      return False

   def size( self ):
      return 0

   def date( self ):
      return 0

   def permission( self ):
      return ( True, True, True, True )

   renameto = LocalUrl._notSupported
   delete = LocalUrl._notSupported
   mkdir = LocalUrl._notSupported
   rmdir = LocalUrl._notSupported


# This scheme is for providing a file interactively, at the terminal, that is,
# it maps to file:/dev/fd/0. It can also be used in the output direction to
# dump stuff to the screen ("copy file:adsf terminal:" = "more file:asdf")
class TerminalFilesystem( LocalFilesystem ):
   urlClass_ = TerminalUrl

   def __init__( self, scheme ):
      LocalFilesystem.__init__( self, scheme, 'terminal', 'rw',
                                         noPathComponent=True )

   def supportsListing( self ):
      return False

class SystemFilesystem( LocalFilesystem ):
   urlClass_ = SystemUrl

   def __init__( self, scheme ):
      LocalFilesystem.__init__( self, scheme, 'system', 'rw' )

   def supportsListing( self ):
      return True

def Plugin( context=None ):
   Url.registerFilesystem( SystemFilesystem( 'system:' ) )
   Url.registerFilesystem( TerminalFilesystem( 'terminal:' ) )

