#!/usr/bin/env python
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import FileReplicationCmds, Tac, Fru
import Url, FlashUrl
import errno, os, re, datetime

# In breadth test mode, loopback == True.
if os.environ.has_key( 'SIMULATION_VMID' ):
   loopback = True
else:
   loopback = False

listDirRegex = re.compile( '^\S+ +\S+ +\S+ +\S+ +\d+ +\S+ +\S+ +(\S+)',
                           re.MULTILINE )

lsErrorRegex = re.compile( '^ls: cannot open directory \S+: Permission denied$' )

class PeerSupeUrl( FlashUrl.LocalUrl ):
   def __init__( self, fs, url, rawpathname, pathname, context, cmdCache=None ):
      self.fileInfoCache_ = None
      if cmdCache is None:
         self.cmdCache_ = {}
      else:
         self.cmdCache_ = cmdCache
      FlashUrl.LocalUrl.__init__( self, fs, url, rawpathname, pathname, context )

   def _run( self, cmd, useCache=False, ignoreReturnCode=True ):
      # Run a command using Tac.run, caching the output. Use the cache
      # instead of running the command if useCache=True
      cmdString = " ".join( cmd )

      if cmdString not in self.cmdCache_ or not useCache:
         stdout = os.tmpfile()
         stderr = os.tmpfile()
         Tac.run( cmd, stdout=stdout, stderr=stderr, asRoot=True,
                  ignoreReturnCode=ignoreReturnCode )
      
         stdout.seek( 0 )
         stderr.seek( 0 )
         self.cmdCache_[ cmdString ] = ( stdout.read(), stderr.read() )
      return self.cmdCache_[ cmdString ]
   
   def _fileinfo( self ):
      # Parse the output of ls -l --time-style=long-iso, searching for this filename.
      # If the filename is found, return a dictionary of permissions, date, size, and
      # file type. Cache the results in this object since _fileinfo() may be called
      # several times.
      if self.fileInfoCache_ is not None:
         return self.fileInfoCache_

      path = self.pathname
      if path != '/':
         path = path.rstrip( '/' )
         folder = os.path.dirname( path )
         filename = os.path.basename( path )
      else:
         folder = '/'
         filename = '.'
      filename = re.escape( filename )

      cmd = FileReplicationCmds.listDir( Fru.slotId(), folder,
                                         loopback=loopback, useKey=True )
      output = self._run( cmd, useCache=True )[ 0 ]
      
      regex = '^(\S+) +\S+ +\S+ +\S+ +(\d+) +(\S+ +\S+) +%s$' % filename
      match = re.search( regex, output, re.MULTILINE )
      
      if not match:
         return None
      
      # File type
      filetype = match.group( 1 )[ 0 ]
      if filetype == 'l':
         filetype = 'link'
      elif filetype == 'd':
         filetype = 'directory'
      else:
         filetype = 'file'

      # Date
      filedate = match.group( 3 )
      filedate = datetime.datetime.strptime( filedate, '%Y-%m-%d %H:%M' )

      self.fileInfoCache_ =  { 'filetype' : filetype,
                               'size' : int( match.group( 2 ) ),
                               'date' : filedate,
                               'permission' : match.group( 1 ) }
      return self.fileInfoCache_

   def listdir( self ):
      # Return a list of strings corresponding to all files+directories (except
      # hidden) in this url (if it's a directory).
      lscmd = FileReplicationCmds.listDir( Fru.slotId(), self.pathname,
                                           loopback=loopback, useKey=True )
      
      output = self._run( lscmd )
      # Check output for errors
      if( lsErrorRegex.match( output[ 1 ] ) ):
         raise OSError( errno.EACCES, os.strerror( errno.EACCES ) )
      
      fileNames = listDirRegex.findall( output[ 0 ] )
      return filter( lambda x: not x == '.' and not x == '..', fileNames )

   def child( self, f ):
      # For a file f in this directory, return a url for the file.
      # Note that this is the same as the method from LocalUrl, but it
      # passes the command cache to the new url as well.
      assert not '/' in f
      childPathname = os.path.join( self.pathname, f )
      return self.__class__( self.fs, 
                             self.fs.scheme + childPathname,
                             childPathname,
                             childPathname,
                             self.context,
                             cmdCache=self.cmdCache_ )

   def parent( self ):
      # Return a url for the parent directory of this url.
      # Note that this is the same as the method from LocalUrl, but it
      # passes the command cache to the new url as well.
      parentPathname = os.path.dirname( self.pathname )
      return self.__class__( self.fs,
                             self.fs.scheme + parentPathname,
                             parentPathname,
                             parentPathname,
                             self.context,
                             cmdCache=self.cmdCache_ )

   def isdir( self ):
      fileinfo = self._fileinfo()
      return fileinfo and fileinfo[ 'filetype' ] == 'directory'

   def islink( self ):
      fileinfo = self._fileinfo()
      return fileinfo and fileinfo[ 'filetype' ] == 'link'

   def exists( self ):
      return self._fileinfo() is not None
   
   def date( self ):
      fileinfo = self._fileinfo()
      if fileinfo is None:
         return 0
      return float( fileinfo[ 'date' ].strftime( '%s' ) )

   def permission( self ):
      fileinfo = self._fileinfo()
      if fileinfo is None:
         return ( False, False, False, False )
      perm = fileinfo[ 'permission' ]
      return ( perm[ 0 ] == 'd',
               perm[ 1 ] != '-',
               perm[ 2 ] != '-',
               perm[ 3 ] != '-' )
   
   def size( self ):
      fileinfo = self._fileinfo()
      if fileinfo is None:
         return 0
      return fileinfo[ 'size' ]

   def verifyHash( self, hashName, mode=None, hashInitializer=None ):
      cmd = FileReplicationCmds.verifyHash( Fru.slotId(), self.pathname, hashName,
                                        loopback=loopback, useKey=True )
      output = self._run( cmd ) 
      hashValue = output[ 0 ].split()[ 0 ]
      return hashValue

   def renameto( self, dst ):
      assert dst.fs == self.fs
      if dst.exists():
         raise EnvironmentError( errno.EEXIST, os.strerror( errno.EEXIST ) )
      cmd = FileReplicationCmds.rename( Fru.slotId(), dst.pathname,
                                        self.pathname, loopback=loopback,
                                        useKey=True )
      self._run( cmd )
      
   def get( self, dstFn ):
      cmd = FileReplicationCmds.copyFile( Fru.slotId(), dstFn,
                                          self.pathname, loopback=loopback,
                                          useKey=True, peerSource=True )
      self._run( cmd )

   def put( self, srcFn, append=False ):
      cmd = FileReplicationCmds.copyFile( Fru.slotId(), self.pathname,
                                          source=srcFn, loopback=loopback,
                                          useKey=True )
      self._run( cmd )
      
   def open( self ):
      if self.isdir():
         raise EnvironmentError( errno.EISDIR, os.strerror( errno.EISDIR ) )
      return FlashUrl.LocalUrl.open( self )

   def copyfrom( self, srcUrl ):
      if srcUrl.isdir():
         raise EnvironmentError( errno.EISDIR, os.strerror( errno.EISDIR ) )
      # When available, use the srcUrl's localFileName(). For example,
      # a FlashUrl "flash:/" will have a pathname of "/" but a
      # localFilename() of "/mnt/flash/". In this case, we want to use
      # the localFilename().
      if srcUrl.localFilename():
         sourcePath = srcUrl.localFilename()
      else:
         sourcePath = srcUrl.pathname
      cmd = FileReplicationCmds.copyFile( Fru.slotId(), self.pathname,
                                          source=sourcePath,
                                          loopback=loopback, useKey=True )
      try:
         Tac.run( cmd, stderr=Tac.CAPTURE, stdout=Tac.DISCARD, asRoot=True )
      except Tac.SystemCommandError, e:
         # Remove 'Warning: Permanently added ... to list of known hosts'
         # and trailing newline from output shown to the user
         userError = e.output.strip()
         if userError.startswith( 'Warning: Permanently added' ):
            userError = userError.split( '\n', 1 )[ 1 ]
         raise EnvironmentError( 0, userError )

   def mkdir( self ):
      cmd = FileReplicationCmds.createDir( Fru.slotId(), self.pathname,
                                           loopback=loopback, useKey=True )
      self._run( cmd )
      
   def rmdir( self ):
      if not self.isdir():
         raise EnvironmentError( errno.ENOTDIR, os.strerror( errno.ENOTDIR ) )
      elif self.pathname == '/':
         raise EnvironmentError( errno.EPERM, os.strerror( errno.EPERM ) )
      elif self.listdir():
         raise EnvironmentError( errno.ENOTEMPTY, os.strerror( errno.ENOTEMPTY ) )
      self.delete( True )
      
   def delete( self, recursive ):
      if not recursive and self.isdir():
         raise EnvironmentError( errno.EISDIR, os.strerror( errno.EISDIR ) )
      cmd = FileReplicationCmds.deleteFile( Fru.slotId(), self.pathname,
                                            loopback=loopback, useKey=True,
                                            recursive=recursive )
      self._run( cmd )

class PeerSupeFilesystem( Url.Filesystem ):
   urlClass_ = PeerSupeUrl
   path_ = None
   
   def __init__( self, scheme, location, fsType='peer' ):
      Url.Filesystem.__init__( self, scheme, fsType, 'rw' )
   
   def parseUrl( self, url, rest, context ):
      pathname = os.path.join( '/', rest )
      self.path_ = pathname
      return self.urlClass_( self, url, rest, pathname, context )
   
   def supportsListing( self ):
      return True
   
   def supportsRename( self ):
      return True

   def supportsDelete( self ):
      return True

   def supportsMkdir( self ):
      return True

   def supportsAppend( self ):
      return False
   
   def stat( self ):
      path = "/"
      if self.path_ is not None:
         path = self.path_
      cmd = FileReplicationCmds.stat( Fru.slotId(), path,
                                      loopback=loopback, useKey=True )
      output = Tac.run( cmd, stdout=Tac.CAPTURE, stderr=Tac.DISCARD,
                        ignoreReturnCode=True, asRoot=True )
      
      bsMatch = re.search( 'Block size: (\d+)', output )
      sizeMatch = re.search( 'Blocks: +Total: +(\d+).* +Available: +(\d+)', output )

      if not bsMatch or not sizeMatch:
         return ( 0, 0 )

      bs = int( bsMatch.group( 1 ) )
      size = int( sizeMatch.group( 1 ) ) * bs
      free = int( sizeMatch.group( 2 ) ) * bs

      return ( size, free )

def Plugin( context=None ):
   pass
