#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import errno
import os
import time
import cStringIO
import FileUrlDefs
import SecMonUtil
from UrlPlugin.FlashUrl import FlashUrl, FlashFilesystem
import UtmpDump

class StartupConfigUrl( FlashUrl ):
   headerPrefix_ = '! Startup-config last modified at '

   def historyEntryName( self ):
      return "startup"

   def getHeader( self ):
      # Return string formatted with save time and user
      # that will be inserted in the startup-config.
      info = UtmpDump.getUserInfo()
      saveTime = time.asctime( time.localtime() )
      return '%s%s by %s\n' % ( self.headerPrefix_, saveTime, info[ 'user' ] )

   def ignoreTrailingWhitespaceInDiff( self ):
      return True

class SecureMonitorStartupConfigUrl( StartupConfigUrl ):
   headerPrefix_ = '! Secure-monitor startup-config last modified at '

   def encrypted( self ):
      return True

   def encrypt( self, content ):
      return SecMonUtil.encryptfile( content )

   def open( self ):
      self._checkSecureMonitor()
      content = SecMonUtil.readfile( self.localFilename() )
      return cStringIO.StringIO( content )

   def _checkSecureMonitor( self ):
      if not ( self.context.cliSession and self.context.cliSession.secureMonitor() ):
         if ( self.context.entityManager is None and self.context.disableAaa and
              not self.context.cliSession ):
            # this is LoadConfig
            return
         raise EnvironmentError( errno.EPERM, os.strerror( errno.EPERM ) )

   def checkOpSupported( self, supportsFunc ):
      # piggyback on this function
      StartupConfigUrl.checkOpSupported( self, supportsFunc )
      self._checkSecureMonitor()

   def get( self, dstFn ):
      self._checkSecureMonitor()
      content = SecMonUtil.readfile( self.localFilename() )
      with open( dstFn, "wb" ) as f:
         f.write( content )

   # Don't allow direct writing - only copy command is allowed
   # which will use writeLocalFile().
   put = StartupConfigUrl._notSupported

def Plugin( context=None ):
   FlashFilesystem.registerClass(
      FileUrlDefs.STARTUP_CONFIG_FILE_NAME,
      StartupConfigUrl )
   FlashFilesystem.registerClass(
      FileUrlDefs.SECURE_MONITOR_STARTUP_CONFIG_FILE_NAME,
      SecureMonitorStartupConfigUrl )
