# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import errno
import os

import Tac
import DefaultSslProfile
import Url
from UrlPlugin.FlashUrl import FlashUrl
from UrlPlugin.FlashUrl import FlashFilesystem

sslConstants_ = Tac.Type( "Mgmt::Security::Ssl::Constants" )

def _isStandby( entityManager ):
   redundancyStatus = getattr( entityManager, 'redundancyStatus', None )
   if redundancyStatus and redundancyStatus().mode != 'active':
      return True
   return False

def _getUrlLogInfo( url ):
   import SslCertKey
   return SslCertKey.getLogActionAndFileHash( url.localFilename(),
                                              url.fs.scheme, "imported" )

def _generateSysLog( url, action, oldFileHash='' ):
   import SslCertKey
   SslCertKey.generateSslKeyCertSysLog( url.localFilename(), url.fs.scheme,
                                        action, oldFileHash=oldFileHash )

class CliFileSystemUrl( FlashUrl ):
   def _checkStandby( self ):
      if ( self.fs.entityManager and _isStandby( self.fs.entityManager ) ):
         raise EnvironmentError( errno.EOPNOTSUPP, 
                                 "Operation not supported on this supervisor" )

   def put( self, srcFn, append=False ):
      self._checkStandby()
      FlashUrl.put( self, srcFn, append )
      self.fs.handleFileChange( self.context )

   def renameto( self, dst ):
      self._checkStandby()
      FlashUrl.renameto( self, dst )
      self.fs.handleFileChange( self.context )
      
   def delete( self, recursive ):
      self._checkStandby()
      FlashUrl.delete( self, recursive )
      self.fs.handleFileChange( self.context )
      
   def empty( self ):
      self._checkStandby()
      FlashUrl.empty( self )
      self.fs.handleFileChange( self.context )

   def writeLocalFile( self, srcUrl, filename ):
      self._checkStandby()
      FlashUrl.writeLocalFile( self, srcUrl, filename )
      self.fs.handleFileChange( self.context )
   
   def listdir( self ):
      return os.listdir( self.realFilename_ )
      
class SslFilesystemUrl( CliFileSystemUrl ):
   def isBuiltinFile( self ):
      builtinFiles = [ DefaultSslProfile.ARISTA_ROOT_CA_FILE,
                       DefaultSslProfile.ARISTA_SIGNING_CA_FILE ]
      return self.basename() in builtinFiles

   def addCliWarning( self, message ):
      if self.context and self.context.cliSession:
         mode = self.context.cliSession.mode
         mode.addWarning( message )

   def put( self, srcFn, append=False ):
      logAction, oldFileHash = _getUrlLogInfo( self )
      CliFileSystemUrl.put( self, srcFn, append )
      _generateSysLog( self, logAction, oldFileHash )

   def renameto( self, dst ):
      _, oldFileHash = _getUrlLogInfo( self )
      dstLogAction, dstOldFileHash = _getUrlLogInfo( dst )
      CliFileSystemUrl.renameto( self, dst )
      _generateSysLog( self, "deleted", oldFileHash )
      _generateSysLog( dst, dstLogAction, dstOldFileHash )

   def delete( self, recursive ):
      if self.isBuiltinFile():
         self.addCliWarning( "Not deleting built-in file %s" % self.basename() )
         return
      _, oldFileHash = _getUrlLogInfo( self )
      CliFileSystemUrl.delete( self, recursive )
      _generateSysLog( self, "deleted", oldFileHash )

   def empty( self ):
      _, oldFileHash = _getUrlLogInfo( self )
      CliFileSystemUrl.empty( self )
      _generateSysLog( self, "deleted", oldFileHash )

   def writeLocalFile( self, srcUrl, filename ):
      logAction, oldFileHash = _getUrlLogInfo( self )
      CliFileSystemUrl.writeLocalFile( self, srcUrl, filename )
      _generateSysLog( self, logAction, oldFileHash )

class CliFileSystem( FlashFilesystem ):
   """
   This is a base class for CLI file system. A CLI file system isn't really a file
   system; It's supposed to appear as one, to CLI users in the context of CLI
   commands. Examples: sslkey:, certificate:, ssh-ca-key:, ssh-host-cert: etc.

   Internally it maps to a directory such as /persist/secure/*. It avoids exposing
   internal paths to CLI users, and you can do file-system-specific validation of
   files. On a dual Sup, the files in /persist path are replicated from active Sup to
   standby Sup by agent 'FileReplicator'. Hence the copy/rename/delete commands on a
   CliFileSystemUrl are not allowed on standby Sup.
   """

   urlClass_ = CliFileSystemUrl

   def realFileSystem( self ):
      return True
   
   def stat( self ):
      return ( 0, 0 )
   
   def supportsMkdir( self ):
      return False

   def supportsListing( self ):
      return True

   def handleFileChange( self, context ):
      """This is the base class implementation. Derived classes may override 
      to be updated when the files pointed to by @affectedFiles has been changed.
      Currently this is called after the file has been updated in put, rename,
      delete, and copy operations."""
      pass

class SslFileSystem( CliFileSystem ):

   urlClass_ = SslFilesystemUrl

   def __init__( self, scheme, rawlocation, permission, mask ):
      CliFileSystem.__init__( self, scheme, rawlocation, permission=permission,
                              fsType='ssl', mask=mask )
   
class SslCertFilesystem( SslFileSystem ):
   def __init__( self, scheme, rawlocation ):
      SslFileSystem.__init__( self, scheme, rawlocation, 'rw', 0002 )

   def validateFile( self, filename, durl=None, context=None ):
      import SslCertKey
      try:
         SslCertKey.validateCertificateOrCrl( filename )
      except SslCertKey.SslCertKeyError as e:
         raise EnvironmentError( 0, e )

class SslKeyFilesystem( SslFileSystem ):
   def __init__( self, scheme, rawlocation ):
      SslFileSystem.__init__( self, scheme, rawlocation, 'w', 0006 )

   def validateFile( self, filename, durl=None, context=None ):
      import SslCertKey
      try:
         SslCertKey.validateRsaPrivateKey( filename )
      except SslCertKey.SslCertKeyError as e:
         raise EnvironmentError( 0, e )
         
def Plugin( context=None ):
   Url.registerFilesystem( SslCertFilesystem( 'certificate:', 
                                              sslConstants_.certsDirPath() ) )
   Url.registerFilesystem( SslKeyFilesystem( 'sslkey:', 
                                              sslConstants_.keysDirPath() ) )
