# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Url
from UrlPlugin.SslUrl import CliFileSystem

import Tac
import SshCertLib

sshConstants_ = Tac.Type( "Mgmt::Ssh::Constants" )
sshCertFileType_ = Tac.Type( "Mgmt::Ssh::SshCertFileType" )

class SshFileSystem( CliFileSystem ):
   def __init__( self, scheme, rawlocation, permission, mask ):
      CliFileSystem.__init__( self, scheme, rawlocation, fsType='ssh', 
                              permission=permission, mask=mask )

   def handleFileChange( self, context ):
      # configReq is mounted in CliPlugin/Ssh.py
      if context and context.entityManager:
         root = context.entityManager.root()
         configReq = root.entity[ "mgmt/ssh/configReq" ]
         configReq.sshCertFileUpdateRequest = Tac.now()

class SshHostCertFilesystem( SshFileSystem ):
   def __init__( self ):
      SshFileSystem.__init__( self,
                              scheme='ssh-host-cert:', 
                              rawlocation=sshConstants_.hostCertsDirPath(),
                              permission='rw', mask=0002 )

   def validateFile( self, filename, durl=None, context=None ):
      try:
         SshCertLib.validateHostCert( filename )
      except SshCertLib.SshHostCertError as e:
         raise EnvironmentError( 0, e )

class SshCaKeyFilesystem( SshFileSystem ):
   def __init__( self ):
      SshFileSystem.__init__( self, 
                              scheme='ssh-ca-key:', 
                              rawlocation=sshConstants_.caKeysDirPath(),
                              permission='rw', mask=0002 )

   def validateFile( self, filename, durl=None, context=None ):
      try:
         SshCertLib.validateMultipleKeysFile( filename )
      except SshCertLib.SshKeyError as e:
         raise EnvironmentError( 0, e )

class SshRevokeListFilesystem( SshFileSystem ):
   def __init__( self ):
      SshFileSystem.__init__( self, 
                              scheme='ssh-revoke-list:', 
                              rawlocation=sshConstants_.revokeListsDirPath(),
                              permission='rw', mask=0002 )

   def validateFile( self, filename, durl=None, context=None ):
      try:
         SshCertLib.validateMultipleKeysFile( filename )
      except SshCertLib.SshKeyError as e:
         raise EnvironmentError( 0, e )
       
def Plugin( context=None ):
   Url.registerFilesystem( SshCaKeyFilesystem() )
   Url.registerFilesystem( SshHostCertFilesystem() )
   Url.registerFilesystem( SshRevokeListFilesystem() )
