#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""This module implements the "session" URL scheme.
Valid pathnames are all of the form 
'/<session-name>-session-config[-<options>]'
"""

import hashlib
import CliCommon
import FileCliUtil
import CliSession as CS
import SessionUrlUtil as SUU
import Url, FlashUrl, CliSave, ConfigMount
import os, errno, re
from cStringIO import StringIO
import SystemUrl

class SessionUrl( SystemUrl.SystemUrl ):

   def __init__( self, fs, url, rawpathname, pathname, context, clean=False ):
      SystemUrl.SystemUrl.__init__( self, fs, 
                                    url, rawpathname, pathname, context )
      self.sessionName_ = ""
      self.clean_ = clean
      self.valid_ = self.validPathname()
      self.raiseOnException = False
      self.skipConfigCheck = False
      self.loadAsStartupConfig = False
      # The expected md5sum of the file content, if verification required, else None.
      self.md5 = None
      if self.clean_:
         self.sessionName_ = "clean"
      else:
         m = re.match( r"/(\S+)-session-config\S*", self.pathname )
         if m:
            self.sessionName_ = m.group( 1 )

   def ignoreTrailingWhitespaceInDiff( self ):
      return True

   def validPathname( self ):
      if not self.pathname[0] == "/":
         return False
      if not '-session-config' in self.pathname:
         return False
      return True

   def listdir( self ):
      if not self.isdir():
         raise EnvironmentError( errno.ENOTDIR, os.strerror( errno.ENOTDIR ) )
      names = CS.sessionNames( self.context.entityManager )
      return ([ "clean-session-config" ] + 
              map( lambda name : '%s-session-config' % name, names ))

   def _options( self ):
      '''Return all options as a namedtuple if the syntax is valid, else None.'''
      if not self.valid_:
         return None
      basePathname = '/%s-session-config' % self.sessionName_ 
      if self.pathname == basePathname:
         return CliSave.ShowRunningConfigOptions( False, False, False, False, False )
      paths = set( self.pathname[ ( len( basePathname ) + 1 ): ].split( ':' ) )

      showAll = 'all' in paths
      showAllDetail = showAll and 'detail' in paths
      showNoSeqNum = 'noseqnum' in paths
      showSanitized = 'sanitized' in paths
      showJson = 'json' in paths

      if showAllDetail: # should be first
         return CliSave.ShowRunningConfigOptions( True, True, showSanitized,
                                                  showJson, showNoSeqNum )
      elif showAll:
         return CliSave.ShowRunningConfigOptions( True, False, showSanitized,
                                                  showJson, showNoSeqNum )
      else:
         return CliSave.ShowRunningConfigOptions( False, False, showSanitized,
                                                  showJson, showNoSeqNum )
      assert self.isdir()
      return None

   def exists( self ):
      return self.isdir() or self.valid_

   def historyEntryName( self ):
      if self.valid_ and not self.isdir():
         return "running" # not sure if this is accurate
      return None

   def get( self, dstFn ):
      import Tac

      options = self._options()
      if options is None:
         raise EnvironmentError( errno.ENOENT, os.strerror( errno.ENOENT ) )
      if self.isdir():
         raise EnvironmentError( errno.EISDIR, os.strerror( errno.EISDIR ) )

      entityManager = self.context.entityManager

      # Store the config in a temporary string, in case we get an exception
      # or the session gets disconnected while rendering the config.
      dstString = StringIO()
      if not entityManager.local():
         sStatus = SUU.getSessionStatus( entityManager )
         SUU.doMountsForSaveSession( entityManager, sStatus, None )
         sleep = not Tac.activityManager.inExecTime.isCurrent
         Tac.waitFor( SUU.saveSessionMountsDone, sleep=sleep,
                      description="preparation for saveSessionConfig" )

      SUU.saveSessionConfig( entityManager, self.sessionName_, dstString,
                             options.saveAll, options.saveAllDetail,
                             showSanitized=options.showSanitized,
                             clean=self.clean_,
                             showJson=options.showJson,
                             showNoSeqNum=options.showNoSeqNum )
      config = dstString.getvalue()
      with file( dstFn, 'wb' ) as dstFile:
         dstFile.write( config )

   def put( self, srcFn, append=False ):
      # Checks file content matches md5 digest, if one provided
      if self.md5:
         url = Url.parseUrl( 'file:%s' % srcFn, Url.Context( None, True ) )
         actualMd5 = FileCliUtil.chunkedHashCompute( url, hashlib.md5 )
         if actualMd5 != self.md5:
            raise CliCommon.MD5Error(
                    'md5 mismatch; %r != %r' % ( actualMd5, self.md5 ) )
      srcFile = file( srcFn, 'rb' )
      try:
         if not self.exists():
            raise EnvironmentError( errno.EACCES, os.strerror( errno.EACCES ) )
         assert not self.isdir()
         assert self.validPathname()
         import MainCli
         import CliPlugin.SessionCli
         with ConfigMount.ConfigMountDisabler( disable=False ):
            MainCli.loadConfigFromFile( srcFile, self.context.entityManager,
                           disableAaa=self.context.disableAaa,
                           initialModeClass=CliPlugin.SessionCli.ConfigSessionMode,
                           disableGuards=True,
                           raiseOnException=self.raiseOnException,
                           skipConfigCheck=self.skipConfigCheck,
                           startupConfig=self.loadAsStartupConfig,
                           isEapiClient=self.context.cliSession.isEapiClient(),
                           aaaUser=self.context.cliSession.aaaUser(),
                           privLevel=self.context.cliSession.privLevel_ )
      finally:
         srcFile.close()

class SessionFilesystem( FlashUrl.LocalFilesystem ):
   urlClass_ = SessionUrl

   def __init__( self, scheme ):
      FlashUrl.LocalFilesystem.__init__( self, scheme, 'session', 'rw', hidden=True )

   def supportsListing( self ):
      return True  

def Plugin( context=None ):
   Url.registerFilesystem( SessionFilesystem( 'session:' ) )
