# Copyright (c) 2005-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""This module implements the "extension" URL scheme."""

import os
import errno

import ExtensionMgrLib
import Tac
import Url
from UrlPlugin.FlashUrl import FlashFilesystem, FlashUrl
from ExtensionMgr import errors
from Tracing import t0

def installedExtensions( entityManager, disableAaa, cliSession ):
   context = Url.Context( entityManager, disableAaa, cliSession )
   return Url.parseUrl( "extension:/installed-extensions", context )

def bootExtensions( entityManager, disableAaa, cliSession ):
   context = Url.Context( entityManager, disableAaa, cliSession )
   return Url.parseUrl( "flash:/boot-extensions", context )

# FIXME(nschrenk): I would like to be able to obtain this path from the
# Sysdb at %cellPath/sys/extension/config/extensionsDir, but the current UrlPlugin
# scheme does not provide an EntityManager or any other handle to Sysdb.
# This results in this path being present in two places: the line below
# and ExtensionMgr.tac, and it means that changing the path in Sysdb will
# not actually work.
extensionsDir = None

class ExtensionUrl( FlashUrl ):
   def localFilename( self ):
      return self.realFilename_

   def get( self, dstFn ):
      """Fetch the contents of this Url and write it into the given local file."""
      if self.pathname == '/installed-extensions':
         dstFile = file( dstFn, 'wb' )
         try:
            sysdbRoot = None
            if self.context and self.context.entityManager:
               sysdbRoot = self.context.entityManager.root()
            ExtensionMgrLib.saveInstalledExtensions( sysdbRoot, dstFile )
         except errors.SignatureVerificationError as e:
            if self.context and self.context.cliSession:
               mode = self.context.cliSession.mode
               mode.addWarning( e.message )
         finally:
            dstFile.close()
      else:
         FlashUrl.get( self, dstFn )

   def _checkFilename( self ):
      if self.isdir():
         return
      validExtensions = [ '.swix', '.rpm' ]
      _, ext = os.path.splitext( self.realFilename_ )
      if ext not in validExtensions:
         t0( "_checkFilename: Bad filename", self.realFilename_ )
         raise EnvironmentError( errno.EINVAL,
            "Unsupported file extension '%s'" % ext )

   def _extensionStatus( self ):
      em = self.context.entityManager
      status = em.root().entity[ ExtensionMgrLib.statusPath() ]
      return status

   def _remove( self ):
      t0( "Removing extension", self.realFilename_ )
      extStatus = self._extensionStatus()
      basename = os.path.basename( self.realFilename_ )
      latest = ExtensionMgrLib.latestExtensionForName( basename, extStatus )
      if latest:
         if latest.status == 'notInstalled':
            # I can remove this because I only need to retain entries for
            # installed extensions.
            del extStatus.info[ latest.key ]
         else:
            # Check if the extension is mounted. It would look something like this:
            # /mnt/flash/.extensions/Ext.swix on /your/path/here type squashfs...
            output = Tac.run( [ 'mount' ], stdout=Tac.CAPTURE )
            if '\n' + self.realFilename_ in output:
               raise EnvironmentError( errno.EBUSY,
                              'This extension is in use; please uninstall it first' )
            latest.presence = 'absent'
      else:
         t0( "No record of extension", basename, "in Sysdb?" )

   def _add( self ):
      self._checkFilename()
      t0( "Adding extension", self.realFilename_ )
      extStatus = self._extensionStatus()
      basename = os.path.basename( self.realFilename_ )
      latest = ExtensionMgrLib.latestExtensionForName( basename, extStatus )
      if latest and latest.presence == 'absent':
         # We are replacing a missing installed extension with the same one,
         # and copying was successful so just update the presence.
         latest.presence = 'present'
      else:
         # We are adding a new extension
         ExtensionMgrLib.readExtensionFile( self.realFilename_, extStatus )
      # FIXME(nschrenk): we need to provide the user with some feedback if the
      # extension did not install properly.  That means checking
      # extensionStatus.info[ filename ].presence and .presenceDetail.  But how
      # to communicate it to the user?  I am buried down here at the wrong
      # layer to be printing to the screen -- I don't even know if I am in a
      # CLI process or any process where stdout goes to a user for that matter.
      # I'd like to raise an exception with a message to display.

   def _writeExtension( self, srcFn ):
      if os.path.exists( self.realFilename_ ):
         self._remove()
      t0( "Writing", srcFn, "to", self.realFilename_ )
      FlashUrl.put( self, srcFn, append=False )
      self._add()

   def put( self, srcFn, append=False ):
      """Read the file at path srcFn and overwrite the contents of this Url."""
      if append:
         raise EnvironmentError( errno.EOPNOTSUPP,
            "Appending to an extension is not supported" )
      self._checkFilename()
      self._writeExtension( srcFn )

   def copyfrom( self, srcUrl ):
      """Read the given source url, and update this url with its contents."""
      self._checkFilename()
      FlashUrl.copyfrom( self, srcUrl )
      if os.path.exists( self.realFilename_ ):
         self._remove()
      self._add()

   def renameto( self, dst ):
      raise EnvironmentError( errno.EOPNOTSUPP,
         "Renaming installed extensions is not supported" )

   def delete( self, recursive ):
      t0( "Deleting extension", self.realFilename_ )
      self._remove()
      FlashUrl.delete( self, recursive )

   mkdir = FlashUrl._notSupported
   rmdir = FlashUrl._notSupported
   empty = FlashUrl._notSupported

class ExtensionFilesystem( FlashFilesystem ):
   urlClass_ = ExtensionUrl

   def __init__( self, scheme, location ):
      FlashFilesystem.__init__( self, scheme, location, fsType='extension' )

   def realFileSystem( self ):
      return False

   def supportsListing( self ):
      return True

   def supportsDelete( self ):
      return True

   def fsRootIs( self, fsRoot ):
      FlashFilesystem.fsRootIs( self, fsRoot )
      global extensionsDir
      extensionsDir = self.location_

      if not os.path.isdir( extensionsDir ):
         try:
            os.makedirs( extensionsDir, mode=0770 )
         except OSError:
            # If makedirs fails I register the extension filesystem anyway in the
            # hope that someone will make the directory before I am needed.  This
            # behavior is useful in the context of breadth tests that use CliTest.
            pass

   def _mgmtSecurityConfig( self, context ):
      if not context.entityManager:
         return None

      root = context.entityManager.root()
      return root.entity[ 'mgmt/security/config' ]

   def _mgmtSslConfig( self, context ):
      if not context.entityManager:
         return None

      root = context.entityManager.root()
      return root.entity[ 'mgmt/security/ssl/config' ]

   def addCliWarning( self, message, context ):
      mode = context.cliSession.mode
      mode.addWarning( message )

   def _rpmIsDuplicate( self, fn, info, fmt=None, context=None ):
      rpmHeaderInfo = {}
      try:
         rpmHeaderInfo = ExtensionMgrLib.readExtensionRpmData( fn, fmt=fmt )
      except errors.ExtensionReadError:
         return False
      newRpmVersions = { rpm[ 'name' ]: rpm[ 'version' ]
                           for rpm in rpmHeaderInfo.values() }
      currRpmVersions = { rpm.packageName : rpm.version
                           for rpm in info.package.values() }
      if newRpmVersions == currRpmVersions:
         return True
      return False

   def _extensionStatus( self, context ):
      em = context.entityManager
      status = em.root().entity[ ExtensionMgrLib.statusPath() ]
      return status

   def _checkExists( self, filename, destfile, context ):
      """@filename - the location of the actual file to check
         @destfile - the location where the file is to be saved
         If the user has an extension installed they should
      not be able to overwrite unless it's an equivalent extension.
      It must be uninstalled first."""
      status = self._extensionStatus( context )
      fmt = ExtensionMgrLib.getPackageFormat( destfile )
      for infoKey, info in status.info.iteritems():
         if destfile == infoKey.filename and info.status == 'installed':
            if self._rpmIsDuplicate( filename, info, fmt=fmt, context=context ):
               self.addCliWarning( "Overwriting installed extension with"
                                   " equivalent extension file", context )
               break
            else:
               raise EnvironmentError( errno.EOPNOTSUPP,
                     "Cannot overwrite existing installed extension"
                     " with nonequivalent extension" )


   def validateFile( self, filename, durl=None, context=None ):
      """ 1. Check if we are replacing an existing installation with
             something equivalent
          2. Check SWIX signature if signature-verification is enabled and this
             is a SWIX file (rather than RPM). """
      import zipfile
      if durl and context:
         destfile = os.path.basename( durl.pathname )
         self._checkExists( filename, destfile, context )

      if zipfile.is_zipfile( filename ):
         mgmtSecConfig = self._mgmtSecurityConfig( context )
         mgmtSslConfig = self._mgmtSslConfig( context )
         if ExtensionMgrLib.signatureVerificationEnabled( mgmtSecConfig ):
            try:
               if not ExtensionMgrLib.verifySignature( filename,
                                                       mgmtSecConfig,
                                                       mgmtSslConfig ):
                  msg = 'The signature of the SWIX file is missing or invalid'
                  self.addCliWarning( msg, context )
            except errors.SignatureVerificationError as e:
               self.addCliWarning( e.message, context )

def Plugin( context=None ):
   Url.registerFilesystem( ExtensionFilesystem( 'extension:',
                                                'flash/.extensions' ) )

