#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
This module implements the "checkpoint" URL scheme.
"""

import errno
import os
import time

import Tac
import Url
from UrlPlugin.FlashUrl import FlashFilesystem, FlashUrl
import UtmpDump
from Tracing import t0

class CheckpointUrl( FlashUrl ):
   headerPrefix_ = '! The checkpoint file saved at '

   def getHeader( self ):
      # Return string formatted with save time and user
      # that will be inserted in the first line of checkpoint file
      info = UtmpDump.getUserInfo()
      saveTime = time.asctime( time.localtime() )
      return '%s%s by %s\n' % ( self.headerPrefix_, saveTime, info[ 'user' ] )

   def ignoreTrailingWhitespaceInDiff( self ):
      return True

   def localFilename( self ):
      return self.realFilename_

   mkdir = FlashUrl._notSupported
   rmdir = FlashUrl._notSupported

   def getUserName( self ):
      # get the create userName of checkpoint file
      userName = 'Unknown'
      with open( self.realFilename_, 'rb' ) as f:
         # header is the first line of checkpoint file
         header = f.readline()
         # user name is the last word in header
         userName = header.split()[ -1 ]
      return userName

   def copyfrom( self, srcUrl ):
      # Only running-config and  startup-config can be srcFile
      localName = srcUrl.pathname
      if localName in ( '/running-config', '/startup-config' ):
         FlashUrl.copyfrom( self, srcUrl )
      else:
         raise EnvironmentError( errno.EOPNOTSUPP,
                                 "Only running-config and startup-config "
                                 "can be copied to checkpoint dir" )

class CheckpointFileSystem( FlashFilesystem ):
   urlClass_ = CheckpointUrl

   def __init__( self, scheme, location ):
      FlashFilesystem.__init__( self, scheme, location, fsType='checkpoint' )

   def supportsMkdir( self ):
      return False

   def stat( self ):
      return ( 0, 0 )

   def fsRootIs( self, fsRoot ):
      t0( 'checkpoint fs location is ', self.location_ )
      FlashFilesystem.fsRootIs( self, fsRoot )

      flashRoot = self.location_.rsplit( '.', 1 )[ 0 ]
      if os.access( flashRoot, os.X_OK ) and not os.path.isdir( self.location_ ):
         try:
            Tac.run( [ "mkdir", "-m", "777", self.location_ ], asRoot=True )
         except Exception: # pylint: disable=broad-except
            pass

def Plugin( context=None ):
   Url.registerFilesystem( CheckpointFileSystem( 'checkpoint:',
                                                 'flash/.checkpoints' ) )
