#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import errno
import os
import simplejson as json

import Url
import Tracing
from UrlPlugin.FlashUrl import FlashUrl
from UrlPlugin.FlashUrl import FlashFilesystem
from AlertBaseImporter import AlertBaseImporter

t0 = Tracing.trace0
EmptyAlertBase = "/etc/AlertBase.json"

def _isStandby( entityManager ):
   redundancyStatus = getattr( entityManager, 'redundancyStatus', None )
   if redundancyStatus and redundancyStatus().mode != 'active':
      return True
   return False

class AlertbaseUrl( FlashUrl ):
   def _checkStandby( self ):
      if ( self.fs.entityManager and _isStandby( self.fs.entityManager ) ):
         raise EnvironmentError( errno.EOPNOTSUPP,
                                 "Operation not supported on this supervisor" )

   def _checkFileName( self, filename ):
      basename = filename.split('/')[-1]
      if basename != 'AlertBase.json':
         t0( "_checkFilename: Bad filename ", basename )
         raise EnvironmentError( errno.EINVAL,
            "Invalid file name '%s'" % basename )

   def _checkValidJson( self, filename ):
      try:
         alertBase = open( filename )
         json.load( alertBase )
      except json.JSONDecodeError as e:
         t0( "JSONDecodeError exception:", e )
         os.remove( filename )
         raise EnvironmentError( errno.EINVAL, "Invalid JSON file" )
      finally:
         alertBase.close()

   def put( self, srcFn, append=False ):
      self._checkStandby()
      FlashUrl.put( self, srcFn, append )

   def writeLocalFile(self, srcUrl, filename):
      self._checkStandby()
      self._checkFileName( filename )
      tmpFilename = filename + ".tmp"
      FlashUrl.writeLocalFile( self, srcUrl, tmpFilename )
      self._checkValidJson( tmpFilename )
      os.rename( tmpFilename, filename )

      sysname = self.context.entityManager.sysname()
      alertBaseImporter = AlertBaseImporter( filename, sysname )
      alertBaseImporter.loadAlertBase()

   def delete( self, recursive ):
      FlashUrl.delete( self, recursive )
      # Delete Alertbase entries
      sysname = self.context.entityManager.sysname()
      alertBaseImporter = AlertBaseImporter( EmptyAlertBase, sysname )
      alertBaseImporter.loadAlertBase()

   mkdir = FlashUrl._notSupported
   rmdir = FlashUrl._notSupported
   empty = FlashUrl._notSupported

class AlertbaseFilesystem( FlashFilesystem ):
   urlClass_ = AlertbaseUrl

   def __init__( self, scheme, rawlocation ):
      FlashFilesystem.__init__( self, scheme, rawlocation,
                                fsType='alert-base', permission='rw',
                                mask=0002 )

   def realFileSystem( self ):
      return True

   def stat( self ):
      return ( 0, 0 )
   
   def supportsMkdir( self ):
      return False

   def supportsListing( self ):
      return True

   def supportsDelete( self ):
      return True

def Plugin( context=None ):
   Url.registerFilesystem( AlertbaseFilesystem( 'alert-base:', '/persist/sys/' ) )
