# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing

import re

traceHandle = Tracing.Handle( 'JsonUwsgiServer' )
log = traceHandle.trace0
warn = traceHandle.trace1
info = traceHandle.trace2
trace = traceHandle.trace3
debug = traceHandle.trace4

urlMap = {}

OPENSTACK_ENDPOINT = "openstack"

class UrlToRegex( object ):
   """
   Takes in a URL format and returns a regular expression so we can match it.

   Anything between {} will be named and stored in a regex group.
   Anything between [] is indicated as optional.

   Example input: '/api/region[/{region}][/]'.
   This will create a regex to match any of:
   /api/region
   /api/region/
   /api/region/foo
   /api/region/foo/

   Example call: UrlToRegex('/api/region[/{region}][/]')
   """
   vStart, vEnd = '{', '}'
   oStart, oEnd = '[', ']'
   special = ( vStart, vEnd, oStart, oEnd )

   def _makeGroups( self, url ):
      while url.find( self.vStart ) != -1:
         start = url.find( self.vStart )
         end = url.find( self.vEnd )
         groupName = url[ start + 1 : end ]
         url = url.replace( url[ start : end + 1 ], '(?P<%s>[^/]+)' % groupName )
      return url

   def __call__( self, url ):
      # if there are no special chars just escape it and return
      if not any( [ c for c in self.special if c in url ] ):
         return '^%s$' % re.escape( url )

      url = url.replace( '/', '\\/' )
      url = url.replace( self.oStart, '(' )
      url = url.replace( self.oEnd, ')?' )
      url = self._makeGroups( url )
      return '^%s$' % url

def urlHandler( types=(), url='' ):
   """
   This is a decorator that desribes the URL that this function will handle.
   The consumer must supply the HTTP type as a string an the URL that the decorator
   will handle.
   """
   def addToUrlMap( func ):
      for httpType in types:
         tMap = urlMap.get( httpType )
         if not tMap:
            urlMap[ httpType ] = {}
            tMap = urlMap[ httpType ]

         # parse the URL into a regex
         uri = UrlToRegex()( url )
         tMap[ uri ] = func
      return func
   return addToUrlMap

def getHandler( rType, parsedUrl ):
   typeMap = urlMap.get( rType )
   if typeMap is None:
      return None, None

   for regex, func in typeMap.iteritems():
      match = re.match( regex, parsedUrl.path )
      if match:
         return func, match.groupdict()

   return None, None
