# Copyright (c) 2005-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

# This module encapsulates the file location syntax used by the
# industry-standard CLI.  See AID 108 for a more complete description of URLs.
# UrlPlugins are used to extend this module with support for additional URL
# schemes and filesystems.

import os
import threading
import Tac
import Plugins
import CliParser
import CliAaa
import CliCommon
import CliMatcher
import CliParserCommon
import errno

entityManager_ = None

class Context( object ):
   def __init__( self, entityManager, disableAaa, cliSession=None ):
      self.entityManager = entityManager
      self.disableAaa = disableAaa
      self.cliSession = cliSession

def urlArgsFromMode( mode ):
   return ( mode.entityManager, mode.session_.disableAaa_, mode.session )

_syncFlashFilesystems = None
_syncFilesystemRoot = None

# threadLocalData object needs to be created just once and not per thread. This
# object grabs the locks when attributes are accessed. The attributes are local
# to the thread.
threadLocalData = threading.local()

def _fsRoot():
   return getattr( threadLocalData, 'fsRootDir', None )

def _fsRootIs( root ):
   threadLocalData.fsRootDir = root

def _getFsRoot():
   if not _fsRoot():
      _fsRootIs( os.path.abspath( os.environ.get( 'FILESYSTEM_ROOT', '/mnt' ) ) )
      if _syncFilesystemRoot:
         _syncFilesystemRoot( _fsRoot() )
   return _fsRoot()

def clearFsRoot():
   _fsRootIs( None )

def invokeSyncFlashFilesystems():
   _getFsRoot()
   if _syncFlashFilesystems is not None:
      _syncFlashFilesystems()

# 'flash:' is a special case.  There must always be an entry named 'flash:' in
# the _filesystems dict - even if for some strange reason no such directory exists
# under the filesystem root - so that homeDirectory() and localStartupConfig() are
# happy.  If that directory doesn't exist, any attempts to access the filesystem
# will fail with ENOENT or ENOTDIR, which is a reasonable enough error message for
# this unexpected situation.

def setSyncFilesystemRoot( Fn ):
   """Sets a callback when the filesystem root is known."""
   global _syncFilesystemRoot
   _syncFilesystemRoot = Fn

def setSyncFlashFilesystems( Fn ):
   """Sets the syncFlashFilesystems function."""
   global _syncFlashFilesystems
   _syncFlashFilesystems = Fn

def _homeDirectory():
   return getattr( threadLocalData, 'homeDir', None )

def _homeDirectoryIs( url ):
   threadLocalData.homeDir = url

def setHomeDirectoryUnchecked( url ):
   """Sets the home directory."""
   _homeDirectoryIs( url )

def fsRoot():
   return _fsRoot()

def homeDirectory():
   """Returns the directory in which the process begins, and which 'cd' without any
   parameter should return us to."""
   if not _homeDirectory():
      invokeSyncFlashFilesystems()
   return _homeDirectory()

def _currentDirectory():
   return getattr( threadLocalData, 'currentDir', None )

def _currentDirectoryIs( url ):
   threadLocalData.currentDir = url

def currentDirectory():
   """Returns the current directory.  Note that we do not maintain a current
   directory on each filesystem, like Windows does."""
   if not _currentDirectory():
      invokeSyncFlashFilesystems()
   return _currentDirectory()

def setCurrentDirectoryUnchecked( url ):
   """Sets the current directory without checking for valid url."""
   _currentDirectoryIs( url )

def setCurrentDirectory( url ):
   """Sets the current directory."""
   if not url.exists():
      raise EnvironmentError( errno.ENOENT, os.strerror( errno.ENOENT ) )
   if not url.isdir():
      raise EnvironmentError( errno.ENOTDIR, os.strerror( errno.ENOTDIR ) )
   _currentDirectoryIs( url )

def setEntityManager( entityManager ):
   global entityManager_
   for fs in filesystems():
      fs.entityManager = entityManager
   entityManager_ = entityManager

class Url( object ):
   headerPrefix_ = None

   def __init__( self, fs, url ):
      # url is the string that was originally passed to parseUrl.
      self.url = url
      self.fs = fs
      self.context = None

   def __cmp__( self, other ):
      if isinstance( other, Url ):
         return cmp( ( self.fs, self.url ), ( other.fs, other.url ) )
      else:
         return NotImplemented

   def historyEntryName( self ):
      '''If we want to track the URL's change history, return a valid name'''
      return None

   def localFilename( self ):
      '''Return the filename of a local file that corresponds to this object.
      If there is no local file, e.g., this is an http url, then return None.'''
      # This is the base class implementation.  Derived classes may override.
      return None

   def size( self ):
      '''Return the size of the URL, if can be provided relatively quickly
      (e.g., remote URLs don't have it).'''
      return None

   def exists( self ):
      # This is the base class implementation.  Derived classes may override.
      return False

   def hasHeader( self ):
      # This is the base class implementation.  Derived classes may override.
      return self.headerPrefix_ is not None

   def isHeader( self, header ):
      # This is the base class implementation.  Derived classes may override.
      return self.headerPrefix_ and header.startswith( self.headerPrefix_ )

   def getHeader( self ):
      # This is the base class implementation.  Derived Classes may override.
      if self.headerPrefix_:
         raise NotImplementedError
      else:
         return None

   def encrypted( self ):
      return False

   def encrypt( self, content ):
      return content

   def ignoreTrailingWhitespaceInDiff( self ):
      return False

   def writeHeaderAndRenameFile( self, srcFileName, dstFileName, dstUrl=None ):
      # If there is a custom-header to be prepended, create a tempfile,
      # write the custom-header and then append the contents of src-file.
      # Finally rename the tempfile to the src-file.
      headerStr = dstUrl.getHeader() if dstUrl else None
      if headerStr and os.path.getsize( srcFileName ):
         tempfilename = self.tempfilename( dstFileName )
         with open( srcFileName, 'rb' ) as srcFile:
            # Read the first line of the src-file and
            # match it with the expected prefix. Accordingly
            # create the first line of the startup-config.
            line = srcFile.readline()
            if dstUrl.isHeader( line ):
               line = headerStr
            else:
               line = headerStr + line

            # Write the first line(s) of the startup-config
            # we got from above and then copy the rest of
            # the file.
            with open( tempfilename, 'wb' ) as dstFile:
               content = line + srcFile.read()
               dstFile.write( self.encrypt( content ) )

         os.rename( tempfilename, srcFileName )

      # do an fsync to ensure the file data is synced to disk
      with open( srcFileName, 'ab' ) as f:
         os.fsync( f.fileno() )
      os.rename( srcFileName, dstFileName )
      # sync the parent dir
      fd = os.open( os.path.dirname( dstFileName ), os.O_DIRECTORY )
      try:
         os.fsync( fd )
      finally:
         os.close( fd )

   def writeLocalFile( self, srcUrl, filename ):
      # This function copies the srcUrl content into a local file.
      #
      # To minimize risk when the copy operation is interrupted, it writes to
      # a temporary file, syncs the data to disk and renames it to the
      # destination file. The destination file is not affected until the final
      # rename() operation.
      #
      # Depending on the filesystem and the underlying storage device, rename()
      # may or may not be exactly atomic when the power is lost, but this is the
      # best we can do.
      self.checkOpSupported( self.fs.supportsWrite )
      tempfilename = self.tempfilename( filename )
      try:
         if self.fs.mask is not None:
            oldMask = os.umask( self.fs.mask )
         srcUrl.get( tempfilename )

         # If available pass the dstUrl as well in the calls below.
         dstUrl = self if self.localFilename() == filename else None
         self.fs.validateFile( tempfilename, durl=dstUrl, context=self.context )
         self.writeHeaderAndRenameFile( tempfilename, filename, dstUrl )

      except:
         # remember the original exception information so we can raise again
         import sys
         exc_info = sys.exc_info()
         # clean up the tempfile
         try:
            os.unlink( tempfilename )
         except OSError:
            pass
         raise exc_info[ 0 ], exc_info[ 1 ], exc_info[ 2 ]
      finally:
         if self.fs.mask is not None:
            os.umask( oldMask )

   def copyfrom( self, srcUrl ):
      '''Read the given source url, and update this url with its contents.'''
      # Strategy #1: the destination is a local file.  Use "srcUrl.get(...)" to
      # overwrite that file with the contents of the source URL.  We check this
      # condition first to make sure that we sync the local file to disk before
      # returning.
      import tempfile
      dstFn = self.localFilename()
      if dstFn:
         self.writeLocalFile( srcUrl, dstFn )
         return
      # Strategy #2: the source is a local file.  Use "self.put(...)" to overwrite
      # the destination URL with that file.
      srcFn = srcUrl.localFilename()
      if srcFn and not srcUrl.encrypted():
         self.put( srcFn )
         return
      # Strategy #3: neither the source nor the destination is a local file.  Copy
      # via a temporary file.
      ( fd, filename ) = tempfile.mkstemp()
      try:
         os.close( fd )
         # Some url types are unhappy if the target file already exists
         os.unlink( filename )
         srcUrl.get( filename )
         self.put( filename )
      finally:
         try:
            os.unlink( filename )
         except OSError, e:
            # unlink() will fail in cases(incorrect password) where temp file is not
            # created, but actual failure will be masked, hence ignoring ENOENT
            if e.errno == errno.ENOENT:
               pass
            else:
               raise

   def open( self ):
      '''Return a file-like object from which the contents of this URL can
      be read.'''
      # This is a default implementation, that works by fetching
      # "self" into a temporary local file and then returning that.
      # Certain derived classes override this to provide a more
      # efficient implementation.  An obvious case is if the underlying
      # URL names a local file, in which case open() can just open the
      # file without making a temporary copy.
      self.checkOpSupported( self.fs.supportsRead )
      import tempfile
      ( fd, filename ) = tempfile.mkstemp()
      try:
         os.close( fd )
         self.get( filename )
         f = file( filename, 'rb' )
         return f
      finally:
         os.unlink( filename )

   def _notSupported( self, *a, **kw ):
      raise EnvironmentError( errno.EOPNOTSUPP, os.strerror( errno.EOPNOTSUPP ) )

   def get( self, dstFn ):
      '''Fetch the contents of this Url and write it into the given local file.'''
      # All derived classes must override this, or your url is unreadable.
      self._notSupported()

   def put( self, srcFn, append=False ):
      '''Read the file and overwrite the contents of this Url.'''
      # All derived classes must override this, or your url is unwritable.
      self._notSupported()

   def tempfilename( self, prefix ):
      # Generate a temporary file for prefix by appending a random encoded string.
      # We do not use tempfile.mkstemp because we do not want to create the file
      # here, and os.tempnam() prints a security warning, so we implement our own
      # little function.
      import string
      import random
      vlist = string.ascii_letters + string.digits
      return '%s.%s' % ( prefix, string.join( random.sample( vlist, 6 ), '' ) )

   def verifyHash( self, hashName, mode=None, hashInitializer=None ):
      # Derived classes must override this if needed.
      self._notSupported()

   def checkOpSupported( self, supportsFunc ):
      if not supportsFunc():
         self._notSupported()

   def aaaToken( self ):
      return self.__str__()

class Filesystem( object ):
   def __init__( self, scheme, fsType, permission, hidden=False, mask=None,
                 noPathComponent=False, entityManager=None ):
      self.scheme = scheme
      self.fsType = fsType
      self.permission = permission
      self.hidden = hidden
      self.noPathComponent = noPathComponent
      self.mask = mask
      self.entityManager = entityManager

   def __repr__( self ):
      return "Filesystem( %r )" % self.scheme

   def __cmp__( self, other ):
      if isinstance( other, Filesystem ):
         return cmp( self.scheme, other.scheme )
      else:
         return NotImplemented

   def stat( self ):
      return ( 0, 0 )

   def mountInfo( self ):
      return ( None, None, '-' )

   def ignoresCase( self ):
      """Filesystems that ignore case (for example, vfat/ntfs should override
      this to return True."""
      return False

   def realFileSystem( self ):
      """Filesystems that act like a real file system that supports normal
      file operations and can be used in places that require a filename."""
      return False

   def supportsListing( self ):
      """Filesystems that support listing the contents of a directory should
      override this to return True."""
      return self.realFileSystem() and 'r' in self.permission

   def supportsRename( self ):
      """Filesystems that support renaming files should override this to
      return True."""
      return self.realFileSystem() and 'w' in self.permission

   def supportsDelete( self ):
      """Filesystems that support deleting files should override this to
      return True."""
      return self.realFileSystem() and 'w' in self.permission

   def supportsMkdir( self ):
      """Filesystems that support mkdir/rmdir should override this to
      return True."""
      return self.realFileSystem() and 'w' in self.permission

   def supportsWrite( self ):
      """Filesystems that support writing to files should override this to
      return True."""
      return 'w' in self.permission

   def supportsRead( self ):
      """Filesystems that support reading from files should override this to
      return True."""
      return 'r' in self.permission

   def supportsAppend( self ):
      """Filesystems that support appending to files should override this to
      return True."""
      return self.realFileSystem() and 'w' in self.permission

   def filenameToUrl( self, filename ):
      """Filesystems that support converting from a filename to a URL should
      override this to return a string representing the URL of this filename."""
      return None

   def filenameToUrlQuality( self ):
      """In a hierarchical filesystem, there may be multiple matches to a given
      filename -> URL conversion (e.g.,
      file:/mnt/flash/.extensions/foo
      flash:/.extensions/foo
      extensions:/foo
      ).  The higher the returned Quality, the better the match."""
      return 0

   def validateFile( self, filename, durl=None, context=None ):
      """This is the base class implementation. Derived Classes may override
      to validate the file for particular filesystem. Currently, this is called
      with the temporary file before copying the file to the local filesystem.
      The Derived classes should raise EnvironmentError exception if not
      a valid file for the particular filesystem. For example: the
      certificate: and sslkey: file systems will raise exception if its
      not a valid certificate/key file."""
      pass

_filesystems = {}

def registerFilesystem( fs ):
   assert fs.scheme not in _filesystems
   _filesystems[ fs.scheme ] = fs
   if entityManager_:
      fs.entityManager = entityManager_

def unregisterFilesystem( fs ):
   assert _filesystems.get( fs.scheme ) is fs
   del _filesystems[ fs.scheme ]

def filesystemsUnsynced():
   """Returns a list of all currently registered filesystems."""
   return _filesystems

def filesystems( showHidden=True ):
   """Returns a list of all currently registered filesystems."""
   invokeSyncFlashFilesystems()
   if showHidden:
      return _filesystems.values()
   else:
      return [ v for v in _filesystems.values() if not v.hidden ]

def getFilesystem( scheme ):
   """Returns the filesystem with the specified scheme, or None if no such filesystem
   exists."""
   invokeSyncFlashFilesystems()
   return _filesystems.get( scheme )

currentFilesystemNoLongerExists = object()

def parseUrl( url, context, fsFunc=None, acceptSimpleFile=True ):
   """Constructs a Url object from the given string.  Raises a ValueError if the
   scheme is not recognized, or if fsFunc does not return True when passed the
   filesystem.  Returns the special object currentFilesystemNoLongerExists if no
   scheme was specified and the current filesystem no longer exists. Also raises
   a ValueError if acceptSimpleFile is set to False and the given string does not
   have a ':' somewhere in it."""

   if ':' in url:
      # A scheme was specified.
      scheme = url[ : url.find( ':' ) + 1 ]
      fs = getFilesystem( scheme )
      if not fs:
         raise ValueError( "Unknown URL scheme: %r" % scheme )
      if fsFunc and not fsFunc( fs ):
         raise ValueError(
            "Filesystem %s does not meet the requirements" % fs )
      rest = url[ url.find( ':' ) + 1 : ]
   else:
      # If no scheme was specified, the URL is on the current filesystem.  In this
      # case, we ignore the fsTypes - we always accept relative URLs, and it is up to
      # any individual file CLI command to reject them with an appropriate error
      # message if they are not acceptable to that command.
      fs = currentDirectory().fs
      rest = url
      if not acceptSimpleFile:
         raise ValueError( "Want URL with colon" )
      if not fs in filesystems():
         return currentFilesystemNoLongerExists

   return fs.parseUrl( url, rest, context )

# Filesystems that represent actual locations on the UNIX filesystem
# have an attribute "location_", which specifies the root.  For any
# registered filesystem with a location_ attribute, if the filename
# starts with the location, pick the filesystem with the longest
# prefix.
def filenameToUrl( filename, simpleFile=True, relativeTo=None ):
   filename = os.path.normpath( os.path.abspath( filename ) )
   if relativeTo is not None and filename.startswith( relativeTo ):
      return os.path.relpath( filename, relativeTo )
   bestMatch = None
   for fs in filesystems():
      if fs.filenameToUrl( filename ) is not None and (
            bestMatch is None or
            bestMatch.filenameToUrlQuality() < fs.filenameToUrlQuality() ):
         bestMatch = fs
   if bestMatch:
      return bestMatch.filenameToUrl( filename )
   else:
      assert simpleFile
      return filename

class UrlMatcher( CliMatcher.Matcher ):
   """Class used by CLI plugins to match a URL."""

   def __init__( self, fsFunc, helpdesc, notAllowed=[], acceptSimpleFile=True,
                 **kargs ):
      """I match a URL if the filesystem matched by the URL passes the check
      specified by fsFunc, which must be a callable that takes a single
      parameter, a Filesystem instance, and return True when passed a
      Filesystem that I should match. If acceptSimpleFile is set to False,
      then any filename without a ':' in it will lead to a ValueError."""
      self.fsFunc_ = fsFunc
      self.acceptSimpleFile = acceptSimpleFile

      # notAllowed is a list of strings that no match should be a prefix of.  These
      # are typically keywords that could appear in place of a URL (e.g. '/recursive'
      # in the 'dir [/recursive] <url>' command).
      self.notAllowed_ = notAllowed
      CliMatcher.Matcher.__init__( self, helpdesc=helpdesc, **kargs )

   def __str__( self ):
      return "<url>"

   def _contextFromMode( self, mode ):
      return Context( *urlArgsFromMode( mode ) )

   def match( self, mode, context, token ):
      if any( k.startswith( token ) for k in self.notAllowed_ ):
         # We return noMatch so that the keyword is the unique match for this
         # token, which will allow this token to auto-complete to that keyword.
         return CliCommon.noMatch
      try:
         url = parseUrl( token, self._contextFromMode( mode ),
                         self.fsFunc_, self.acceptSimpleFile )
         if isinstance( url, Url ):
            aaaToken = url.aaaToken()
         else:
            aaaToken = str( url )
         return CliParserCommon.MatchResult( url, aaaToken )
      except ValueError:
         # The path contains a scheme, but either it's not a valid scheme or it's not
         # one of the schemes accepted by this command, so we reject this token.
         # Note that this means that filenames that contain colons can never be
         # entered as relative paths.  This is not a real problem as we don't
         # anticipate many filenames containing colons, and there's an easy
         # workaround (specify the filename as an absolute path, with the scheme).
         return CliCommon.noMatch

   def completions( self, mode, context, token ):
      # Note that we only show filename completions if a valid scheme has been
      # specified, not for relative paths.

      if not ':' in token:
         # The user has not finished typing a scheme yet.  Return a list of
         # completions for the scheme.
         completionsList = []
         for fs in filesystems( showHidden=False ):
            if fs.scheme.startswith( token ) and self.fsFunc_( fs ):
               completionsList.append(
                  CliParser.Completion( name=fs.scheme,
                                        help=self.helpdesc_,
                                        partial=( not fs.noPathComponent ) ) )
      else:
         try:
            url = parseUrl( token, self._contextFromMode( mode ),
                            self.fsFunc_, self.acceptSimpleFile )
            # The user has typed a scheme.  Return a list of filename completions.
            if hasattr( url, 'getCompletions' ):
               # Completion reveals directory content so we need to check AAA.
               # This is a bit unusual but I guess there is no other way. We
               # just send "dir <url>" and see if it's allowed.
               if mode.session_.authorizationEnabled():
                  ok, _ = CliAaa.authorizeCommand( mode,
                                                   mode.session_.privLevel_,
                                                   [ 'dir', url.url ] )
                  if not ok:
                     return []
               completionsList = url.getCompletions()
            else:
               completionsList = [
                  CliParser.Completion( name='A URL beginning with this prefix',
                                        help='', literal=False ) ]
         except ValueError:
            # The user has typed a scheme, but it's either not a valid scheme or it's
            # not one of the schemes accepted by this command.
            completionsList = []

      return completionsList

# The current UrlPlugin approach would need to be extended to support moving
# the flash filesystem code into a plugin because there is no way to register
# a callback to invoke _syncFlashFilesystems() at the appropriate times.  It
# is straightforward to extend the UrlPlugin mechanism to do this if/when we
# develop a UrlPlugin that needs it.
Plugins.loadPlugins( "UrlPlugin" )
