#!/usr/bin/env python
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""A Python logging module Logger that uses TACC Tracing to emit logs.

This module is typically used prior to importing third-party code;
Python code typically uses the standard library's "logging" module
to emit debugging, informational and error messages. Arista uses a
similar implementation in TACC, Tracing. This module causes Python
log messages to be sent via the TACC Tracing framework.

To use this service, you must first import this module and call the
setup() method, with a Tracing handler name prefix. Python logger
names are appended to this name and trace handlers are created for
each new Python logger. A trace map can also be supplied, overriding
the built in Python log level to Trace log level mapping.

After calling setup(), import the third party code that uses Python's
logging library. Logs emitted by Python's logging module will be sent
via the Tracing system, using trace handles and trace levels based on
the Python logger names and logging levels, respectively.

e.g.,

---foo.py---

# Import and setup the trace logger
import TraceLogger

# Setup Python logging via trace, prefix
# all trace handles with "Xmpp."
TraceLogger.setup( "Xmpp" )

# For no prefix (trace handles names match Python logger names), use:
TraceLogger.setup()

# Now that trace logging is configured, import third party code
# to use it
import thirdParty

---thirdParty.py---

import logging
log = logging.getLogger(__name__)
# Logs emitted use the trace handle "<prefix>.thirdParty"
log.info("This will be seen via trace 3 (INFO). No code changes required!")

# also works with the default logger, e.g.,
import logging
logging.info("This is seen at trace level 3, capping off an enormous victory.")
"""

from __future__ import absolute_import, division, print_function
import logging
import os

import Tracing
import _Tac

# Python log levels (ints) for levels supported only by tracing.
# Example:
# import TraceLogger, logging
# TraceLogger.setup( "Foo" )
# logging.log( TraceLogger.FUNCTION, "barMethod" )
# logging.info( "please read me later" )

# python NOTSET = 0
COVERAGE = 1
FUNCTION = 2
# python DEBUG = 10

# The default map of Python logging level to TACC trace levels
# defined in Tracing.py's allLevels_ array. Trace levels are strings.
TRACE_LEVEL_MAP = {
  logging.CRITICAL: '0',
  logging.FATAL: '0',
  logging.ERROR: '1',
  logging.WARNING: '2',
  logging.WARN: '2',
  logging.INFO: '3',
  logging.DEBUG: '9',
  logging.NOTSET: '9',
  COVERAGE: 'coverage',
  FUNCTION: 'function',
}
# Supply an alternate dictionary as the traceMap argument to
# setup() for a different mapping.

# Trace level used for Python messages with unknown levels
DEFAULT_LEVEL = '9'


def setup( prefix="", level=logging.INFO, traceMap=None ):
   """Sets up Python's logging system to emit Trace records.

   Args:
     prefix: str, the name prefix to use on trace handles for Python loggers.
     traceMap: dict of int (Python logging level) to string (Trace level).
   """

   class Logger( logging.getLoggerClass() ):
      """A Python Logger object that emits events via a TraceLogHandler."""

      def __init__( self, name, level=level, traceMap=traceMap ):
         super(Logger, self).__init__( name, level=level )
         loggerName = name
         if prefix:
            loggerName = prefix + "." + name
         # Completely replace the handlers list
         self.handlers = [ TraceLogHandler( loggerName, traceMap=traceMap ) ]

   # Set the default logging class, causing loggers returned by
   # logging.getLogger() to be redirected to tracing.
   logging.setLoggerClass( Logger )
   # Replace the root logger also, to cause logging.info(...) calls
   # to be redirected to tracing also.
   logging.root = Logger( "root", level=level )


# Shortcuts to optimize attribute lookup and aggregate pylint warning
# management.
# pylint: disable-msg=W0212
_convert = Tracing._convert
_traceLineSuppressed = Tracing._traceLineSuppressed


def _trace_pylog_h( handle, level, record, *msg ):
   """Logs a Python log message via a Tracing handle.

   Args:
     handle: Tracing.Handle, the trace handle to use for logging
     level: str, int; a tracing log level
     record: logging.LogRecord; Python log record used for trace context metadata
     msg: An iterable of message sections that will be separated by spaces
       in the output
   """
   level = Tracing.levelToEnumValue_[level]
   if not _traceLineSuppressed( handle.facility_, level ):
      line = record.lineno
      filename = os.path.basename( record.pathname )
      funcName = record.funcName
      _Tac.trace_h( handle.facility_,
                    filename,
                    line,
                    funcName,
                    ' '.join( _convert(x) for x in msg ),  # _format
                    level )


class TraceLogHandler( logging.Handler ):
   """A Python logging handler using TACC Tracing to emit messages.

   This handler is attached to a Logger object (with Logger.addHandler). When
   attached, that logger will emit messages to the trace handle named by the
   'name' constructor argument.
   """

   def __init__( self, name, traceMap=None ):
      """Initializer.

      Args:
        name: str, the full name for this handler's trace handle.
        traceMap: dict, a int -> int map of Python log level to trace log levels.
      """
      logging.Handler.__init__( self )
      self.traceMap_ = traceMap or TRACE_LEVEL_MAP
      self.handle_ = Tracing.Handle( name )

   def emit( self, record ):
      """Emits the supplied log record to TACC tracing."""
      try:
         msg = self.format( record )
         # Use a default trace level for Python level values without constants
         _trace_pylog_h(
            self.handle_,
            self.traceMap_.get( record.levelno, DEFAULT_LEVEL ),
            record,
            msg )
      except (KeyboardInterrupt, SystemExit):
         # Never block these exceptions
         raise
      except Exception:
         # Enable built-in error messages with logging.raiseExceptions=True.
         self.handleError( record )
