# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
import DateTimeRule
import calendar
import datetime

# Builds up a timeRangeExpression that allows input of two dates, which could
# be used as input to the time-filter utility to filter log messages within
# a certain time frame ( used by "show logging" and "show aaa accounting logging"
# commands )

def flexibleDateExpression( name ):
   # Date supporting various formats
   dateName = name + "_DATE"
   monthName = name + "_MONTH"
   dayName = name + "_DAY"

   class DateExpression( CliCommand.CliExpression ):
      expression = "%s | ( %s %s ) | ( %s %s )" % ( dateName, monthName, dayName,
                                                    dayName, monthName )
      data = {
         dateName: DateTimeRule.dateExpression( dateName,
                                                 helpdesc='Date to filter' ),
         monthName: DateTimeRule.monthMatcher,
         dayName: DateTimeRule.dayMatcher
         }

      @staticmethod
      def adapter( mode, args, argsList ):
         # set value to ( month, day, year )
         date = args.pop( dateName, None )
         if date:
            # dateExpression already dealt with Iteration
            args[ name ] = date
         else:
            month = args.pop( monthName, None )
            if month:
               day = args.pop( dayName )
               # deal with Iteration
               if isinstance( month, list ):
                  month = month[ 0 ]
                  assert isinstance( day, list )
                  day = day[ 0 ]
               args[ name ] = ( month, day, None )

   return DateExpression

def dateTimeExpression( name ):
   dateName = name + "_DATE"
   timeName = name + "_TIME"
   dateExp = flexibleDateExpression( dateName )

   class DateTimeExpression( CliCommand.CliExpression ):
      expression = "( %s %s ) | %s | %s" % ( dateName, timeName,
                                             dateName, timeName )
      data = { dateName: dateExp,
               timeName: DateTimeRule.ValidTimeMatcher(
                  helpdesc="Time to filter" )
               }

      @staticmethod
      def adapter( mode, args, argsList ):
         dateVal = args.pop( dateName, None )
         timeVal = args.pop( timeName, None )
         if dateVal or timeVal:
            if dateVal is None:
               dateVal = ( None, None, None )
            if timeVal is None:
               timeVal = ( None, None, None )
            elif isinstance( timeVal, list ):
               timeVal = timeVal[ 0 ]
            args[ name ] = ( dateVal, timeVal )

   return DateTimeExpression

# timeRangeToken is an optional argument, used when more than one
# time-range needs to be specified in the same command,
# in which case the default token name cannot be repeated in the syntax
def timeRangeExpression( name, timeRangeToken='time-range', **kargs ):
   startName = name + "_START"
   endName = name + "_END"
   timeRangeNode = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
      'time-range',
      helpdesc='Filter by begin and end time' ), **kargs )

   class TimeRangeFilter( CliCommand.CliExpression ):
      expression = "%s %s %s" % ( timeRangeToken, startName, endName )
      data = {
         timeRangeToken: timeRangeNode,
         startName: dateTimeExpression( startName ),
         endName: dateTimeExpression( endName )
      }

      @staticmethod
      def adapter( mode, args, argsList ):
         start = args.pop( startName, None )
         if start:
            end = args.pop( endName )
            args[ name ] = ( start, end )
   return TimeRangeFilter

# Utility function to convert a date and time to a format accepted by the
# time-range utility
def datetimeToStr( mode, date, time_ ):
   month, day, year = date
   hour, minutes, sec = time_

   datetimeStr = ""
   if year is not None and month is not None and day is not None:
      try:
         datetime.date( year, month, day )
      except ValueError:
         mode.addError( "Invalid date entered" )
         return datetimeStr

   if year is not None:
      datetimeStr = datetimeStr + str( year ) + " "
   if month is not None:
      datetimeStr = datetimeStr + calendar.month_abbr[ month ] + " "
   if day is not None:
      datetimeStr = datetimeStr + str( day ) + " "
   if hour is not None:
      datetimeStr = datetimeStr + str( hour ) + ":" + str( minutes ) + ":" + \
            str( sec )

   return datetimeStr
