# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import threading

UNBYPASSED_FIELDS = ( 'threadLocal_', 'setFile', 'unsetFile', '_getFile',
                      'origFile_' )

# ThreadLocalFile is a File demultiplexer. It is originally created with 
# an original file, and if a particular thread hasn't setFile, then any operation
# for that thread on that ThreadLocalFile object will go to the original file. 
# However if a thread has done a setFile then any operation in that thread will 
# go the File that as set. A thread may call setFile multiple times, but MUST
# call unsetFile the same number of times.

class ThreadLocalFile:
   def __init__( self, origFile ):
      self.__dict__[ 'threadLocal_' ] = threading.local()
      self.__dict__[ 'origFile_' ] = origFile

   def _getFile( self ):
      localFileStack = getattr( self.threadLocal_, 'localFileStack', None )
      if not localFileStack: # either None is set or it is empty
         return self.origFile_
      return localFileStack[ -1 ]
   
   def setFile( self, f ):
      localFileStack = getattr( self.threadLocal_, 'localFileStack', None )
      if localFileStack is None:
         self.threadLocal_.localFileStack = []
      self.threadLocal_.localFileStack.append( f )

   def unsetFile( self ):
      return self.threadLocal_.localFileStack.pop()

   def __getattr__( self, name ):
      if name in UNBYPASSED_FIELDS:
         result = self.__dict__[ name ]
      else:
         result = self._getFile().__getattribute__( name )
      return result

   def __setattr__( self, name, value ):
      if name in UNBYPASSED_FIELDS:
         self.__dict__[ name ] = value
      else:
         self._getFile().__setattr__( name, value )
