# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import threading

class _Method( object ):
   def __init__( self, name, threadLocalEnv, threadLocal ):
      self.name_ = name
      self.threadLocal_ = threadLocal
      self.threadLocalEnv_ = threadLocalEnv

   def __call__( self, *args, **kwargs ):
      self.threadLocalEnv_._initialize() # pylint: disable-msg=W0212
      func = getattr( self.threadLocal_.localDict, self.name_ )
      return func( *args, **kwargs )

   def __doc__( self ):
      self.threadLocalEnv_._initialize() # pylint: disable-msg=W0212
      func = getattr( self.threadLocal_.localDict, self.name_ )
      return func.__doc__()

   def __str__( self ):
      self.threadLocalEnv_._initialize() # pylint: disable-msg=W0212
      func = getattr( self.threadLocal_.localDict, self.name_ )
      return func.__str__()

   def __repr__( self ):
      self.threadLocalEnv_._initialize() # pylint: disable-msg=W0212
      func = getattr( self.threadLocal_.localDict, self.name_ )
      return func.__repr__()

class ThreadLocalEnv( object ):
   def __init__( self, originalDict ):
      self.threadLocal_ = threading.local()
      self.originalDict_ = originalDict.copy()
      for attr in dir( originalDict ):
         if attr in ( '__getattribute__', '__setattr__' ):
            continue
         setattr( originalDict, attr, _Method( attr, self, self.threadLocal_ ) )

      originalDict.__getattribute__ = self.__getattribute__
      originalDict.__setattr__ = self.__setattr__

   def _initialize( self ):
      initialized = getattr( self.threadLocal_, 'initialized', None )
      if initialized is not None:
         return

      self.threadLocal_.initialized = True
      self.threadLocal_.localDict = self.originalDict_.copy()

   def __getattribute__( self, name ):
      if ( name in ( 'threadLocal_', 'originalDict_', '_initialize' ) or
           name.startswith( '__' ) ):
         return object.__getattribute__( self, name )
      self._initialize()
      return self.threadLocal_.localDict.__getattribute__( name )

   def __setattr__( self, name, value ):
      if name in ( 'threadLocal_', 'originalDict_' ):
         return dict.__setattr__( self, name, value )
      self._initialize()
      return self.threadLocal_.localDict.__setattr__( name, value )
