# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import math, os, Tac, Tracing
from Arnet.NsLib import DEFAULT_NS
from IpLibConsts import DEFAULT_VRF
import PyWrappers.Tcpdump as tcpdump
import MirroringLib

__defaultTraceHandle__ = Tracing.Handle( "TcpdumpLib" )

t0 = Tracing.trace0
t2 = Tracing.trace2

OperState = MirroringLib.OperStateType()
DeviceType = Tac.Type( 'Tcpdump::DeviceType' )

anyInterfaceName = 'any'
fabricName = 'fabric'

def getCtxFileName( tcpdumpPid ):
   return '/var/tmp/tcpdump/' + str( tcpdumpPid )

def getLastCompatibleDumpIndex( filePath, fileCount ):
   ''' Returns the index of the most recent file compatible with a file name pattern
   '''
   fileList = []
   ( dumpDir, baseName ) = os.path.split( filePath )
   baseNameLen = len( baseName )
   for f in os.listdir( dumpDir ):
      if f.startswith( baseName ):
         fileList.append( [ os.stat( dumpDir + '/' + f ).st_mtime,
                          f ] )
   fileList.sort( reverse=True )
   for f in fileList:
      dumpFileName = f[ 1 ]
      # if fileCount is 0 dump files rotate without a limit, but the first one
      # doesn't have the 0 suffix
      if fileCount == 0 and dumpFileName == baseName:
         return 0
      try:
         dumpIndex = int( dumpFileName[ baseNameLen : ] )
         if fileCount > 0 and dumpIndex > fileCount:
            continue
         return dumpIndex
      except ValueError:
         pass
   return None

def appendResumeOption( cmd, args, lastPid ):
   # Logs shouldn't rotate or this is the first time it starts
   if not args.get( 'maxFileSize' ) or lastPid == 0:
      return
   # Custom argument for resuming from a previously dumped context
   lastCtxFileName = getCtxFileName( lastPid )
   lastCtxFile = None
   lastDumpFileIndex = None
   try:
      # This should work unless tcpdump crashed before we killed it
      t2( "Looking for previous context in", lastCtxFileName )
      with open( lastCtxFileName ) as lastCtxFile:
         lastDumpFileName = lastCtxFile.readline().strip()
         if lastDumpFileName == args.get( 'file', '' ):
            lastDumpFileIndex = int( lastCtxFile.readline().strip() ) 
            t2( "Found previous context:", lastDumpFileName, "with index",
                lastDumpFileIndex )
   except IOError:
      # tcpdump likely crashed, attempt to infer where it left off from the
      # filesystem
      t2( "No context found for", lastCtxFileName, "looking for previous dumps" )
      lastDumpFileIndex = getLastCompatibleDumpIndex( args.get( 'file', '' ),
                                                      args.get( 'fileCount', 0 ) )
   finally:
      if lastDumpFileIndex is not None:
         cmd.extend( [ '-@', str( lastDumpFileIndex + 1 ) ] )
      if lastCtxFile:
         os.remove( lastCtxFileName )

class InvalidTargetException( Exception ):
   def __init__( self, msg ):
      Exception.__init__( self, msg )

class InvalidDeviceException( InvalidTargetException ):
   def __init__( self, deviceType, intf, deviceStatusAll ):
      if deviceType == DeviceType.any:
         InvalidTargetException.__init__( self, "Unspecified 'any' interface " +
                                          "exception" ) 
      elif deviceType == DeviceType.interface:
         InvalidTargetException.__init__( self, "Invalid interface: %s" % intf ) 
      elif deviceType == DeviceType.mirror:
         mirroringSession = intf
         mirroringStatus = deviceStatusAll
         if not mirroringSession in mirroringStatus:
            InvalidTargetException.__init__( self, 
                           "Monitor session %s does not exist" % mirroringSession )
         else:
            sessionStatus = mirroringStatus[ mirroringSession ]
            if not any( s.state == 'operStateActive' for s in
                        sessionStatus.srcOperStatus.values() ):
               InvalidTargetException.__init__( self, "Monitor session " + 
                              mirroringSession + " does not have an active source" )
            elif 'Cpu' not in sessionStatus.targetOperStatus:
               InvalidTargetException.__init__( self, "Monitor session " + 
                              mirroringSession + " does not have destination cpu" )
            elif sessionStatus.targetOperStatus[ 'Cpu' ].state != \
                                         OperState.operStateActive:
               # In order to tcpdump a mirroring session, a mirroring session
               # needs to have cpu destination and the status of the 
               # cpu destination need to be active
               InvalidTargetException.__init__( self, "Monitor session " + 
                                                mirroringSession
                     + " does not have an active destination" )
            else:
               InvalidTargetException.__init__( self, 
                                             "Unexpected error: no kernel device. "
                     + "Contact your customer support representative." )
      elif deviceType == DeviceType.lanz:
         InvalidTargetException.__init__( self, 
                                          "Queue-monitor interface unavailable" )
         
class InvalidVrfException( InvalidTargetException ):
   def __init__( self, deviceType, vrfName ):
      InvalidTargetException.__init__( self, "Invalid vrf name: %s" % vrfName ) 

def deviceName( deviceStatusAll, device, deviceType ):
   if not device:
      return fabricName
   
   if deviceType == DeviceType.any:
      return anyInterfaceName
   elif deviceType in ( DeviceType.interface, DeviceType.interfaceAddress ):
      try:
         return deviceStatusAll[ deviceType ][ device ].deviceName
      except KeyError:
         raise InvalidDeviceException( deviceType, device,
                                       deviceStatusAll[ deviceType ] )
   elif deviceType in ( DeviceType.mirror, DeviceType.lanz ):
      try:
         if deviceType == DeviceType.lanz:
            return deviceStatusAll[ deviceType ][ '_InternalLanz' ].mirrorDeviceName
         return deviceStatusAll[ deviceType ][ device ].mirrorDeviceName
      except KeyError:
         raise InvalidDeviceException( deviceType, device,
                                       deviceStatusAll[ deviceType ] )

def deviceNameFromSessionConfig( sessionConfig ):
   if sessionConfig.deviceType == DeviceType.lanz:
      # lanz deviceType uses a special name.
      return '_InternalLanz'
   return sessionConfig.device

def anyToNsName( allVrfStatusLocal, sessionConfig ):
   return anyIntfToNetNs( allVrfStatusLocal, sessionConfig.deviceType, 
                          sessionConfig.vrfName )
   
def anyIntfToNetNs( allVrfStatusLocal, deviceType, vrfName ):
   netNs = DEFAULT_NS
   if deviceType == DeviceType.any:
      if vrfName and vrfName != DEFAULT_VRF:
         if vrfName in allVrfStatusLocal.vrf:
            netNs = allVrfStatusLocal.vrf[ vrfName ].networkNamespace
         else:
            raise InvalidVrfException( deviceType, vrfName )
      return netNs
   return None # not an 'any' interface

def netNsName( deviceStatusAll, intfStatusLocal, allVrfStatusLocal, sessionConfig ):
   netNs = anyToNsName( allVrfStatusLocal, sessionConfig )
   if netNs:
      return netNs
   return getNetNsName( deviceStatusAll, intfStatusLocal, allVrfStatusLocal,
                     sessionConfig.device, sessionConfig.deviceType )
      
def getNetNsName( deviceStatusAll, intfStatusLocal, allVrfStatusLocal, intf, 
               deviceType ):
   # Mirroring and lanz devices are in DEFAULT_NS
   # namespaces due to VRFs for 'any' interfaces are dealt with earlier
   if deviceType in ( DeviceType.mirror, DeviceType.lanz, DeviceType.any ):
      return DEFAULT_NS
   netNs = DEFAULT_NS
   if deviceName( deviceStatusAll, intf, deviceType ) != fabricName:
      intfStatus = intfStatusLocal.intfStatusLocal.get( intf )
      netNs = intfStatus.netNsName if intfStatus else DEFAULT_NS
   return netNs

def tcpdumpCmdSyntax( deviceStatusAll, args, deviceType, lastPid=0 ):
   # List all possible arguments for tcpdump with their native syntax
   tcpdumpSyntax = { 'device': '-i',
                     'size': '-s', 
                     'maxFileSize': '-C', 
                     'file': '-w',
                     'readFile': '-r',
                     'fileCount': '-W', 
                     'packetCount': '-c' }

   cmd = [ tcpdump.name() ]
   for var in tcpdumpSyntax.keys():
      if not var in args: 
         continue # avoid KeyError for args not specified
      optionValue = None
      # Map EOS interface name to kernel
      if var == 'device':
         optionValue = deviceName( deviceStatusAll, args[ var ], deviceType )
         if not optionValue:
            raise InvalidDeviceException( deviceType, args[ var ],
                                          deviceStatusAll[ deviceType ] )
      # Add the argument to the tcpdump arguments if not null or 0
      elif args[ var ]:
         optionValue = str( args[ var ] )
      if optionValue:
         cmd.extend( [ tcpdumpSyntax[ var ], optionValue ] )
   
   lookupNames = args.get( 'lookupNames' ) 
   if not lookupNames:
      cmd.extend( [ '-n' ] ) 

   if not args.get( 'file' ) and args.get( 'verbose' ):
      cmd.extend( [ '-vv' ] )

   # In fc14, by default we always use flush the buffer.
   # But in fc18, we are not flushing the buffer until it fills up completely.
   # For further detail kindly go through the link:
   # https://www.sourceware.org/ml/libc-alpha/2012-09/msg00198.html
   # So, adding -U option to make buffer flush as soon as we get the raw pkt o/p.
   cmd.extend( [ '-U' ] )

   appendResumeOption( cmd, args, lastPid )

   # the filter must come last   
   dumpFilter = args.get( 'filter' )
   if dumpFilter:
      cmd.extend( [ dumpFilter ] )

   return cmd

def isPacketLine( line ):
   """Check if one line has packet info or not"""

   return line and line.strip()

def isPacketStart( line ):
   """Tcpdump starts one packet without leading space or tab."""

   return line[0] != ' ' and line[0] != '\t'

def linesToPackets( lines ):
   """To generate a new list which contains one packet per item.
   Each item includes both packet header and details if present.
   """

   dump = []
   packet = None

   for line in lines:
      if not isPacketLine( line ):
         # Bug37808: 'lines' may include illegal lines if they are
         # generated by Cli
         continue

      if isPacketStart( line ):
         if packet is not None:
            # Done with one packet
            dump.append( packet )
         # Starting a new packet
         packet = line
      else:
         # Add newline and append details
         packet = '%s\n%s' % (packet, line)

   if packet:
      dump.append( packet )

   return dump

def getDumpFromFile( filePath, maxPackets, verbose, lookupNames, dumpFilter ):
   t0( "Attempting to extract up to %d packets from %s" % ( maxPackets, filePath ) )
   options = dict()
   options[ 'readFile' ] = filePath
   options[ 'verbose' ] = verbose
   options[ 'lookupNames' ] = lookupNames
   options[ 'filter' ] = dumpFilter
   cmd = tcpdumpCmdSyntax( None, options, None )
   try:
      t2( "Running:", cmd )
      dump = Tac.run( cmd,
                      stdout=Tac.CAPTURE,
                      stderr=Tac.DISCARD ).split( '\n' )
   except Tac.SystemCommandError, e:
      t0( e )
      return None
   if len( dump[ -1 ] ) < 1:
      dump.pop()
   # Remove tcpdump header info

   newDump = linesToPackets( dump )
   dumpLen = len( newDump )
   t0( "Dump contains %d packets" % dumpLen )
   # If there's a limit use the last packets
   if maxPackets and dumpLen > maxPackets :
      t0( "Adding the last %d packets" % maxPackets )
      output = newDump[ -maxPackets : ]
   # Return the entire file
   else:
      t0( "Adding all packets" )
      output = newDump
   return output

DEFAULT_FILE_PATH_FMT = '/tmp/tcpdump/%s/%s'
def dumpFile( session ):
   if session.file:
      return session.file
   return DEFAULT_FILE_PATH_FMT % ( session.name, session.name )

def getDumpFromMultipleFiles( session, maxPackets, lookupNames, dumpFilter, 
                              verbose ):
   packetsLeft = maxPackets
   currentIndex = None
   indexDigits = 0
   filePath = dumpFile( session )
   t0( "Looking for files with prefix", filePath )
   if session.fileCount > 0:
      indexDigits = int( math.log10( session.fileCount - 1 ) ) + 1
   try:
      # Try to figure out what is the last file that was written
      currentIndex = getLastCompatibleDumpIndex( filePath, session.fileCount )
   except OSError, e:
      t0( e )
   if currentIndex is None:
      t0( "No dump file found" )
      return None
   else:
      t0( "Starting from index %d" % currentIndex )
   output = []
   startIndex = None
   # Loop until we reach maxPackets or there's no more data
   while packetsLeft:
      if startIndex is None:
         startIndex = currentIndex
      else:
         # Prevent infinite loop
         if currentIndex == startIndex:
            break
      # When fileCount is not set the first file doesn't have a '0' suffix
      if not currentIndex and session.fileCount == 0:
         t0( "Reading %s" % filePath )
         currentOutput = getDumpFromFile( filePath, 
                                          packetsLeft, 
                                          verbose, lookupNames,
                                          dumpFilter )
         if currentOutput:
            output = currentOutput + output
         break
      else:
         currentFile = filePath + str( currentIndex ).zfill( indexDigits )
         t0( "Reading %s" % currentFile )
         currentOutput = getDumpFromFile( currentFile, 
                                          packetsLeft, 
                                          verbose, lookupNames, 
                                          dumpFilter )
         if currentOutput:
            packetsLeft -= len( currentOutput )
            output = currentOutput + output
         currentIndex -= 1
         # Try to wrap around to the biggest index
         if currentIndex < 0:
            currentIndex = session.fileCount - 1
   return output

def getDumpOutput( session, maxPackets, lookupNames, dumpFilter, verbose ):
   ''' Returns a list of packets dumped by a session '''
   
   t2( "Reading output from session %s" % session )
   # If there is maximum file size we expect to have everything in a file
   if session.maxFileSize > 0:
      output = getDumpFromMultipleFiles( session, maxPackets, lookupNames, 
                                         dumpFilter, verbose )
   # There is only one file: read as much data as you can form there
   else:
      filePath = dumpFile( session )
      t0( "Reading single file %s" % filePath )
      output = getDumpFromFile( filePath, maxPackets, verbose, 
                                lookupNames, dumpFilter )
   return output

def cliTokenMap():
   return { 'duration' : 'duration',
            'filecount' : 'fileCount',
            'file' : 'file',
            'filter' : 'filter',
            'intfMonitorLanz' : 'device',
            'max-file-size' : 'maxFileSize',
            'packet-count' : 'packetCount',
            'size' : 'size',
            'vrf' : 'vrfName' }

def sessionAttributeMap():
   return dict( ( v, k ) for k, v in cliTokenMap().iteritems() )

def getTcpdumpCliParameter( session, attrName, includeDefaults=True ):
   '''Return "<cli token> <value>" based on the data in a session object'''
   token = sessionAttributeMap()[ attrName ]
   if attrName == 'filter':
      attrValue = getattr( session, attrName )
      if attrValue:
         return token + ' ' + attrValue
   elif attrName == 'file':
      attrValue = getattr( session, attrName )
      if attrValue:
         import Url
         return token + ' ' + Url.filenameToUrl( attrValue )
   elif attrName == 'verbose' or attrName == 'lookupNames':
      if getattr( session, attrName ):
         return token
   elif attrName == 'device':
      deviceTypeToToken = { 'kernel' : '',
                            'interface': 'interface',
                            'interfaceAddress': 'interface-address',
                            'mirror': 'monitor',
                            'lanz': 'queue-monitor', 'any' : 'interface' }
      token = deviceTypeToToken[ session.deviceType ]
      if session.deviceType in [ 'interface', 'mirror', 'any' ]:
         attrValue = getattr( session, attrName )
         cmd = ''
         if attrValue != '':
            cmd = token + ' ' + str( attrValue )
            if session.deviceType == 'any':
               vrf = session.vrfName
               if vrf and not vrf == DEFAULT_VRF:
                  cmd += ' vrf ' + vrf
         return cmd
      elif session.deviceType == 'lanz':
         return token
      elif session.deviceType == 'interfaceAddress':
         otherFilters = session.filter
         cmd = 'interface-address ' + session.device + otherFilters
         return cmd
   elif attrName == 'vrfName':
      return None
   else:
      attrValue = getattr( session, attrName )
      # These attributes have defaults that should not show up in the CLI
      # (0=unlimited or 'fabric')
      if attrName in [ 'duration', 'device', 'packetCount', 'size', 'vrfName' ]:
         includeDefaults = False
      if includeDefaults or attrValue != getattr( session.options,
                                                  attrName + 'Default' ):
         return token + ' ' + str( attrValue )
   return None

def filterFromIntf( session, ipAddressDir, ip6AddressDir, dumpFilter ):
   intf = session.device
   return composeFilter( intf, ipAddressDir, ip6AddressDir, dumpFilter )

def composeFilter( intf, ipAddressDir, ip6AddressDir, dumpFilter ):
   filterExpr = computeAddrFilter( intf, ipAddressDir, ip6AddressDir )
   return combineFilters( filterExpr, dumpFilter )

def computeAddrFilter( intf, ipAddressDir, ip6AddressDir ):
   filterExpr = "host "
   ip6IntfDir = ip6AddressDir.intf.get( intf )
   ip4IntfDir = ipAddressDir.ipIntfStatus.get( intf )
   if not ip6IntfDir and not ip4IntfDir:
      return None
   ip6s = []
   if ip6IntfDir:
      for addr in ip6IntfDir.addr:
         ip6s.append( str( addr.address ) )
      filterExpr += " or ".join( ip6s )
   secondaryIp4s = []
   if ip4IntfDir:
      for addr in ip4IntfDir.activeSecondaryWithMask:
         secondaryIp4s.append( addr.address )
      primaryIp4 = str( ip4IntfDir.activeAddrWithMask.address )
      if filterExpr != "host ":
         filterExpr += " or "
      filterExpr += primaryIp4
      if secondaryIp4s:
         filterExpr += " or "
   filterExpr += " or ".join( secondaryIp4s )
   return filterExpr

def combineFilters( filterExpr, dumpFilter ):
   if dumpFilter:
      if filterExpr:
         filterExpr = "(" + dumpFilter + ") and (" + filterExpr + ")"
      else:
         return dumpFilter
   return filterExpr
