#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This library of routines is used to provide common services for
# Pulse clients

import Tac
import Tracing
import Smash
import os

__defaultTraceHandle__ = Tracing.Handle( 'TacPulse' )
t0 = __defaultTraceHandle__.trace0

def useLocalMem( local ):
   if local:
      # When Pulse::writer() creates the Smash, it uses this env-var
      # to determine whether local mem or shared mem should be used
      os.environ[ 'ABUILD' ] = '1'
   else:
      del os.environ[ 'ABUILD' ]

def Counter( name, value ):
   return Tac.Value( 'TacPulse::Counter', name, value )

class ReaderLib( object ):
   def __init__( self ):
      self.Counters = Tac.Type( 'TacPulse::Counters' )
      self.Finder = Tac.Type( 'TacPulse::Finder' )

   def counter( self, pulseName, counterName ):
      pulse = self.Counters( pulseName, 'reader' )
      try:
         return pulse.counter[ counterName ]
      except KeyError:
         # Trace for detection of BUG229722
         t0( "TacPulse counter not found with pulseName: %s and counterName: %s"
               % ( pulseName, counterName ) )
         return Tac.Value( 'TacPulse::Counter' )

   def counters( self, pulseName ):
      pulse = self.Counters( pulseName, 'reader' )
      return pulse.counter

   def findall( self, keyword ):
      finder = self.Finder( keyword )
      results = finder.found.splitlines()
      return results

class WriterLib( object ):
   def __init__( self ):
      self.Counters = Tac.Type( 'TacPulse::Counters' )

   def write( self, tableName, counterName, value ):
      pulse = self.Counters( tableName, 'writer' )
      c = Tac.Value( 'TacPulse::Counter', counterName, value )
      pulse.counter.addMember( c )

   def cleanup( self, tableName ):
      pulse = self.Counters( tableName, 'writer' )
      pulse.cleanup()

def rootPath():
   return Smash.rootPath()

def defaultPulseName():
   counters = Tac.Type( 'TacPulse::Counters' )
   return counters.defaultPulseName()

def defaultPulseNameIs( name ):
   counters = Tac.Type( 'TacPulse::Counters' )
   counters.defaultPulseNameIs( name )

def disableForkedDefaultWriter():
   counters = Tac.Type( 'TacPulse::Counters' )
   counters.disableForkedDefaultWriter()
