# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import six
import re
from collections import OrderedDict



def getDelimsFromDashes( dashes ):
   """
      Gets delimiters from a line of dashes that define a table.
         
         dashes - String of dashes (with potential spaces)

      Returns - Array of integers where the ith entry is the maximum size of any
                value that should appear in the ith column of the table
      
      Ex. ( "--- ---" ) --> [ 3, 3 ]
          ( "-   -  " ) --> [ 3, 3 ]
   """
   delims = []
   j = 0
   while j < len( dashes ):
      nxt = dashes.find( ' -', j )
      k = nxt if nxt != -1 else len( dashes )
      delims.append( k - j )
      j = k + 1
   return delims


def splitLineByDelims( line, delims ):
   """
      Splits a string into chunks depending on the supplied delimiters.

         line - String to split
         delims - Delimiters used to split the line

      Returns - False if line is incompatible with delims. Else, a list of
                strings where the i'th item is the trimmed substring as split by
                the delims list. There is assumed to be exactly one space between
                each 'chunk'.

      Ex. ( "123 1234", [ 3, 4 ] ) --> [ "123", "1234" ]
          ( "123 1234", [ 3, inf ] ) --> [ "123", "1234" ]
   """
   # check line exists
   if not line or line.isspace():
      return False
   a = 0
   build = []
   for i in delims:
      b = a + i
      if b < len( line ):
         if line[ b ] != ' ': # has to be a space to be a valid line
            return False
      build.append( line[ a : min( b, len( line ) ) ].strip() )
      a = b + 1
   return build


class Table( object ):
   """
      Helper class for TableParser.

         title - Table's title
         table - List of dictionaries, each entry represents one row
         colTitles - Column titles
         colStructure - List where the i'th element is the max length of a value for
            the i'th column
   """

   def __init__( self ):
      self.title = None
      self.table = []
      self.colTitles = []
      self.colStructure = []

   def reset( self ):
      self.title = None
      self.table = []
      self.colStructure = []
      self.colTitles = []

   def addRow( self, row ):
      self.table.append( row )


class TableParser( object ):
   """
      FSM-like class for parsing tables. A single instance of TableParser will store
      any amount of tables, therefore one parser should be used per input.

         titleRegex - None if we aren't looking for titles, otherwise the RegEx that
            will be used to match titles for tables
         currTable - The current table being built
         status - The status of the FSM
         tables - All tables found
         titles - List where the i'th item is the title of the i'th table. None if
            we aren't looking for titles
   """

   def __init__( self, titleRegex ):
      self.titleRegex = titleRegex
      self.currTable = Table()
      self.status = "none"
      self.tables = []
      self.titles = []

   def saveTable( self ):
      """
         Writes the table that we are building to the array of tables that we have
         found. Writes the appropriate title to the titles array. Resets the FSM.
      """
      if self.currTable.colStructure: # we have a table to save
         self.tables.append( self.currTable.table ) # save the table
         if self.titleRegex: # looking for titles
            if self.currTable.title: # we found a title
               self.titles.append( self.currTable.title )
            else:
               self.titles.append( "table{}".format( len( self.tables ) ) )
         self.resetFSM()

   def resetFSM( self ):
      """
         Resets the FSM and resets the current table.
      """
      self.status = "none"
      self.currTable.reset()

   def parseTableDefinition( self, tableTitleLine, colTitlesLine, dashesLine ):
      """
         Determines validitity and reads in the three lines that define a new table.

            tableTitleLine - the line that potentially contains the table's title
            colTitlesLine - the line that potentially containts the column titles
            dashesLine - the line that contains the dashes that break up columns

         Returns - False if improper table defininition. Else True.
      """

      # look for table title
      if self.titleRegex:
         titleMatch = re.search( self.titleRegex, tableTitleLine )
         if titleMatch:
            self.currTable.title = titleMatch.group( 1 )

      # define width delimiters for columns
      delims = getDelimsFromDashes( dashesLine )
      delims[ -1 ] = float( "inf" ) # last column can be as big as it wants
      self.currTable.colStructure = delims

      # check to see if columns title line abides by column spacing
      colTitles = splitLineByDelims( colTitlesLine, self.currTable.colStructure )
      if colTitles:
         self.currTable.colTitles = colTitles
         return True
      else:
         self.resetFSM()
         return False

   def parseLine( self, line ):
      """
         Determines validity and reads in a line of a table.

            line - The line that potentially contains data for the current table

         Returns - False if invalid. Else True.
      """
      arr = splitLineByDelims( line, self.currTable.colStructure )
      if arr:
         if len( arr ) != len( self.currTable.colTitles ):
            raise ValueError( "# of columns in row conflicts with table definition"
                  "{} for line {{{}}}".format( self.currTable.colTitles, line ) )
         # cast to ints if numeric
         vals = [ int( v ) if six.u( v ).isnumeric() else v for v in arr ]

         # assign column titles
         # Using an OrderedDict here to implicitly encode the order of the columns.
         # This is useful for users of the `importTables` function because the
         # column ordering is not provided explicitly in its output.
         row = OrderedDict( ( title, value )
                 for title, value in zip( self.currTable.colTitles, vals ) )
         self.currTable.addRow( row )
      else:
         self.saveTable() # save table and reset FSM

   def dumpTables( self, strict ):
      """
         Returns currently built tables.

            strict - If True and a table is being built, it will be included in the
               dump. Else only fully built tables will be included.
      """

      if strict and self.status == "data":
         self.saveTable()
      if not self.tables:
         return None
      if self.titles: # assign titles to each table and return a dict
         return { self.titles[ i ]: self.tables[ i ]
                  for i, _ in enumerate( self.tables ) }
      else: # return simply an array of tables
         return self.tables


def importTables( data, titleRegex=None ):
   """
   Converts any TableOutput-like table into JSON and returns all tables.
   See AID/6469 for a more detailed explanation.

      data - string to parse for tables. Preferebly printed using TableOutput module.
      titleRegex - title regex to search for when a table is discovered

   Returns - None if no table is found in 'data'. Else, a list representing the rows
      of a table (if 'titleRegex' is None) or a dictionary of all tables found in
      'data'. A row of a table is an OrderedDict whose keys are the column headers
      as found in 'data'.

   Example:
   >>> import TableImport
   >>> table1 = '''
   ... Foo Bar
   ... --- ---
   ...   1   2
   ...   3   4
   ... '''
   >>> TableImport.importTables( table1 )
   [[OrderedDict([('Foo', 1), ('Bar', 2)]), OrderedDict([('Foo', 3), ('Bar', 4)])]]
   >>> table2 = '''
   ... MyTableA
   ... A B
   ... - -
   ... 1 2
   ... My Other Table
   ... XXX YYY
   ... --- ---
   ... foo bar
   ... '''
   >>> TableImport.importTables( table2, '(.*)' )
   {'My Other Table': [OrderedDict([('XXX', 'foo'), ('YYY', 'bar')])],
    'MyTableA': [OrderedDict([('A', 1), ('B', 2)])]}
   '''
   """
   # FSM-like parsing
   parser = TableParser( titleRegex )

   data = data.splitlines()
   for i, line in enumerate( data ):
      if parser.status == "none": # look for column declarations
         if line:
            colsFound = True
            for chunk in line.split():
               if not re.search( "^-+$", chunk ): # each chunk must be "----"
                  colsFound = False
                  break
            if colsFound:
               # define columns and titles
               foundTable = parser.parseTableDefinition( data[ i-2 ],
                                                         data[ i-1 ],
                                                         line )
               if foundTable:
                  parser.status = "data"

      elif parser.status == "data":
         parser.parseLine( line )

   return parser.dumpTables( True )
