# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import Cell
import sys
import os
import weakref

t0 = Tracing.trace0

class RedundancyStatusReactor( Tac.Notifiee ):
   """ Helper reactor responsible for notifying the Sysdb agent
   class when we move from standby to active mode. """

   notifierTypeName = "Redundancy::RedundancyStatus"

   def __init__( self, redundancyStatus, sysdbAgent, handleSwitchover ):
      # sysdbAgent's switchover reactor is not used unless handleSwitchover is
      # set. Not calling weakref.proxy when handleSwitchover is False helps
      # tests which are only concerned with peerMode, protocol, etc.
      self.sysdbAgent_ = weakref.proxy( sysdbAgent ) if handleSwitchover else None
      Tac.Notifiee.__init__( self, redundancyStatus )
      self.handleSwitchover_ = handleSwitchover # True on standby.

   @Tac.handler( 'mode' )
   def handleMode( self ):
      # The SM is effectively not required if sup is active or if it becomes active.
      # However, running it in both of these cases for now, should be no-op
      print >> sys.stderr, "mode changed to", self.notifier_.mode
      if self.handleSwitchover_ and self.notifier_.mode == "active":
         self.sysdbAgent_.handleSwitchover()

class RedStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Redundancy::RedundancyStatus"

   def __init__( self, redStatus, cleanupSm ):
      Tac.Notifiee.__init__( self, redStatus )
      self.cleanupSm_ = cleanupSm
      self.handleMode()

   @Tac.handler( 'mode' )
   def handleMode( self ):
      self.cleanupSm_.handleChange()

class ElectionStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Election::Status"

   def __init__( self, electionStatus, cleanupSm ):
      Tac.Notifiee.__init__( self, electionStatus )
      self.cleanupSm_ = cleanupSm
      self.handlePeerRedundancyMode()

   @Tac.handler( 'peerRedundancyMode' )
   def handlePeerRedundancyMode( self ):
      self.cleanupSm_.handleChange()

class StandbyPathCleanupSm( object ):
   def __init__( self, redStatus, electionStatus, cellDir, fdlCellBaseDir ):
      self.redStatus_ = redStatus
      self.electionStatus_ = electionStatus
      self.cellDir_ = cellDir
      self.fdlCellBaseDir_ = fdlCellBaseDir
      self.elecStatusReactor_ = ElectionStatusReactor(
            electionStatus, weakref.proxy( self ) )
      self.redStatusReactor_ = RedStatusReactor(
            redStatus, weakref.proxy( self ) )

   def handleChange( self ):
      if self.electionStatus_.peerRedundancyMode == 'disabled':
         key = str( Cell.peerCellId() )
         # Delete peer cell directory if scd-em determines that peer is disabled
         if key in self.cellDir_.entryState:
            self.cellDir_.deleteEntity( key )
         if self.redStatus_.mode == 'active':
            # Can't delete this until locallReadOnly is False
            peerFdlCellDir = self.fdlCellBaseDir_.get( key )
            if peerFdlCellDir:
               if 'local' in peerFdlCellDir.entryState:
                  peerFdlCellDir.deleteEntity( 'local' )
               if 'peer' in peerFdlCellDir.entryState:
                  peerFdlCellDir.deleteEntity( 'peer' )

def setRedundancyValues( electionStatus, redundancyStatus ):
   # BUG271560: If we are a modular system and ElectionMgr is not supported,
   # then we only have a single supervisor slot and RedSup will not run.
   # Need to initialize the electionStatus and redundancyStatus entities
   # to single supe values.
   # The exception of this case is if we are running a RedSup NamespaceDut.
   # Here we don't have ElectionMgr but we still have 2 supervisors running,
   # so we don't want to enter here.
   if( Cell.cellType() == "supervisor" and not Cell.electionMgrSupported()
         and 'REDSUPDUT_PEER_REDMODE' not in os.environ ):
      t0( "setRedundancyValues for the system without ElectionMgr support" )
      RedundancyMode = Tac.Type( 'Election::RedundancyMode' )
      PeerState = Tac.Type( 'Election::PeerState' )

      electionStatus.lastRedundancyModeChangeTime = Tac.now()
      electionStatus.redundancyMode = RedundancyMode.active
      electionStatus.peerRedundancyMode = RedundancyMode.disabled
      electionStatus.peerState = PeerState.notInserted

      redundancyStatus.hwMode = RedundancyMode.active
      redundancyStatus.peerMode = RedundancyMode.disabled
      redundancyStatus.peerState = PeerState.notInserted
      redundancyStatus.peerActive = False
      redundancyStatus.activeCellId = 1
      redundancyStatus.initialized = True

def _initSysdbRedundancy( sysdbAgentContainer ):
   t0( " Sysdb Agent _initSysdbRedundancy " )
   redundancyStatus = sysdbAgentContainer.entityManager_.redundancyStatus()
   sysdbCell = sysdbAgentContainer.entityManager_.root()[ 'cell' ]
   ourSysdbCell = sysdbCell[ str( Cell.cellId() ) ]
   cellElection = ourSysdbCell[ 'redundancy' ].get( 'election' )
   if cellElection is not None:
      cellElectionStatus = cellElection[ 'status' ]
      setRedundancyValues( cellElectionStatus, redundancyStatus )
      # We want this functionality to run only when entity manager is
      # either EntityManager.Local or SysdbEntityManager.Standby
      # SysdbProxy, which instantiates Remote EM EntityManager.Sysdb should not
      # run this functionality.
      if sysdbAgentContainer.standalone_ or sysdbAgentContainer.sysdbStandby_:
         fdlCellDir = sysdbAgentContainer.entityManager_.root() \
                      [ 'hardware' ][ 'fdl' ][ 'cell' ]
         if fdlCellDir is not None:
            return StandbyPathCleanupSm( redundancyStatus,
               cellElectionStatus, sysdbCell, fdlCellDir )
   return None
