# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import argparse
from enum import Enum
import os
import sys
import SignatureFile
import SignatureRequest
import SwiSignLib
import Tac

class SWIX_SIGN_RESULT( Enum ):
   SUCCESS = 0
   ALREADY_SIGNED_INVALID = 1
   ALREADY_SIGNED_VALID = 2
   SERVER_ERROR = 3
   ERROR_NOT_A_SWIX = 4

class SwixSignException( Exception ):
   ''' Exception that occurs when we fail to sign a SWIX '''
   def __init__( self, code, message ):
      self.code = code
      super( SwixSignException, self ).__init__( message )

def signSwix( swixFile, forceSign=False, useDevCA=False, devCaKeyPair=None ):
   ''' Sign a SWIX. Throws SwixSignException if it failed to sign. '''
   if not SwiSignLib.isSwixFile( swixFile ):
      raise SwixSignException( SWIX_SIGN_RESULT.ERROR_NOT_A_SWIX,
                               "Input is not a SWIX." )
   # Check if there's an existing signature
   signed = SwiSignLib.swiSignatureExists( swixFile )
   validSig, _ = SwiSignLib.verifySwiSignature( swixFile )
   if signed:
      if not forceSign:
         message = 'Warning: SWIX is signed with an invalid signature.'
         code = SWIX_SIGN_RESULT.ALREADY_SIGNED_INVALID
         if validSig:
            message = 'SWIX is already signed with a valid signature.'
            code = SWIX_SIGN_RESULT.ALREADY_SIGNED_VALID
         raise SwixSignException( code, message )
      else:
         # Force sign, remove swix-signature file from SWI
         Tac.run( [ 'zip', '-d', swixFile, SwiSignLib.SWIX_SIG_FILE_NAME ] )

   # Start signing process
   swixSignature = SignatureFile.Signature()
   # Use the basename of the file as the version
   swixData = SignatureFile.prepareDataForServer( swixFile,
                                                  os.path.basename( swixFile ),
                                                  swixSignature )
   try:
      if useDevCA:
         signatureData = SignatureRequest.getDataFromDevCA( swixFile, swixData,
                                                      devCaKeyPair=devCaKeyPair )
      else:
         signatureData = SignatureRequest.getDataFromServer( swixFile, swixData )
      SignatureFile.generateSigFileFromServer( signatureData, swixFile, 
                                               swixSignature )
   except SignatureRequest.SigningServerError, e:
      # Remove null signature
      Tac.run( [ 'zip', '-d', swixFile, SwiSignLib.SWIX_SIG_FILE_NAME ] )
      raise SwixSignException( SWIX_SIGN_RESULT.SERVER_ERROR, e.message )

def sign( swixFile, forceSign=False, useDevCA=False, devCaKeyPair=None ):
   ''' Sign a SWIX, returning a status code (enum) from SWIX_SIGN_RESULT '''
   try:
      signSwix( swixFile, forceSign, useDevCA, devCaKeyPair )
   except SwixSignException as e:
      print( e )
      return e.code
   else:
      return SWIX_SIGN_RESULT.SUCCESS

def signHandler( args ):
   parser = argparse.ArgumentParser( prog="swix sign" )
   parser.add_argument( "swix", metavar="EXTENSION.swix",
                        help="Path of the SWIX to be signed" )
   parser.add_argument( "--force-sign", help="Force signing the SWIX",
                        action="store_true" )
   parser.add_argument( "--dev-ca", help="Use development certificates for signing",
                        action="store_true" )

   args = parser.parse_args( args )
   retCode = sign( args.swix, args.force_sign, args.dev_ca )
   exit( retCode.value )

if __name__ == "__main__":
   signHandler( sys.argv[ 1: ] )
